/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization.json;

import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.ast.api.DependencyResolutionMode.COMPILED;
import static org.mule.runtime.ast.api.DependencyResolutionMode.getDependencyResolutionMode;
import static org.mule.runtime.ast.internal.serialization.json.JsonArtifactAstSerializerFormat.JSON;

import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.Objects.requireNonNull;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.TypedComponentIdentifier;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.serialization.ArtifactAstSerializer;
import org.mule.runtime.ast.internal.serialization.ArtifactAstSerializerMetadata;
import org.mule.runtime.ast.internal.serialization.ArtifactAstSerializerMetadataSerializer;
import org.mule.runtime.ast.internal.serialization.dto.ArtifactAstDTO;
import org.mule.runtime.ast.internal.serialization.dto.ParserAttributesDTO;
import org.mule.runtime.ast.internal.serialization.dto.factory.AstDTOFactoryProvider;
import org.mule.runtime.ast.internal.serialization.json.gson.ComponentIdentifierJsonSerializer;
import org.mule.runtime.ast.internal.serialization.json.gson.ComponentLocationJsonSerializer;
import org.mule.runtime.ast.internal.serialization.json.gson.ParserAttributesJsonTypeAdapter;
import org.mule.runtime.ast.internal.serialization.json.gson.TypedComponentIdentifierJsonSerializer;

import java.io.IOException;
import java.io.InputStream;
import java.io.PipedReader;
import java.io.PipedWriter;
import java.io.Reader;
import java.io.Writer;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Executor;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;

import org.apache.commons.io.input.ReaderInputStream;


/**
 * This is an implementation of an {@link ArtifactAstSerializer} that uses Gson to generate a Json String as the body of the
 * serialized representation
 */
public class JsonArtifactAstSerializer implements ArtifactAstSerializer {

  private final Executor executor;
  private final Gson gson;
  private final ArtifactAstSerializerMetadataSerializer artifactAstSerializerMetadataSerializer =
      new ArtifactAstSerializerMetadataSerializer();
  private final AstDTOFactoryProvider astDTOFactoryProvider;
  private final ArtifactAstSerializerMetadata serializerMetadata;

  /**
   * Creates a serializer for the specified version.
   *
   * @param executor the {@link Executor} to use for the serialization work.
   * @param version  the version this serializer must adhere to.
   */
  public JsonArtifactAstSerializer(Executor executor, String version) {
    this(null, executor, version);
  }

  /**
   * This constructor is for testing, and it allows overriding the {@link Gson} instance
   * 
   * @param gson     a pre-configured {@link Gson}, for testing purposes.
   * @param executor the {@link Executor} to use for the serialization work.
   * @param version  the version this serializer must adhere to.
   */
  public JsonArtifactAstSerializer(Gson gson, Executor executor, String version) {
    requireNonNull(executor);
    requireNonNull(version);

    this.gson = gson != null ? gson : createGson();
    this.executor = executor;
    this.serializerMetadata = new ArtifactAstSerializerMetadata(JSON, version, UTF_8);
    this.astDTOFactoryProvider = new AstDTOFactoryProvider(serializerMetadata);
  }

  @Override
  public InputStream serialize(ArtifactAst artifactAst) {
    if (getDependencyResolutionMode() != COMPILED) {
      throw new IllegalStateException("Artifact AST serialization with `DependencyResolutionMode != COMPILE` would have limited usability when deserialized.");
    }

    // Map the ArtifactAst to the serializable DTO
    ArtifactAstDTO artifactAstDTO = astDTOFactoryProvider.getArtifactAstDTOFactory().from(artifactAst);

    // Serialize it to JSON
    Reader reader = getJsonBodyReader(artifactAstDTO);

    // Create the header
    InputStream bodyInputStream = new ReaderInputStream(reader, serializerMetadata.getCharset());
    return artifactAstSerializerMetadataSerializer
        .addArtifactAstSerializerMetadataToInputStream(bodyInputStream, serializerMetadata);
  }

  private Reader getJsonBodyReader(ArtifactAstDTO artifactAstDTO) {
    CountDownLatch pipesConnected = new CountDownLatch(1);
    PipedReader pipedReader = new PipedReader();

    executor.execute(() -> {
      try (Writer pipedWriter = new PipedWriter(pipedReader)) {
        pipesConnected.countDown();
        gson.toJson(artifactAstDTO, pipedWriter);
      } catch (IOException e) {
        throw new MuleRuntimeException(createStaticMessage("Could not serialize Json"), e);
      }
    });

    try {
      pipesConnected.await();
      return pipedReader;
    } catch (InterruptedException e) {
      throw new MuleRuntimeException(createStaticMessage("Could not serialize Json"), e);
    }
  }

  private Gson createGson() {
    GsonBuilder gsonBuilder = new GsonBuilder();
    gsonBuilder.registerTypeAdapter(ComponentIdentifier.class, new ComponentIdentifierJsonSerializer());
    gsonBuilder.registerTypeAdapter(ComponentLocation.class, new ComponentLocationJsonSerializer());
    gsonBuilder.registerTypeAdapter(TypedComponentIdentifier.class, new TypedComponentIdentifierJsonSerializer());
    gsonBuilder.registerTypeAdapter(ParserAttributesDTO.class, new ParserAttributesJsonTypeAdapter());
    return gsonBuilder.create();
  }

  public Executor getExecutor() {
    return executor;
  }
}
