/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization.json.gson;

import static org.mule.runtime.api.util.classloader.MuleImplementationLoaderUtils.getMuleImplementationsLoader;

import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.google.gson.JsonSerializationContext;
import com.google.gson.JsonSerializer;
import org.mule.runtime.ast.internal.serialization.dto.ParserAttributesDTO;

import java.lang.reflect.Type;
import java.util.HashMap;
import java.util.Map;

/**
 * Type adapter for {@link ParserAttributesDTO} instances. Will take care of serializing the type of each value in order to be
 * able to replicate the same type of instance on deserialization.
 */
public class ParserAttributesJsonTypeAdapter
    implements JsonSerializer<ParserAttributesDTO>, JsonDeserializer<ParserAttributesDTO> {

  @Override
  public ParserAttributesDTO deserialize(JsonElement jsonElement, Type type,
                                         JsonDeserializationContext jsonDeserializationContext)
      throws JsonParseException {
    if (!jsonElement.isJsonObject()) {
      throw new JsonParseException("Expected JSON object, but got: " + jsonElement);
    }

    final JsonObject jsonObject = jsonElement.getAsJsonObject();
    final Map<String, Object> parserAttributes = new HashMap<>();
    jsonObject.entrySet().forEach(entry -> parserAttributes
        .put(entry.getKey(), fromTypedJsonElement(entry.getValue(), jsonDeserializationContext)));

    return new ParserAttributesDTO(parserAttributes);
  }

  @Override
  public JsonElement serialize(ParserAttributesDTO parserAttributesDTO, Type type,
                               JsonSerializationContext jsonSerializationContext) {
    final JsonObject jsonObject = new JsonObject();
    parserAttributesDTO.get().forEach((key, value) -> jsonObject.add(key, toTypedJsonElement(value, jsonSerializationContext)));

    return jsonObject;
  }

  private JsonElement toTypedJsonElement(Object value, JsonSerializationContext jsonSerializationContext) {
    final JsonObject attributeJsonObject = new JsonObject();
    attributeJsonObject.addProperty("type", value.getClass().getName());
    attributeJsonObject.add("value", jsonSerializationContext.serialize(value, value.getClass()));
    return attributeJsonObject;
  }

  private Object fromTypedJsonElement(JsonElement attributeElement, JsonDeserializationContext jsonDeserializationContext) {
    if (!attributeElement.isJsonObject()) {
      throw new JsonParseException("Expected JSON object, but got: " + attributeElement);
    }

    final JsonObject attributeObject = attributeElement.getAsJsonObject();

    // attempts to get the class from the serialized type
    final Type attributeType;
    try {
      attributeType = getMuleImplementationsLoader().loadClass(attributeObject.get("type").getAsString());
    } catch (ClassNotFoundException e) {
      throw new JsonParseException("Type not found" + attributeElement, e);
    }

    // deserializes using the class as type
    return jsonDeserializationContext.deserialize(attributeObject.get("value"), attributeType);
  }
}
