/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.test.internal.serialization;

import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_JSON_DESERIALIZER;
import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_JSON_SERIALIZER;
import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_SERIALIZATION;
import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_SERIALIZATION_VERSIONING;

import static java.util.concurrent.Executors.newSingleThreadExecutor;

import static org.apache.commons.lang3.StringUtils.join;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsInAnyOrder;
import static org.hamcrest.Matchers.hasItems;
import static org.junit.rules.ExpectedException.none;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.spy;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.serialization.ArtifactAstSerializer;
import org.mule.runtime.ast.internal.serialization.ArtifactAstSerializerFactory;

import java.util.List;
import java.util.concurrent.Executor;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Stories;
import io.qameta.allure.Story;

@Feature(AST_SERIALIZATION)
@Stories({
    @Story(AST_SERIALIZATION_VERSIONING),
    @Story(AST_JSON_DESERIALIZER),
    @Story(AST_JSON_SERIALIZER)
})
public class ArtifactAstSerializerFactoryTestCase {

  @Rule
  public ExpectedException exception = none();

  private ArtifactAstSerializerFactory artifactAstSerializerFactory;
  private Executor executor;

  @Before
  public void setUp() {
    executor = spy(newSingleThreadExecutor());
    artifactAstSerializerFactory = new ArtifactAstSerializerFactory(executor, true);
  }

  @Test
  public void testGetSerializerThrowsIllegalArgumentExceptionUnexistingFormat_WhenGettingSerializerForUnexistingFormat() {
    // Given

    // Then
    exception.expect(IllegalArgumentException.class);
    exception.expectMessage("No serializer named 'UNEXISTING' version '1.0' is available. Available serializers are: ["
        + join(artifactAstSerializerFactory.getAvailableSerializers(), ", ") + "]");

    // When
    artifactAstSerializerFactory.getSerializer("UNEXISTING", "1.0");
  }

  @Test
  public void testGetSerializerThrowsIllegalArgumentException123Version_WhenGettingSerializerFor123Version() {
    // Given

    // Then
    exception.expect(IllegalArgumentException.class);
    exception.expectMessage("No serializer named 'JSON' version '12.3' is available. Available serializers are: ["
        + join(artifactAstSerializerFactory.getAvailableSerializers(), ", ") + "]");

    // When
    artifactAstSerializerFactory.getSerializer("JSON", "12.3");
  }

  @Test
  @Issue("MULE-19748")
  public void testGetSerializerReturnsASerializerThatUsesTheExecutorToExecuteARunnable_WhenGettingASerializerAndAnExecutorIsConfigured() {
    // Given
    ArtifactAst artifactAst = mock(ArtifactAst.class);
    when(artifactAst.getErrorTypeRepository()).thenReturn(mock(ErrorTypeRepository.class));

    // Then
    ArtifactAstSerializer jsonSerializer = artifactAstSerializerFactory.getSerializer("JSON", "1.0");
    jsonSerializer.serialize(artifactAst);

    // When
    verify(executor, times(1)).execute(any(Runnable.class));
  }

  @Test
  public void testGetDeserializerThrowsIllegalArgumentExceptionUnexisting2Format_WhenGettingDeserializerForUnexisting2Format() {
    // Given

    // Then
    exception.expect(IllegalArgumentException.class);
    exception.expectMessage("No deserializer named 'UNEXISTING2' version '2.3' is available. Available deserializers are: ["
        + join(artifactAstSerializerFactory.getAvailableSerializers(), ", ") + "]");

    // When
    artifactAstSerializerFactory.getDeserializer("UNEXISTING2", "2.3");
  }

  @Test
  public void testGetDeserializerThrowsIllegalArgumentException123Version_WhenGettingDeserializerFor123Version() {
    // Given

    // Then
    exception.expect(IllegalArgumentException.class);
    exception.expectMessage("No deserializer named 'JSON' version '12.3' is available. Available deserializers are: ["
        + join(artifactAstSerializerFactory.getAvailableSerializers(), ", ") + "]");

    // When
    artifactAstSerializerFactory.getDeserializer("JSON", "12.3");
  }

  @Test
  public void testGetDeserializerThrowsNullPointerException_WhenGettingDeserializerWithNullFormatParameter() {
    // Given

    // Then
    exception.expect(NullPointerException.class);
    exception.expectMessage("format");

    // When
    artifactAstSerializerFactory.getDeserializer(null, "12.3");
  }

  @Test
  public void testGetDeserializerThrowsNullPointerException_WhenGettingDeserializerWithNullSerializerVersionParameter() {
    // Given

    // Then
    exception.expect(NullPointerException.class);
    exception.expectMessage("serializerVersion");

    // When
    artifactAstSerializerFactory.getDeserializer("JSON", null);
  }

  @Test
  public void testGetAvailableSerializers() {
    // Given

    // When
    List<String> availableSerializers = artifactAstSerializerFactory.getAvailableSerializers();

    // Then
    assertThat(availableSerializers, hasItems("JSON[1.0]", "JSON[1.1]"));
  }
}
