/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization.dto.factory;

import org.mule.runtime.ast.internal.serialization.ArtifactAstSerializerMetadata;
import org.mule.runtime.ast.internal.serialization.dto.ArtifactAstSerializerMetadataAware;

/**
 * An abstract factory for DTOs which allows for centralizing some common logic.
 *
 * @param <Type>    The original type.
 * @param <DTOType> The DTO type.
 *
 * @since 1.3, 1.2, 1.1.4
 */
public abstract class AbstractDTOFactory<Type, DTOType> {

  private final ArtifactAstSerializerMetadata serializerMetadata;

  protected AbstractDTOFactory(ArtifactAstSerializerMetadata serializerMetadata) {
    this.serializerMetadata = serializerMetadata;
  }

  /**
   * Creates the DTO from the original object.
   *
   * @param original Original object.
   * @return The DTO that represents the original object.
   */
  public DTOType from(Type original) {
    DTOType dto = doCreateFrom(original);
    inject(dto);
    return dto;
  }

  /**
   * Subclasses should implement this to do the actual DTO creation.
   *
   * @param original Original object.
   * @return The DTO that represents the original object.
   */
  protected abstract DTOType doCreateFrom(Type original);

  private void inject(DTOType dto) {
    if (dto instanceof ArtifactAstSerializerMetadataAware) {
      ((ArtifactAstSerializerMetadataAware) dto).setArtifactAstSerializerMetadata(serializerMetadata);
    }
  }
}
