/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization.dto;

import static org.mule.runtime.ast.api.error.ErrorTypeBuilder.builder;

import static java.lang.String.format;
import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Stream.concat;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.api.message.ErrorType;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

/**
 * A decorator for {@link ErrorTypeRepository} that enriches it with additional related error types.
 */
public class EnrichedErrorTypeRepository implements ErrorTypeRepository {

  private final ErrorTypeRepository delegate;
  private final Map<ComponentIdentifier, ErrorType> deploymentErrorTypes = new HashMap<>();
  private final Map<ComponentIdentifier, ErrorType> deploymentInternalErrorTypes = new HashMap<>();

  /**
   * Constructs a new {@code DeploymentEnricherErrorTypeRepository}.
   *
   * @param delegate the underlying {@link ErrorTypeRepository} to be enriched
   */
  public EnrichedErrorTypeRepository(ErrorTypeRepository delegate) {
    this.delegate = delegate;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ErrorType addErrorType(ComponentIdentifier errorTypeIdentifier, ErrorType parentErrorType) {
    return delegate.getErrorType(errorTypeIdentifier)
        .orElse(createNewErrorType(errorTypeIdentifier, parentErrorType, deploymentErrorTypes));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ErrorType addInternalErrorType(ComponentIdentifier errorTypeIdentifier, ErrorType parentErrorType) {
    return delegate.getErrorType(errorTypeIdentifier)
        .orElse(createNewErrorType(errorTypeIdentifier, parentErrorType, deploymentInternalErrorTypes));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Optional<ErrorType> lookupErrorType(ComponentIdentifier errorTypeComponentIdentifier) {
    return ofNullable(delegate.lookupErrorType(errorTypeComponentIdentifier)
        .orElse(deploymentErrorTypes.get(errorTypeComponentIdentifier)));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Optional<ErrorType> getErrorType(ComponentIdentifier errorTypeIdentifier) {
    return ofNullable(ofNullable(delegate.getErrorType(errorTypeIdentifier)
        .orElse(deploymentErrorTypes.get(errorTypeIdentifier)))
            .orElse(deploymentInternalErrorTypes.get(errorTypeIdentifier)));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Collection<String> getErrorNamespaces() {
    return concat(
                  concat(
                         concat(getErrorTypes().stream(), getInternalErrorTypes().stream()),
                         new HashSet<>(deploymentErrorTypes.values()).stream()),
                  new HashSet<>(deploymentInternalErrorTypes.values()).stream())
                      .map(errorType -> errorType.getNamespace().toUpperCase())
                      .distinct()
                      .collect(toList());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ErrorType getAnyErrorType() {
    return delegate.getAnyErrorType();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ErrorType getSourceErrorType() {
    return delegate.getSourceErrorType();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ErrorType getSourceResponseErrorType() {
    return delegate.getSourceResponseErrorType();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ErrorType getCriticalErrorType() {
    return delegate.getCriticalErrorType();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Set<ErrorType> getErrorTypes() {
    Set<ErrorType> errorTypes = new HashSet<>(delegate.getErrorTypes());
    errorTypes.addAll(deploymentErrorTypes.values());
    return errorTypes;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Set<ErrorType> getInternalErrorTypes() {
    Set<ErrorType> errorTypes = new HashSet<>(delegate.getInternalErrorTypes());
    errorTypes.addAll(deploymentInternalErrorTypes.values());
    return errorTypes;
  }

  /**
   * Builds a new error type with the given identifier and parent.
   *
   * @param identifier the component identifier for the error type
   * @param parent     the parent error type
   * @return a new {@link ErrorType} instance
   * @throws IllegalStateException if an error type with the given identifier already exists
   */
  private ErrorType buildErrorType(ComponentIdentifier identifier, ErrorType parent) {
    if (deploymentErrorTypes.containsKey(identifier) || deploymentInternalErrorTypes.containsKey(identifier)) {
      throw new IllegalStateException(format("An error type with identifier '%s' already exists", identifier));
    }
    return builder()
        .namespace(identifier.getNamespace())
        .identifier(identifier.getName())
        .parentErrorType(parent)
        .build();
  }

  /**
   * Creates a new error type and stores it in the specified error type map.
   *
   * @param errorTypeIdentifier the identifier for the new error type
   * @param parentErrorType     the parent error type
   * @param errorTypes          the map in which to store the new error type
   * @return the newly created {@link ErrorType}
   */
  private ErrorType createNewErrorType(ComponentIdentifier errorTypeIdentifier, ErrorType parentErrorType,
                                       Map<ComponentIdentifier, ErrorType> errorTypes) {
    ErrorType errorType = buildErrorType(errorTypeIdentifier, parentErrorType);
    errorTypes.put(errorTypeIdentifier, errorType);
    return errorType;
  }
}
