/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization.dto.factory;

import static java.util.stream.Collectors.toList;

import org.mule.runtime.api.functional.Either;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.ast.internal.serialization.dto.ParameterValueContainer;
import org.mule.runtime.extension.api.error.ErrorMapping;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class ParameterValueContainerFactory {

  private final AstDTOFactoryProvider astDTOFactoryProvider;

  public ParameterValueContainerFactory(AstDTOFactoryProvider astDTOFactoryProvider) {
    this.astDTOFactoryProvider = astDTOFactoryProvider;
  }

  ParameterValueContainer from(ComponentParameterAst param) {
    Either<String, Object> value = param.getValue();

    if (value.isLeft()) {
      return new ParameterValueContainer(value.getLeft(), null);
    }

    if (value.getRight() instanceof ComponentAst) {
      return new ParameterValueContainer(null, astDTOFactoryProvider.getComponentAstDTOFactory()
          .from(param.<ComponentAst>getValue().getRight()));
    }

    if (value.getRight() instanceof List) {
      List right = (List) value.getRight();

      if (right.isEmpty()) {
        return new ParameterValueContainer(null, value.getRight());
      }

      Object o = right.get(0);

      if (o instanceof ComponentAst) {
        return new ParameterValueContainer(null, param.<List<ComponentAst>>getValue().getRight()
            .stream()
            .map(componentAstItem -> astDTOFactoryProvider.getComponentAstDTOFactory().from(componentAstItem))
            .collect(toList()));
      }

      if (o instanceof ErrorMapping) {
        return new ParameterValueContainer(null, param.<List<ErrorMapping>>getValue().getRight());
      }

      // A list of something other than a component or an error mapping
      return new ParameterValueContainer(null, value.getRight());
    }

    if (value.getRight() instanceof Map) {
      Map<String, Object> right = param.<Map<String, Object>>getValue().getRight();

      Map<String, Object> hashMap = new HashMap<>();

      for (String key : right.keySet()) {
        hashMap.put(key, right.get(key));
      }

      return new ParameterValueContainer(null, hashMap);
    }

    return new ParameterValueContainer(null, value.getRight());
  }

}
