/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.test.internal.serialization.dto.factory;

import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.UNKNOWN;
import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_DTO;
import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_SERIALIZATION;
import static org.mule.runtime.ast.internal.serialization.json.JsonArtifactAstSerializerFormat.JSON;

import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.Arrays.asList;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toSet;
import static java.util.stream.Stream.of;

import static org.hamcrest.MatcherAssert.assertThat;

import static org.mockito.Mockito.inOrder;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.notNullValue;
import static org.mockito.ArgumentMatchers.any;

import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentMetadataAst;
import org.mule.runtime.ast.api.ImportedResource;
import org.mule.runtime.ast.internal.model.ParameterModelUtils;
import org.mule.runtime.ast.internal.serialization.ArtifactAstSerializerMetadata;
import org.mule.runtime.ast.internal.serialization.dto.ArtifactAstDTO;
import org.mule.runtime.ast.internal.serialization.dto.ComponentAstDTO;
import org.mule.runtime.ast.internal.serialization.dto.ErrorTypeRepositoryDTO;
import org.mule.runtime.ast.internal.serialization.dto.ExtensionModelDTO;
import org.mule.runtime.ast.internal.serialization.dto.factory.ArtifactAstDTOFactory;
import org.mule.runtime.ast.internal.serialization.dto.factory.AstDTOFactoryProvider;
import org.mule.runtime.ast.internal.serialization.dto.factory.ComponentAstDTOFactory;
import org.mule.runtime.ast.internal.serialization.dto.factory.ErrorTypeRepositoryDTOFactory;
import org.mule.runtime.ast.internal.serialization.dto.factory.ExtensionModelDTOFactory;
import org.mule.runtime.ast.internal.serialization.resolver.DefaultGenerationInformationResolver;
import org.mule.runtime.ast.testobjects.DummyExtensionModel;
import org.mule.runtime.ast.testobjects.TestArtifactAstFactory;
import org.mule.runtime.ast.testobjects.TestExtensionModelResolver;

import java.util.HashSet;
import java.util.Optional;
import java.util.Set;

import org.junit.Before;
import org.junit.Test;

import org.mockito.InOrder;

import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;

@Feature(AST_SERIALIZATION)
@Story(AST_DTO)
public class ArtifactAstDTOFactoryTestCase {

  private TestArtifactAstFactory testArtifactAstFactory;
  private AstDTOFactoryProvider astDTOFactoryProvider;
  private ArtifactAstDTOFactory innerArtifactAstDTOFactory;
  private ExtensionModelDTOFactory extensionModelDTOFactory;
  private ComponentAstDTOFactory componentAstDTOFactory;
  private ArtifactAstDTOFactory artifactAstDTOFactory;
  private ErrorTypeRepositoryDTOFactory errorRepositoryDTOBuilder;
  private DummyExtensionModel coreExtModel;
  private DummyExtensionModel dummyExtensionModel;
  private Set<ExtensionModel> extModels;
  private DefaultGenerationInformationResolver generationInformationResolver;
  private TestExtensionModelResolver extensionModelResolver;

  @Before
  public void setUp() throws Exception {
    testArtifactAstFactory = new TestArtifactAstFactory();

    innerArtifactAstDTOFactory = mock(ArtifactAstDTOFactory.class);
    extensionModelDTOFactory = mock(ExtensionModelDTOFactory.class);
    componentAstDTOFactory = mock(ComponentAstDTOFactory.class);
    errorRepositoryDTOBuilder = mock(ErrorTypeRepositoryDTOFactory.class);

    astDTOFactoryProvider = new TestAstDTOFactoryProvider();

    artifactAstDTOFactory = new ArtifactAstDTOFactory(astDTOFactoryProvider);

    coreExtModel = new DummyExtensionModel("mule");
    dummyExtensionModel = new DummyExtensionModel("Dummy name");

    extModels = new HashSet<>(asList(coreExtModel, dummyExtensionModel));
    // When parsing the xml, an ImmutableExtensionModel with 'mule' name/prefix is added on top of the input parameters of the
    // parse method, so we need to be able to resolve it on deserialization
    extensionModelResolver = new TestExtensionModelResolver(extModels);
    generationInformationResolver = new DefaultGenerationInformationResolver();
  }

  @Test
  public void testFromArtifactAstDTOReturnsADTOWithTheCorrectAmountOfTopLevelComponentsAndImportedResources_WhenCreatingADTOFromAnArtifactAST() {
    // Given
    AstDTOFactoryProvider astDTOFactoryProvider =
        new AstDTOFactoryProvider(new ArtifactAstSerializerMetadata(JSON, "1.0", UTF_8));
    ArtifactAstDTOFactory artifactAstDTOFactory = new ArtifactAstDTOFactory(astDTOFactoryProvider);

    ArtifactAst artifactAst = mock(ArtifactAst.class);

    ExtensionModel extensionModel = mock(ExtensionModel.class);
    String extensionModelName = "some extension model name";
    when(extensionModel.getName()).thenReturn(extensionModelName);
    when(artifactAst.dependencies()).thenReturn(singleton(extensionModel));

    ComponentAst componentAst = mock(ComponentAst.class);
    when(componentAst.getComponentType()).thenReturn(UNKNOWN);
    when(componentAst.getLocation()).thenReturn(mock(ComponentLocation.class));
    when(artifactAst.topLevelComponents()).thenReturn(singletonList(componentAst));
    when(artifactAst.topLevelComponentsStream()).thenReturn(of(componentAst));

    ErrorTypeRepository errorTypeRepository = mock(ErrorTypeRepository.class);
    when(artifactAst.getErrorTypeRepository()).thenReturn(errorTypeRepository);

    ArtifactAst parentArtifact = mock(ArtifactAst.class);
    Optional<ArtifactAst> optionalParentArtifactAst = Optional.of(parentArtifact);
    when(parentArtifact.getErrorTypeRepository()).thenReturn(errorTypeRepository);
    when(artifactAst.getParent()).thenReturn(optionalParentArtifactAst);

    // When
    ArtifactAstDTO artifactAstDTO = artifactAstDTOFactory.from(artifactAst);

    // Then
    assertThat(artifactAstDTO.topLevelComponents().size(), is(artifactAst.topLevelComponents().size()));

    assertThat(artifactAstDTO.getImportedResources().size(), is(artifactAst.getImportedResources().size()));
    assertThat(artifactAstDTO.getParent(), notNullValue());

  }

  @Test
  public void testMappingSimpleHttpApp() {
    // Given
    ArtifactAst simpleHttpArtifact =
        testArtifactAstFactory.createArtifactFromXmlFile("test-apps/http-listener-with-logger.xml", extModels);

    when(innerArtifactAstDTOFactory.from(any(ArtifactAst.class))).thenReturn(mock(ArtifactAstDTO.class));

    ExtensionModelDTO immutableExtensionModelDto = mock(ExtensionModelDTO.class);
    when(immutableExtensionModelDto.getName()).thenReturn("mule");
    ExtensionModelDTO dummyExtensionModelDTO = mock(ExtensionModelDTO.class);
    when(dummyExtensionModelDTO.getName()).thenReturn("Dummy name");
    when(extensionModelDTOFactory.from(any(ExtensionModel.class))).thenReturn(immutableExtensionModelDto, dummyExtensionModelDTO);

    when(componentAstDTOFactory.from(any(ComponentAst.class))).thenReturn(mock(ComponentAstDTO.class));

    when(errorRepositoryDTOBuilder.from(any(ErrorTypeRepository.class))).thenReturn(mock(ErrorTypeRepositoryDTO.class));

    // When
    ArtifactAstDTO artifactAstDTO = artifactAstDTOFactory.from(simpleHttpArtifact);

    artifactAstDTO.enrich(extensionModelResolver, generationInformationResolver, new ParameterModelUtils());

    // Then
    // Map the parent (no parent, no call)
    verify(innerArtifactAstDTOFactory, never()).from(any(ArtifactAst.class));

    // Map the dependencies
    assertThat(artifactAstDTO.dependencies().size(),
               is(simpleHttpArtifact.dependencies().stream().map(ExtensionModel::getName).collect(toSet()).size()));
    verify(extensionModelDTOFactory, times(1)).from(coreExtModel);
    verify(extensionModelDTOFactory, times(1)).from(dummyExtensionModel);

    // Map the child components (http config and flow)
    InOrder inOrderComponentAst = inOrder(componentAstDTOFactory, componentAstDTOFactory);
    inOrderComponentAst.verify(componentAstDTOFactory, times(2)).from(any(ComponentAst.class));

    verify(errorRepositoryDTOBuilder, times(1)).from(simpleHttpArtifact.getErrorTypeRepository());

    assertThat(artifactAstDTO.topLevelComponentsStream().count(), is(simpleHttpArtifact.topLevelComponentsStream().count()));
    assertThat(artifactAstDTO.getParent(), is(simpleHttpArtifact.getParent()));
  }

  @Test
  @Issue("MULE-19836")
  public void importTagUsingProperty() {
    ImportedResource imported = mock(ImportedResource.class);
    when(imported.getRawResourceLocation()).thenReturn("${someProperty}.xml");
    when(imported.getResourceLocation()).thenReturn("resolved.xml");
    when(imported.getMetadata()).thenReturn(mock(ComponentMetadataAst.class));

    ArtifactAst artifact = mock(ArtifactAst.class);
    when(artifact.getImportedResources()).thenReturn(singleton(imported));

    ArtifactAstDTO artifactAstDTO = artifactAstDTOFactory.from(artifact);

    ImportedResource importedDto = artifactAstDTO.getImportedResources().iterator().next();
    assertThat(importedDto.getRawResourceLocation(), is("${someProperty}.xml"));
  }

  private class TestAstDTOFactoryProvider extends AstDTOFactoryProvider {

    private TestAstDTOFactoryProvider() {
      super(new ArtifactAstSerializerMetadata(JSON, "1.0", UTF_8));
    }

    @Override
    public ArtifactAstDTOFactory getArtifactAstDTOFactory() {
      return innerArtifactAstDTOFactory;
    }

    @Override
    public ExtensionModelDTOFactory getExtensionModelDTOFactory() {
      return extensionModelDTOFactory;
    }

    @Override
    public ComponentAstDTOFactory getComponentAstDTOFactory() {
      return componentAstDTOFactory;
    }

    @Override
    public ErrorTypeRepositoryDTOFactory getErrorRepositoryDTOFactory() {
      return errorRepositoryDTOBuilder;
    }
  }
}
