/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization.dto;

import static java.lang.Thread.currentThread;
import static java.util.stream.Stream.concat;
import static org.apache.commons.lang3.StringUtils.replaceOnce;
import static org.apache.commons.lang3.StringUtils.reverse;
import static org.slf4j.LoggerFactory.getLogger;

import static org.mule.metadata.api.utils.MetadataTypeUtils.getTypeId;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.SUB_FLOW;
import static org.mule.runtime.ast.internal.builder.adapter.MetadataTypeModelAdapter.createKeyValueWrapperTypeModelAdapter;
import static org.mule.runtime.ast.internal.builder.adapter.MetadataTypeModelAdapter.createMetadataTypeModelAdapterWithStereotype;
import static org.mule.runtime.ast.internal.builder.adapter.MetadataTypeModelAdapter.createParameterizedTypeModelAdapter;
import static org.mule.runtime.ast.internal.builder.adapter.MetadataTypeModelAdapter.createSimpleWrapperTypeModelAdapter;
import static org.mule.runtime.extension.api.declaration.type.StreamingStrategyTypeBuilder.REPEATABLE_FILE_STORE_BYTES_STREAM_ALIAS;
import static org.mule.runtime.extension.api.declaration.type.StreamingStrategyTypeBuilder.REPEATABLE_IN_MEMORY_BYTES_STREAM_ALIAS;
import static org.mule.runtime.extension.api.declaration.type.StreamingStrategyTypeBuilder.NON_REPEATABLE_BYTE_STREAM_ALIAS;
import static org.mule.runtime.extension.api.declaration.type.StreamingStrategyTypeBuilder.REPEATABLE_IN_MEMORY_OBJECTS_STREAM_ALIAS;
import static org.mule.runtime.extension.api.declaration.type.StreamingStrategyTypeBuilder.REPEATABLE_FILE_STORE_OBJECTS_STREAM_ALIAS;
import static org.mule.runtime.extension.api.declaration.type.StreamingStrategyTypeBuilder.NON_REPEATABLE_OBJECTS_STREAM_ALIAS;
import static org.mule.runtime.ast.internal.dsl.DslConstants.CORE_PREFIX;
import static org.mule.runtime.extension.internal.dsl.DslConstants.TLS_KEY_STORE_ELEMENT_IDENTIFIER;
import static org.mule.runtime.extension.internal.dsl.DslConstants.TLS_TRUST_STORE_ELEMENT_IDENTIFIER;
import static org.mule.runtime.ast.internal.dsl.DslConstants.KEY_ATTRIBUTE_NAME;
import static org.mule.runtime.ast.internal.dsl.DslConstants.VALUE_ATTRIBUTE_NAME;
import static org.mule.runtime.ast.internal.serialization.util.DslUtils.isSubflow;
import static org.mule.runtime.extension.api.loader.util.InfrastructureTypeUtils.TLS_KEY_STORE_PARAMETER_NAME;
import static org.mule.runtime.extension.api.loader.util.InfrastructureTypeUtils.TLS_TRUST_STORE_PARAMETER_NAME;
import static org.mule.runtime.extension.api.loader.util.InfrastructureTypeUtils.getTypeFor;
import static org.mule.runtime.extension.api.stereotype.MuleStereotypes.ON_ERROR;
import static org.mule.runtime.extension.api.util.ExtensionModelUtils.supportsMultiple;

import static java.util.Collections.unmodifiableCollection;
import static java.util.Collections.unmodifiableList;

import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;

import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.model.SimpleType;
import org.mule.metadata.api.model.UnionType;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.meta.model.ComponentModel;
import org.mule.runtime.api.meta.model.ComposableModel;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.config.ConfigurationModel;
import org.mule.runtime.api.meta.model.connection.ConnectionProviderModel;
import org.mule.runtime.api.meta.model.construct.ConstructModel;
import org.mule.runtime.api.meta.model.nested.NestableElementModel;
import org.mule.runtime.api.meta.model.nested.NestedComponentModel;
import org.mule.runtime.api.meta.model.nested.NestedRouteModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentGenerationInformation;
import org.mule.runtime.ast.api.ComponentMetadataAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.ast.api.model.ExtensionModelHelper;
import org.mule.runtime.ast.api.serialization.ExtensionModelResolver;
import org.mule.runtime.ast.api.util.BaseComponentAst;
import org.mule.runtime.ast.internal.builder.PropertiesResolver;
import org.mule.runtime.ast.internal.model.ParameterModelUtils;
import org.mule.runtime.ast.internal.serialization.resolver.GenerationInformationResolver;
import org.mule.runtime.ast.internal.builder.adapter.MetadataTypeModelAdapter;
import org.mule.runtime.ast.internal.serialization.visitor.DTOMetadataTypeVisitor;
import org.mule.runtime.extension.api.dsl.syntax.DslElementSyntax;
import org.mule.runtime.extension.api.declaration.type.StreamingStrategyTypeBuilder;
import org.mule.runtime.extension.api.property.ClassLoaderModelProperty;
import org.mule.runtime.extension.api.property.NoWrapperModelProperty;
import org.mule.runtime.extension.api.util.ExtensionModelUtils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Optional;

import org.slf4j.Logger;

/**
 * This is a serializable form of a {@link ComponentAst}.
 */
public class ComponentAstDTO extends BaseComponentAst implements ComponentAst {

  private static final Logger LOGGER = getLogger(ComponentAstDTO.class);

  private final List<ComponentAstDTO> directChildren;
  private final String componentType;
  private final ComponentIdentifier identifier;
  private final ComponentLocation location;
  private final ComponentMetadataAstDTO metadata;
  private final Collection<ComponentParameterAstDTO> parameters;
  private final String componentId;
  private final ExtensionModelDTO extensionModel;
  private final String modelName;
  private final ComponentAstDTOModelType modelType;
  private final Map<String, Object> annotations;

  private ComponentGenerationInformationDTO generationInformation;

  private transient ComponentType componentTypeTransient;
  private transient ExtensionModel extensionModelTransient;
  private transient ComponentModel componentModel;
  private transient ConfigurationModel configurationModel;
  private transient ConnectionProviderModel connectionProviderModel;
  private transient NestableElementModel nestableElementModel;
  private transient ParameterizedModel parameterizedModel;
  private transient MetadataType metadataType;
  private transient PropertiesResolver propertiesResolver;

  public ComponentAstDTO(List<ComponentAstDTO> directChildren, String componentType, ComponentIdentifier identifier,
                         ComponentLocation location, ComponentMetadataAstDTO metadata,
                         Collection<ComponentParameterAstDTO> parameters, String componentId, ExtensionModelDTO extensionModel,
                         String modelName, ComponentAstDTOModelType modelType, Map<String, Object> annotations) {
    this.directChildren = directChildren;
    this.componentType = componentType;
    this.componentTypeTransient = getComponentType();
    this.identifier = identifier;
    // force initialization of the location value
    location.getLocation();
    this.location = location;
    this.metadata = metadata;
    this.parameters = new ArrayList<>(parameters);
    this.componentId = componentId;
    this.extensionModel = extensionModel;
    this.modelName = modelName;
    this.modelType = modelType;
    this.annotations = annotations;
  }

  @Override
  public ComponentIdentifier getIdentifier() {
    return this.identifier;
  }

  @Override
  public ComponentType getComponentType() {
    if (this.componentTypeTransient == null) {
      // There is an adaptation for legacy-serialized subflows here
      this.componentTypeTransient = isSubflow(this.identifier) ? SUB_FLOW : ComponentType.valueOf(this.componentType);
    }

    return componentTypeTransient;
  }

  @Override
  public ComponentLocation getLocation() {
    return this.location;
  }

  @Override
  public ComponentMetadataAst getMetadata() {
    return this.metadata;
  }

  @Override
  public ComponentGenerationInformation getGenerationInformation() {
    return this.generationInformation;
  }

  @Override
  public Optional<String> getComponentId() {
    return ofNullable(this.componentId);
  }

  public ComponentAstDTOModelType getModelType() {
    return modelType;
  }

  @Override
  public Map<String, Object> getAnnotations() {
    return annotations;
  }

  @Override
  public ExtensionModel getExtensionModel() {
    return this.extensionModelTransient;
  }

  @Override
  public <M> Optional<M> getModel(Class<M> modelClass) {
    if (modelClass.isInstance(componentModel)) {
      return of(modelClass.cast(componentModel));
    }

    if (modelClass.isInstance(configurationModel)) {
      return of(modelClass.cast(configurationModel));
    }

    if (modelClass.isInstance(connectionProviderModel)) {
      return of(modelClass.cast(connectionProviderModel));
    }

    if (modelClass.isInstance(nestableElementModel)) {
      return of(modelClass.cast(nestableElementModel));
    }

    if (modelClass.isInstance(parameterizedModel)) {
      return of(modelClass.cast(parameterizedModel));
    }

    return empty();
  }

  @Override
  public MetadataType getType() {
    return metadataType;
  }

  public void setType(MetadataType metadataType) {
    this.metadataType = metadataType;
  }

  @Override
  public Collection<ComponentParameterAst> getParameters() {
    // TODO MULE-19734 remove this unmodifiableCollection
    return unmodifiableCollection(this.parameters);
  }

  @Override
  public List<ComponentAst> directChildren() {
    // TODO MULE-19734 remove this unmodifiableList
    return unmodifiableList(this.directChildren);
  }

  public void setGenerationInformation(ComponentGenerationInformationDTO generationInformation) {
    this.generationInformation = generationInformation;
  }

  public ExtensionModelDTO getExtensionModelDTO() {
    return extensionModel;
  }

  public void setComponentModel(ComponentModel model) {
    this.componentModel = model;
  }

  public void setParameterizedModel(ParameterizedModel model) {
    this.parameterizedModel = model;
  }

  public void setConfigurationModel(ConfigurationModel model) {
    this.configurationModel = model;
  }

  public void setConnectionProviderModel(ConnectionProviderModel model) {
    this.connectionProviderModel = model;
  }

  public String getModelName() {
    return modelName;
  }

  public void setExtensionModel(ExtensionModel extensionModel) {
    this.extensionModelTransient = extensionModel;
  }

  public PropertiesResolver getPropertiesResolver() {
    return propertiesResolver;
  }

  public void enrichWithImportedResources(Map<String, ImportedResourceDTO> importResourcesByRawLocation) {
    LOGGER.debug("Enrichment: enrichWithImportedResources: {}", this);
    if (this.metadata != null) {
      this.metadata.enrich(importResourcesByRawLocation);
    }

    this.directChildren.forEach(componentAstDTO -> componentAstDTO.enrichWithImportedResources(importResourcesByRawLocation));

    this.parameters.forEach(param -> param.enrichWithImportedResources(importResourcesByRawLocation));
  }

  public void setPropertiesResolver(PropertiesResolver propertiesResolver) {
    this.propertiesResolver = propertiesResolver;
    this.parameters.forEach(parameter -> parameter.setPropertiesResolver(propertiesResolver));
    this.directChildren.forEach(componentAstDTO -> componentAstDTO.setPropertiesResolver(propertiesResolver));
    if (this.metadata != null) {
      this.metadata.setPropertiesResolver(propertiesResolver);
    }
  }

  /**
   * Given a component, this method will resolve its extension model, its models, its generation information and recursively try
   * to resolve those for its children and parameters
   *
   * @param parentComponentAstDTO         the component that has {@code componentAstDTO} as a direct child, or null if none.
   * @param extensionModelHelper
   * @param extensionModelResolver
   * @param generationInformationResolver
   */
  public void resolveModelsRecursively(ComponentAstDTO parentComponentAstDTO, ExtensionModelHelper extensionModelHelper,
                                       ExtensionModelResolver extensionModelResolver,
                                       GenerationInformationResolver generationInformationResolver,
                                       ParameterModelUtils parameterModelUtils) {
    LOGGER.debug("Enrichment: resolveModelsRecursively({})", this);

    // Resolve extension model of this component
    this.extensionModelTransient = extensionModelResolver.resolve(this.extensionModel.getName());

    this.resolveModelComponent(parentComponentAstDTO, extensionModelHelper, generationInformationResolver);

    // Resolve models for direct children recursively
    this.directChildren.forEach((childComponentAstDTO -> childComponentAstDTO
        .resolveModelsRecursively(this, extensionModelHelper, extensionModelResolver, generationInformationResolver,
                                  parameterModelUtils)));

    this.generationInformation = generationInformationResolver
        .resolveComponentAstGenerationInformation(null, this,
                                                  extensionModelHelper);

    if (parentComponentAstDTO != null
        && getModel(NestedRouteModel.class)
            .map(nrm -> supportsMultiple(nrm)
                && !ON_ERROR.equals(nrm.getStereotype())
                && !nrm.getNestedComponents().get(0).getModelProperty(NoWrapperModelProperty.class).isPresent())
            .orElse(false)
        && "route".equals(getLocation().getParts().get(getLocation().getParts().size() - 2).getPartPath())) {
      // ...remove the temporarily set model that was used to handle the listOfRoutes
      this.parameterizedModel = null;
    }

    List<ComponentParameterAstDTO> filteredParams = new ArrayList<>();
    for (ComponentParameterAstDTO parameter : this.parameters) {
      try {
        parameter.resolveModelParameter(this, extensionModelHelper, extensionModelResolver, generationInformationResolver,
                                        parameterModelUtils);
        filteredParams.add(parameter);
      } catch (IllegalArgumentException e) {
        if (isGlobalErrorHandlerRef(parameter)) {
          // swallow and skip
        } else {
          boolean ownerIsInfra = this.modelType == ComponentAstDTOModelType.INFRASTRUCTURE;
          String paramName = parameter.getModelName();
          String groupName = parameter.getGroupModelName();
          boolean isStreamingParam = (paramName != null && isStreamingStrategyType(paramName))
              || (groupName != null && isStreamingStrategyType(groupName));

          if (ownerIsInfra || isStreamingParam) {
            // Keep unresolved parameters for infra and streaming so validators don't NPE on missing params
            LOGGER.debug("Enrichment: keeping unresolved parameter '{}' (group '{}') for component {}",
                         paramName, groupName, this);
            filteredParams.add(parameter);
          } else {
            throw e;
          }
        }
      }
    }

    this.parameters.clear();
    this.parameters.addAll(filteredParams);
  }

  // can happen because the serialized was generated with a previous version of the core extension model (W-14781155)
  private boolean isGlobalErrorHandlerRef(ComponentParameterAstDTO parameter) {
    return this.extensionModel.getName().equals(CORE_PREFIX)
        && this.getModelName().equals("errorHandler");
  }

  public void resolveModelComponent(ComponentAstDTO parentComponentAstDTO, ExtensionModelHelper extensionModelHelper,
                                    GenerationInformationResolver generationInformationResolver) {
    LOGGER.debug("Enrichment: resolveModelComponent({}, {})", this,
                 this.extensionModelTransient != null ? this.extensionModelTransient.getName() : "(no_ext)");

    switch (modelType) {
      case OPERATION:
        this.resolveOperationModel();
        break;
      case CONSTRUCT:
        if (parentComponentAstDTO != null) {
          // Attempt to fix error handler references after changes from W-14781155
          this.resolveNestedModel(parentComponentAstDTO);
          if (this.getModel(NestedComponentModel.class).isPresent()) {
            break;
          }
        }
        this.resolveConstructModel();
        break;
      case CONFIGURATION:
        this.resolveConfigurationModel();
        break;
      case CONNECTION_PROVIDER:
        this.resolveConnectionProviderModel();
        break;
      case NESTED_COMPONENT:
      case NESTED_ROUTE:
        if (parentComponentAstDTO.getModel(NestedRouteModel.class)
            .map(ExtensionModelUtils::supportsMultiple)
            .orElse(false)) {
          // temporarily set the model that will be used by the children to handle the listOfRoutes...
          this.parameterizedModel = parentComponentAstDTO.getModel(NestedRouteModel.class).get();
        } else {
          this.resolveNestedModel(parentComponentAstDTO);
        }
        break;
      case SOURCE:
        this.resolveSourceModel();
        break;
      case TYPE:
        this.resolveTypeModel(extensionModelHelper);
        break;
      case INFRASTRUCTURE:
        this.resolveInfrastructureModel(extensionModelHelper, generationInformationResolver);
      default:
    }
  }

  private void resolveInfrastructureModel(ExtensionModelHelper extensionModelHelper,
                                          GenerationInformationResolver generationInformationResolver) {
    Optional<MetadataType> tt = getTypeFor(getIdentifier()).flatMap(extensionModelHelper::findMetadataType);
    // try TypeCatalog lookups (by modelName, then by identifier name)
    if (!tt.isPresent()) {
      tt = extensionModelHelper.getTypeCatalog().getType(this.modelName)
          .map(mt -> (MetadataType) mt);

      if (!tt.isPresent()) {
        tt = ofNullable(this.getIdentifier())
            .map(ComponentIdentifier::getName)
            .flatMap(n -> extensionModelHelper.getTypeCatalog().getType(n))
            .map(mt -> (MetadataType) mt);
      }
    }

    if (!tt.isPresent()) {
      String identifierLocalName = this.getIdentifier() != null ? this.getIdentifier().getName() : null;
      if (isStreamingStrategyType(this.modelName)
          || (identifierLocalName != null && isStreamingStrategyType(identifierLocalName))) {
        LOGGER.debug("Enrichment: resolveInfrastructureModel - known streaming strategy type without model: {} - {}",
                     modelType, this.modelName);
      } else {
        LOGGER.warn("Enrichment: resolveInfrastructureModel - NO MODEL RESOLVED!: {} - {}", modelType, this.modelName);
      }
    }
    tt.ifPresent(resolvedType -> {
      LOGGER.debug("Enrichment: resolveInfrastructureModel: {} - {}", modelType, resolvedType);
      this.metadataType = resolvedType;
    });

    if (this.metadataType != null) {
      this.parameterizedModel = createMetadataTypeModelAdapterWithStereotype(this.metadataType, extensionModelHelper)
          .orElseGet(() -> createParameterizedTypeModelAdapter(this.metadataType, extensionModelHelper));
    }

    this.generationInformation = generationInformationResolver
        .resolveComponentAstGenerationInformation(null, this, extensionModelHelper);

    this.parameters.forEach(componentParameterAstDTO -> componentParameterAstDTO
        .resolveInfrastructureParameter(this, extensionModelHelper, generationInformationResolver));
  }

  private void resolveTypeModel(ExtensionModelHelper extensionModelHelper) {
    Optional<? extends MetadataType> t = resolveType(extensionModelHelper, currentThread().getContextClassLoader());
    if (!t.isPresent()) {
      LOGGER.warn("Enrichment: resolveModelComponent - NO MODEL RESOLVED!: {} - {}", getModelType(), this.modelName);
    }
    t.ifPresent(resolvedType -> {
      resolvedType.accept(new DTOMetadataTypeVisitor(this, extensionModelHelper));
      LOGGER.debug("Enrichment: resolveModelComponent: {} - {}", getModelType(), resolvedType);
      this.metadataType = resolvedType;
    });
  }

  private void resolveSourceModel() {
    ExtensionModel extensionModel = this.extensionModelTransient;
    SourceModel sourceModel = extensionModel.getSourceModel(this.modelName).orElseGet(() -> {
      for (ConfigurationModel configModel : extensionModel.getConfigurationModels()) {
        Optional<SourceModel> sourceModelInConfig = configModel.getSourceModel(this.modelName);
        if (sourceModelInConfig.isPresent()) {
          LOGGER.debug("Enrichment: resolveModelComponent: {} - {}", getModelType(), sourceModelInConfig.get().getName());
          return sourceModelInConfig.get();
        }
      }
      LOGGER.warn("Enrichment: resolveModelComponent - NO MODEL RESOLVED!: {} - {}", getModelType(), this.modelName);
      return null;
    });

    this.componentModel = sourceModel;
    this.parameterizedModel = sourceModel;
  }

  private void resolveNestedModel(ComponentAstDTO parentComponentAstDTO) {
    NestableElementModel nestableModel = null;
    for (NestableElementModel nestableModelInParent : parentComponentAstDTO.getModel(ComposableModel.class)
        .orElseThrow(() -> new NoSuchElementException("No composableModel in "
            + parentComponentAstDTO.getExtensionModel().getName() + ":" + parentComponentAstDTO.toString()))
        .getNestedComponents()) {
      if (nestableModelInParent.getName().equals(this.modelName)) {
        LOGGER.debug("Enrichment: resolveModelComponent: {} - {}", getModelType(), nestableModelInParent.getName());
        nestableModel = nestableModelInParent;
        break;
      }
    }

    if (nestableModel == null) {
      LOGGER.warn("Enrichment: resolveModelComponent - NO MODEL RESOLVED!: {} - {}", getModelType(), this.modelName);
    }

    this.parameterizedModel = nestableModel;
  }

  private void resolveConnectionProviderModel() {
    ExtensionModel extensionModel = this.extensionModelTransient;
    ConnectionProviderModel connectionProviderModel = extensionModel.getConnectionProviderModel(this.modelName).orElseGet(() -> {
      for (ConfigurationModel configModel : extensionModel.getConfigurationModels()) {
        Optional<ConnectionProviderModel> connectionProviderModelInConfig =
            configModel.getConnectionProviderModel(this.modelName);
        if (connectionProviderModelInConfig.isPresent()) {
          LOGGER.debug("Enrichment: resolveModelComponent: {} - {}", getModelType(),
                       connectionProviderModelInConfig.get().getName());
          return connectionProviderModelInConfig.get();
        }
      }
      LOGGER.warn("Enrichment: resolveModelComponent - NO MODEL RESOLVED!: {} - {}", getModelType(), this.modelName);
      return null;
    });

    this.connectionProviderModel = connectionProviderModel;
    this.parameterizedModel = connectionProviderModel;
  }

  private void resolveConfigurationModel() {
    ConfigurationModel configurationModel = this.extensionModelTransient.getConfigurationModel(this.modelName).map(m -> {
      LOGGER.debug("Enrichment: resolveModelComponent: {} - {}", getModelType(), m.getName());
      return m;
    }).orElseGet(() -> {
      LOGGER.warn("Enrichment: resolveModelComponent - NO MODEL RESOLVED!: {} - {}", getModelType(), this.modelName);
      return null;
    });

    this.configurationModel = configurationModel;
    this.parameterizedModel = configurationModel;
  }

  private void resolveConstructModel() {
    ConstructModel constructModel = this.extensionModelTransient.getConstructModel(this.modelName).map(m -> {
      LOGGER.debug("Enrichment: resolveModelComponent: {} - {}", getModelType(), m.getName());
      return m;
    }).orElse(null);
    if (constructModel != null) {
      this.componentModel = constructModel;
      this.parameterizedModel = constructModel;
    } else {
      // W-14954379: After changes in the Extension Model, routers (and scopes in Runtime's case), were changed from
      // ConstructModels to OperationModels. We may be deserializing an ast serialized with a previous version of the runtime, so
      // in case we don't find the Construct, we have to search for the Operation
      LOGGER.debug("ConstructModel not found for {}, attempting with OperationModel", this.modelName);
      resolveOperationModel();
    }

  }

  private void resolveOperationModel() {
    ExtensionModel extensionModel = this.extensionModelTransient;
    OperationModel operationModel = extensionModel.getOperationModel(this.modelName).orElseGet(() -> {
      for (ConfigurationModel configModel : extensionModel.getConfigurationModels()) {
        Optional<OperationModel> operationModelInConfig = configModel.getOperationModel(this.modelName);
        if (operationModelInConfig.isPresent()) {
          LOGGER.debug("Enrichment: resolveModelComponent: {} - {}", modelType, operationModelInConfig.get().getName());
          return operationModelInConfig.get();
        }
      }
      LOGGER.warn("Enrichment: resolveModelComponent - NO MODEL RESOLVED!: {} - {}", getModelType(), this.modelName);
      return null;
    });

    this.componentModel = operationModel;
    this.parameterizedModel = operationModel;
  }

  public void resolveMapEntryComponentModels(ComponentAstDTO mapComponentAstDTO,
                                             ComponentParameterAstDTO ownerComponentParameterAst,
                                             ExtensionModelHelper extensionModelHelper,
                                             ExtensionModelResolver extensionModelResolver,
                                             GenerationInformationResolver generationInformationResolver,
                                             ParameterModelUtils parameterModelUtils) {
    ParameterModel ownerParameterModel = ownerComponentParameterAst.getModel();

    this.extensionModelTransient = extensionModelResolver.resolve(getExtensionModelDTO().getName());

    if (!getModel(Object.class).isPresent()) {
      this.parameterizedModel =
          createKeyValueWrapperTypeModelAdapter(KEY_ATTRIBUTE_NAME,
                                                extensionModelHelper.findMetadataType(String.class).orElse(null),
                                                VALUE_ATTRIBUTE_NAME,
                                                ownerParameterModel.getType() instanceof ObjectType
                                                    ? ((ObjectType) (ownerParameterModel.getType())).getOpenRestriction()
                                                        .orElseGet(() -> extensionModelHelper.findMetadataType(String.class)
                                                            .orElse(null))
                                                    : extensionModelHelper.findMetadataType(String.class).orElse(null),
                                                extensionModelHelper);
      this.generationInformation =
          generationInformationResolver.resolveComponentAstGenerationInformation(ownerComponentParameterAst, this,
                                                                                 extensionModelHelper);
    }

    this.parameters.forEach(parameter -> parameter
        .resolveMapEntryAttribute(ownerParameterModel, mapComponentAstDTO.getGenerationInformation(), extensionModelHelper,
                                  extensionModelResolver, generationInformationResolver, this,
                                  parameterModelUtils));
  }

  public Optional<? extends MetadataType> resolveType(ExtensionModelHelper extensionModelHelper,
                                                      ClassLoader applicationClassLoader) {
    Optional<ObjectType> type = extensionModelHelper.getTypeCatalog().getType(this.modelName);

    if (type.isPresent()) {
      return type;
    }

    Optional<? extends MetadataType> subType = this.getSubTypeByName();

    if (subType.isPresent()) {
      return subType;
    }

    return extensionModelHelper
        .findMetadataType(loadClassByName(this.modelName, this.extensionModelTransient, applicationClassLoader));
  }

  public Optional<? extends MetadataType> getSubTypeByName() {
    return this.extensionModelTransient.getSubTypes()
        .stream()
        .flatMap(subtype -> subtype.getSubTypes().stream())
        .filter(subtype -> getTypeId(subtype).map(this.modelName::equals).orElse(false))
        .findFirst();
  }

  public Class<?> loadClassByName(String fullyQualifiedName, ExtensionModel extensionModel, ClassLoader applicationClassLoader) {
    try {
      // Attempt to load from ApplicationClassLoader
      return loadClassFromClassLoader(fullyQualifiedName, applicationClassLoader);
    } catch (ClassNotFoundException e) {
      try {
        // Attempt to load from Plugin ClassLoader
        return loadClassFromClassLoader(fullyQualifiedName, extensionModel.getModelProperty(ClassLoaderModelProperty.class)
            .map(ClassLoaderModelProperty::getClassLoader).orElseThrow(() -> new IllegalStateException(e)));
      } catch (ClassNotFoundException ex) {
        throw new IllegalStateException(ex);
      }
    }
  }

  public Class<?> loadClassFromClassLoader(String fullyQualifiedName, ClassLoader classLoader) throws ClassNotFoundException {
    try {
      // for types that are not declared at the extension level but are inner POJOs of other upper level types from the
      // extension
      return classLoader.loadClass(fullyQualifiedName);
    } catch (ClassNotFoundException e) {
      // Attempt to load inner classes
      return classLoader.loadClass(reverse(replaceOnce(reverse(fullyQualifiedName), ".", "$")));
    }
  }

  /**
   * Given an owner {@link ComponentAstDTO} with a model and a {@link ParameterModel} name. This method resolves the
   * {@link ParameterModel}
   *
   * @param componentParameterAstDTO
   */
  public ParameterModel resolveParameterModel(ComponentParameterAstDTO componentParameterAstDTO) {
    Optional<ParameterizedModel> model = this.getModel(ParameterizedModel.class);

    if (!model.isPresent()) {
      // Infrastructure elements (e.g., TLS, streaming strategies) don't expose a standard ParameterizedModel.
      // Their parameters are defined by a MetadataTypeModelAdapter built from the resolved MetadataType.
      // In that case, resolve the ParameterModel from the adapter.
      if (this.parameterizedModel instanceof MetadataTypeModelAdapter) {
        return componentParameterAstDTO
            .resolveParameterModelFromOwnerMetadataTypeModelAdapter((MetadataTypeModelAdapter) this.parameterizedModel);
      }
      throw new IllegalArgumentException("Could not resolve ParameterModel for [" + componentParameterAstDTO.getModelName()
          + "] in " + this.toString());
    }

    return model
        .flatMap(componentModel -> componentModel.getAllParameterModels().stream()
            .filter(parameterModel -> parameterModel.getName().equals(componentParameterAstDTO.getModelName())).findFirst())
        .orElseGet(() -> {
          if (this.parameterizedModel instanceof MetadataTypeModelAdapter) {
            return componentParameterAstDTO
                .resolveParameterModelFromOwnerMetadataTypeModelAdapter((MetadataTypeModelAdapter) this.parameterizedModel);
          }
          throw new IllegalArgumentException("Could not resolve ParameterModel for ["
              + componentParameterAstDTO.getModelName() + "] in [" + model.get().getName() + "]");
        });
  }

  public Optional<ParameterGroupModel> getParameterGroupModelForSourceModel(ComponentParameterAstDTO componentParameterAstDTO) {
    Optional<SourceModel> sourceModelOptional = getModel(SourceModel.class);

    if (!sourceModelOptional.isPresent()) {
      return empty();
    }

    SourceModel sourceModel = sourceModelOptional.get();

    List<ParameterGroupModel> errorCallbackParameterGroupModels =
        sourceModel.getErrorCallback().map(ParameterizedModel::getParameterGroupModels).orElse(new ArrayList<>());

    List<ParameterGroupModel> successCallbackParameterGroupModels =
        sourceModel.getSuccessCallback().map(ParameterizedModel::getParameterGroupModels).orElse(new ArrayList<>());

    return concat(errorCallbackParameterGroupModels.stream(), successCallbackParameterGroupModels.stream())
        .filter(parameterGroupModel -> parameterGroupModel.getName().equals(componentParameterAstDTO.getGroupModelName()))
        .findFirst();
  }

  /**
   * Given an owner {@link ComponentAstDTO} with a model and a {@link ParameterGroupModel} name. This method resolves the
   * {@link ParameterGroupModel}
   *
   *
   *
   * @param componentParameterAstDTO@return the resolved {@link ParameterGroupModel}
   */
  public ParameterGroupModel resolveParameterGroupModel(ComponentParameterAstDTO componentParameterAstDTO) {
    return getModel(ParameterizedModel.class)
        .flatMap(parameterizedModel -> parameterizedModel.getParameterGroupModels().stream()
            .filter(parameterGroupModel -> parameterGroupModel.getName().equals(componentParameterAstDTO.getGroupModelName()))
            .findFirst())
        .orElseGet(() -> getParameterGroupModelForSourceModel(componentParameterAstDTO)
            .orElseGet(() -> {
              // MetadataTypeModelAdapter does not expose ParameterGroupModels, so return gracefully
              if (this.parameterizedModel instanceof MetadataTypeModelAdapter) {
                String group = componentParameterAstDTO.getGroupModelName();
                if (TLS_KEY_STORE_PARAMETER_NAME.equals(group) || TLS_TRUST_STORE_PARAMETER_NAME.equals(group)
                    || TLS_KEY_STORE_ELEMENT_IDENTIFIER.equals(group) || TLS_TRUST_STORE_ELEMENT_IDENTIFIER.equals(group)
                    || isStreamingStrategyType(group)) {
                  return null;
                }
              }
              throw new IllegalArgumentException("Could not resolve ParameterGroupModel ["
                  + componentParameterAstDTO.getGroupModelName() + "]");
            }));
  }

  private boolean isStreamingStrategyType(String name) {
    return REPEATABLE_FILE_STORE_BYTES_STREAM_ALIAS.equals(name)
        || REPEATABLE_IN_MEMORY_BYTES_STREAM_ALIAS.equals(name)
        || NON_REPEATABLE_BYTE_STREAM_ALIAS.equals(name)
        || REPEATABLE_IN_MEMORY_OBJECTS_STREAM_ALIAS.equals(name)
        || REPEATABLE_FILE_STORE_OBJECTS_STREAM_ALIAS.equals(name)
        || NON_REPEATABLE_OBJECTS_STREAM_ALIAS.equals(name);
  }

  public void resolveSimpleTypeComponentModel(ComponentParameterAstDTO ownerComponentParameterAst, SimpleType simpleType,
                                              ExtensionModelResolver extensionModelResolver,
                                              ExtensionModelHelper extensionModelHelper,
                                              GenerationInformationResolver generationInformationResolver,
                                              ParameterModelUtils parameterModelUtils) {
    this.extensionModelTransient = extensionModelResolver.resolve(getExtensionModelDTO().getName());
    this.parameterizedModel = createSimpleWrapperTypeModelAdapter(simpleType, extensionModelHelper);
    this.generationInformation = generationInformationResolver
        .resolveComponentAstGenerationInformation(ownerComponentParameterAst, this, extensionModelHelper);

    this.parameters.forEach(componentParameterAstDTO -> componentParameterAstDTO
        .resolveSimpleTypeComponentParameter(this, simpleType, extensionModelHelper, generationInformationResolver,
                                             parameterModelUtils));
  }

  boolean isWrapped(ComponentParameterAstDTO componentParameterAstDTO, ExtensionModelHelper extensionModelHelper) {
    if (this.generationInformation == null) {
      return false;
    }

    DslElementSyntax syntax =
        this.generationInformation.getSyntax()
            .flatMap(s -> s.getChild(componentParameterAstDTO.getModel().getName()))
            .orElseGet(() -> extensionModelHelper.resolveDslElementModel(componentParameterAstDTO.getModel(),
                                                                         getIdentifier()));

    return syntax.isWrapped();
  }
}
