/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization.resolver;

import static org.slf4j.LoggerFactory.getLogger;

import org.mule.metadata.api.model.MetadataType;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.meta.NamedObject;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.ast.api.ComponentGenerationInformation;
import org.mule.runtime.ast.api.model.ExtensionModelHelper;
import org.mule.runtime.ast.internal.serialization.dto.ComponentAstDTO;
import org.mule.runtime.ast.internal.serialization.dto.ComponentGenerationInformationDTO;
import org.mule.runtime.ast.internal.serialization.dto.ComponentParameterAstDTO;
import org.mule.runtime.extension.api.dsl.syntax.DslElementSyntax;

import java.util.Optional;

import org.slf4j.Logger;

/**
 * Resolves the {@link ComponentGenerationInformationDTO}
 */
public class DefaultGenerationInformationResolver implements GenerationInformationResolver {

  private static final Logger LOGGER = getLogger(DefaultGenerationInformationResolver.class);

  /**
   * Resolves the {@link ComponentGenerationInformationDTO} of a {@link ComponentAstDTO}
   *
   *
   * @param ownerComponentParameterAst
   * @param componentAstDTO            the {@link ComponentAstDTO} whose {@link ComponentGenerationInformationDTO} needs to be
   *                                   resolved
   * @param extensionModelHelper
   * @return the resolved {@link ComponentGenerationInformationDTO}
   */
  @Override
  public ComponentGenerationInformationDTO resolveComponentAstGenerationInformation(ComponentParameterAstDTO ownerComponentParameterAst,
                                                                                    ComponentAstDTO componentAstDTO,
                                                                                    ExtensionModelHelper extensionModelHelper) {

    LOGGER.debug("Enrichment: resolveComponentAstGenerationInformation({},{})", ownerComponentParameterAst, componentAstDTO);

    return new ComponentGenerationInformationDTO(resolveComponentAstSyntax(extensionModelHelper,
                                                                           ownerComponentParameterAst != null
                                                                               ? ownerComponentParameterAst.getModel()
                                                                               : componentAstDTO
                                                                                   .getModel(ParameterizedModel.class)
                                                                                   .orElse(null),
                                                                           componentAstDTO.getType(),
                                                                           componentAstDTO.getIdentifier(),
                                                                           componentAstDTO.getExtensionModel()));
  }

  /**
   * Resolves the {@link ComponentGenerationInformationDTO} of a {@link ComponentParameterAstDTO}
   *
   * @param componentParameterAstDTO the {@link ComponentParameterAstDTO} whose {@link ComponentGenerationInformationDTO} needs to
   *                                 be resolved
   * @param ownerComponent           the owner {@link ComponentAstDTO}
   * @param extensionModelHelper
   * @return the resolved {@link ComponentGenerationInformationDTO}
   */
  @Override
  public ComponentGenerationInformationDTO resolveComponentParameterAstGenerationInformation(ComponentParameterAstDTO componentParameterAstDTO,
                                                                                             ComponentAstDTO ownerComponent,
                                                                                             ExtensionModelHelper extensionModelHelper) {
    DslElementSyntax syntax =
        ownerComponent.getGenerationInformation().getSyntax()
            .flatMap(s -> s.getChild(componentParameterAstDTO.getModel().getName()))
            .orElseGet(() -> extensionModelHelper.resolveDslElementModel(componentParameterAstDTO.getModel(),
                                                                         ownerComponent.getIdentifier()));

    if (!syntax.isWrapped()) {
      return new ComponentGenerationInformationDTO(syntax);
    }

    if (componentParameterAstDTO.isValueAComponentAstDTO()) {
      ComponentAstDTO componentAstDTO = (ComponentAstDTO) componentParameterAstDTO.getValue().getValue().get();
      return new ComponentGenerationInformationDTO(extensionModelHelper
          .resolveDslElementModel(componentAstDTO.getType(),
                                  componentAstDTO.getIdentifier().getNamespace())
          .get());
    }

    return new ComponentGenerationInformationDTO(syntax);
  }

  /**
   * Resolves the {@link ComponentGenerationInformationDTO} of a {@link ComponentParameterAstDTO}
   *
   * @param componentParameterAstDTO the {@link ComponentParameterAstDTO} whose {@link ComponentGenerationInformationDTO} needs to
   *                                 be resolved
   * @param ownerComponent           the owner {@link ComponentAstDTO}
   * @param extensionModelHelper
   * @return the resolved {@link ComponentGenerationInformationDTO}
   */
  @Override
  public ComponentGenerationInformationDTO resolveRegularComponentParameterAstGenerationInformation(ComponentParameterAstDTO componentParameterAstDTO,
                                                                                                    ComponentAstDTO ownerComponent,
                                                                                                    ExtensionModelHelper extensionModelHelper) {
    DslElementSyntax syntax =
        ownerComponent.getGenerationInformation().getSyntax()
            .flatMap(s -> s.getContainedElement(componentParameterAstDTO.getModel().getName()))
            .orElseGet(() -> extensionModelHelper.resolveDslElementModel(componentParameterAstDTO.getModel(),
                                                                         ownerComponent.getIdentifier()));

    return new ComponentGenerationInformationDTO(syntax);
  }

  /**
   * Resolves the {@link ComponentGenerationInformationDTO} of a {@link ComponentParameterAstDTO}
   *
   * @param componentParameterAstDTO the {@link ComponentParameterAstDTO} whose {@link ComponentGenerationInformationDTO} needs to
   *                                 be resolved
   * @param ownerComponent           the owner {@link ComponentAstDTO}
   * @param extensionModelHelper
   * @return the resolved {@link ComponentGenerationInformationDTO}
   */
  @Override
  public ComponentGenerationInformationDTO resolveWrappedComponentParameterAstGenerationInformation(ComponentParameterAstDTO componentParameterAstDTO,
                                                                                                    ComponentAstDTO ownerComponent,
                                                                                                    ExtensionModelHelper extensionModelHelper) {
    DslElementSyntax syntax =
        ownerComponent.getGenerationInformation().getSyntax()
            .flatMap(s -> s.getChild(componentParameterAstDTO.getModel().getName()))
            .orElseGet(() -> extensionModelHelper.resolveDslElementModel(componentParameterAstDTO.getModel(),
                                                                         ownerComponent.getIdentifier()));

    if (componentParameterAstDTO.isValueAComponentAstDTO()) {
      ComponentAstDTO componentAstDTO = (ComponentAstDTO) componentParameterAstDTO.getValue().getValue().get();
      return new ComponentGenerationInformationDTO(extensionModelHelper
          .resolveDslElementModel(componentAstDTO.getType(),
                                  componentAstDTO.getIdentifier().getNamespace())
          .get());
    }

    return new ComponentGenerationInformationDTO(syntax);
  }

  @Override
  public ComponentGenerationInformationDTO resolveGenerationInformationThroughParent(String parameterName,
                                                                                     ComponentAstDTO component,
                                                                                     ParameterModel ownerParameterModel,
                                                                                     ComponentGenerationInformation ownerParameterGenerationInformation) {
    Optional<DslElementSyntax> child =
        ownerParameterGenerationInformation.getSyntax().flatMap(stx -> stx.getChild(ownerParameterModel.getName()));

    if (!child.isPresent()) {
      return null;
    }

    DslElementSyntax syntax = child.get();
    Optional<DslElementSyntax> genericSyntax = syntax.getGenerics().values().stream()
        .filter(dslElementSyntax -> dslElementSyntax.getElementName().equals(component.getIdentifier().getName()))
        .findFirst();

    return genericSyntax
        .map(dslElementSyntax -> new ComponentGenerationInformationDTO(dslElementSyntax.getContainedElement(parameterName).get()))
        .orElse(null);

  }

  @Override
  public ComponentGenerationInformationDTO crateComponentGenerationInformationFromParent(ComponentAstDTO componentAstDTO,
                                                                                         ComponentParameterAstDTO componentParameterAstDTO) {
    return new ComponentGenerationInformationDTO(componentAstDTO.getGenerationInformation()
        .getSyntax()
        .flatMap(syntax -> syntax.getChild(componentParameterAstDTO.getModelName()))
        .orElse(componentAstDTO
            .getGenerationInformation()
            .getSyntax()
            .flatMap(dslElementSyntax -> dslElementSyntax.getContainedElement(componentParameterAstDTO.getModelName()))
            .orElse(null)));
  }

  private DslElementSyntax resolveComponentAstSyntax(ExtensionModelHelper extensionModelHelper, NamedObject model,
                                                     MetadataType componentAstDtoType,
                                                     ComponentIdentifier componentAstDtoIdentifier,
                                                     ExtensionModel extensionModel) {
    if (componentAstDtoType == null) {
      return extensionModelHelper.resolveDslElementModel(model, componentAstDtoIdentifier);
    }

    // Make sure the elementDsl is resolved consistently with previous implementations
    return extensionModelHelper.resolveDslElementModel(componentAstDtoType, extensionModel)
        .orElseGet(() -> extensionModelHelper.resolveDslElementModel(model, componentAstDtoIdentifier));
  }

}
