/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization.json.gson;

import org.mule.runtime.ast.internal.serialization.ArtifactAstSerializerMetadata;
import org.mule.runtime.ast.internal.serialization.dto.ArtifactAstSerializerMetadataAware;

import java.io.IOException;

import com.google.gson.Gson;
import com.google.gson.TypeAdapter;
import com.google.gson.TypeAdapterFactory;
import com.google.gson.reflect.TypeToken;
import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonWriter;

/**
 * A {@link TypeAdapterFactory} that takes care of enriching deserialized {@link ArtifactAstSerializerMetadataAware} DTOs with the
 * corresponding {@link ArtifactAstSerializerMetadata}.
 *
 * @since 1.3, 1.2, 1.1.4
 */
public class ArtifactAstSerializerMetadataAwareTypeAdapterFactory implements TypeAdapterFactory {

  private final ArtifactAstSerializerMetadata serializerMetadata;

  public ArtifactAstSerializerMetadataAwareTypeAdapterFactory(ArtifactAstSerializerMetadata serializerMetadata) {
    this.serializerMetadata = serializerMetadata;
  }

  @Override
  public <T> TypeAdapter<T> create(Gson gson, TypeToken<T> typeToken) {
    if (!ArtifactAstSerializerMetadataAware.class.isAssignableFrom(typeToken.getRawType())) {
      return null;
    }

    TypeAdapter<T> delegate = gson.getDelegateAdapter(this, typeToken);
    return (TypeAdapter<T>) new ArtifactAstSerializerMetadataAwareTypeAdapter((TypeAdapter<ArtifactAstSerializerMetadataAware>) delegate,
                                                                              serializerMetadata);
  }

  private static class ArtifactAstSerializerMetadataAwareTypeAdapter extends TypeAdapter<ArtifactAstSerializerMetadataAware> {

    private final TypeAdapter<ArtifactAstSerializerMetadataAware> delegate;
    private final ArtifactAstSerializerMetadata serializerMetadata;

    private ArtifactAstSerializerMetadataAwareTypeAdapter(TypeAdapter<ArtifactAstSerializerMetadataAware> delegate,
                                                          ArtifactAstSerializerMetadata serializerMetadata) {
      this.delegate = delegate;
      this.serializerMetadata = serializerMetadata;
    }

    @Override
    public void write(JsonWriter jsonWriter, ArtifactAstSerializerMetadataAware artifactAstSerializerMetadataAware)
        throws IOException {
      delegate.write(jsonWriter, artifactAstSerializerMetadataAware);
    }

    @Override
    public ArtifactAstSerializerMetadataAware read(JsonReader jsonReader) throws IOException {
      ArtifactAstSerializerMetadataAware obj = delegate.read(jsonReader);
      obj.setArtifactAstSerializerMetadata(serializerMetadata);
      return obj;
    }
  }
}
