/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.test.internal.serialization.dto.factory;

import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.UNKNOWN;
import static org.mule.runtime.api.functional.Either.empty;
import static org.mule.runtime.api.functional.Either.left;
import static org.mule.runtime.api.functional.Either.right;
import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_DTO;
import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_SERIALIZATION;
import static org.mule.runtime.ast.internal.serialization.json.JsonArtifactAstSerializerFormat.JSON;

import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.functional.Either;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.ast.api.ComponentGenerationInformation;
import org.mule.runtime.ast.api.ComponentMetadataAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.ast.api.ParameterResolutionException;
import org.mule.runtime.ast.internal.serialization.ArtifactAstSerializerMetadata;
import org.mule.runtime.ast.internal.serialization.dto.ComponentAstDTO;
import org.mule.runtime.ast.internal.serialization.dto.ComponentMetadataAstDTO;
import org.mule.runtime.ast.internal.serialization.dto.ComponentParameterAstDTO;
import org.mule.runtime.ast.internal.serialization.dto.factory.AstDTOFactoryProvider;
import org.mule.runtime.ast.internal.serialization.dto.factory.ComponentMetadataAstDTOFactory;
import org.mule.runtime.extension.api.error.ErrorMapping;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.junit.Before;
import org.junit.Test;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;

@Feature(AST_SERIALIZATION)
@Story(AST_DTO)
public class ComponentParameterAstDTOFactoryTestCase {

  private String unresolvedValue;
  private String resolvedRawValue;
  private Either<String, Object> value;
  private boolean defaultValue;

  private ComponentParameterAst componentParameterAst;
  private ComponentParameterAstDTO componentParameterAstDTO;

  private AstDTOFactoryProvider astDTOFactoryProvider;

  private ComponentMetadataAst componentMetadataAst;
  private ComponentMetadataAstDTOFactory componentMetadataAstDTOFactory;

  private ParameterModel parameterModel;

  @Before
  public void setUp() throws Exception {

    astDTOFactoryProvider = new AstDTOFactoryProvider(new ArtifactAstSerializerMetadata(JSON, "1.0", UTF_8)) {

      @Override
      public ComponentMetadataAstDTOFactory getComponentMetadataAstDTOFactory() {
        return componentMetadataAstDTOFactory;
      }
    };
    unresolvedValue = null;
    resolvedRawValue = null;
    value = empty();
    defaultValue = false;
    parameterModel = mock(ParameterModel.class);
    when(parameterModel.getName()).thenReturn("some parameter model name");

    componentParameterAst = new TestComponentParameterAst();
  }

  @Test
  public void testBuildReturnsComponentParameterAstDTOWithInputSerializableAttributes_WhenBuildingFromComponentParameterAstWithSetSerializableAttributes() {
    // Given
    this.unresolvedValue = "some unresolved value";
    this.resolvedRawValue = "some resolved raw value";
    this.value = right("some value");
    this.defaultValue = true;

    // When
    componentParameterAstDTO = astDTOFactoryProvider.getComponentParameterAstDTOFactory().from(componentParameterAst);

    // Then
    assertThat(componentParameterAstDTO.getRawValue(), is(componentParameterAst.getRawValue()));
    assertThat(componentParameterAstDTO.getResolvedRawValue(), is(this.value.getValue().get()));
    assertThat(componentParameterAstDTO.getValue().getRight(), is(componentParameterAst.getValue().getRight()));
    assertThat(componentParameterAstDTO.isDefaultValue(), is(componentParameterAst.isDefaultValue()));
  }

  @Test
  public void testBuildReturnsComponentParameterAstDTOWithComponentAstAsValue_WhenBuildingFromComponentParameterAstWithComponentAstAsValue() {
    // Given
    this.unresolvedValue = "some unresolved value";
    this.resolvedRawValue = "some resolved raw value";
    this.value = right(testComponentAstDTO());
    this.defaultValue = true;

    // When
    componentParameterAstDTO = astDTOFactoryProvider.getComponentParameterAstDTOFactory().from(componentParameterAst);

    // Then
    assertThat(componentParameterAstDTO.getRawValue(), is(componentParameterAst.getRawValue()));
    assertThat(componentParameterAstDTO.getResolvedRawValue(), is(componentParameterAst.getRawValue()));
    assertThat(((ComponentAstDTO) componentParameterAstDTO.getValue().getRight()).getIdentifier().getName(),
               is(((ComponentAstDTO) componentParameterAst.getValue().getRight()).getIdentifier().getName()));
    assertThat(componentParameterAstDTO.isDefaultValue(), is(componentParameterAst.isDefaultValue()));
  }

  @Test
  public void testBuildReturnsComponentParameterAstDTOWithListOfComponentAstsAsValue_WhenBuildingFromComponentParameterAstWithListOfComponentAstsAsValue() {
    // Given
    this.unresolvedValue = "some unresolved value";
    this.resolvedRawValue = "some resolved raw value";
    ComponentAstDTO componentAstDTO = testComponentAstDTO();
    this.value = right(singletonList(componentAstDTO));
    this.defaultValue = true;

    // When
    componentParameterAstDTO = astDTOFactoryProvider.getComponentParameterAstDTOFactory().from(componentParameterAst);

    // Then
    assertThat(componentParameterAstDTO.getRawValue(), is(componentParameterAst.getRawValue()));
    assertThat(componentParameterAstDTO.getResolvedRawValue(), is(componentParameterAst.getRawValue()));
    assertThat(((List<ComponentAstDTO>) componentParameterAstDTO.getValue().getRight()).get(0).getIdentifier().getName(),
               is(((List<ComponentAstDTO>) componentParameterAst.getValue().getRight()).get(0).getIdentifier().getName()));
    assertThat(componentParameterAstDTO.isDefaultValue(), is(componentParameterAst.isDefaultValue()));
  }

  @Test
  public void testBuildReturnsComponentParameterAstDTOWithEmptyListValue_WhenBuildingFromComponentParameterAstWithEmptyListAsValue() {
    // Given
    this.unresolvedValue = "some unresolved value";
    this.resolvedRawValue = "some resolved raw value";
    this.value = right(emptyList());
    this.defaultValue = true;

    // When
    componentParameterAstDTO = astDTOFactoryProvider.getComponentParameterAstDTOFactory().from(componentParameterAst);

    // Then
    assertThat(componentParameterAstDTO.getRawValue(), is(componentParameterAst.getRawValue()));
    assertThat(componentParameterAstDTO.getResolvedRawValue(), is(componentParameterAst.getRawValue()));
    assertThat(((List<ComponentAstDTO>) componentParameterAstDTO.getValue().getRight()).size(),
               is(((List<ComponentAstDTO>) componentParameterAst.getValue().getRight()).size()));
    assertThat(componentParameterAstDTO.isDefaultValue(), is(componentParameterAst.isDefaultValue()));
  }

  @Test
  public void testBuildReturnsComponentParameterAstDTOWithErrorMappingListValue_WhenBuildingFromComponentParameterAstWithErrorMappingListAsValue() {
    // Given
    this.unresolvedValue = "some unresolved value";
    this.resolvedRawValue = "some resolved raw value in the original";
    this.value = right(singletonList(new ErrorMapping("source", "target")));
    this.defaultValue = true;

    // When
    componentParameterAstDTO = astDTOFactoryProvider.getComponentParameterAstDTOFactory().from(componentParameterAst);

    // Then
    assertThat(componentParameterAstDTO.getRawValue(), is(componentParameterAst.getRawValue()));
    assertThat(componentParameterAstDTO.getResolvedRawValue(), is(componentParameterAst.getRawValue()));
    assertThat(((List<ErrorMapping>) componentParameterAstDTO.getValue().getRight()).size(),
               is(((List<ErrorMapping>) componentParameterAst.getValue().getRight()).size()));
    assertThat(((List<ErrorMapping>) componentParameterAstDTO.getValue().getRight()).get(0).getSource(), is("source"));
    assertThat(((List<ErrorMapping>) componentParameterAstDTO.getValue().getRight()).get(0).getTarget(), is("target"));
    assertThat(componentParameterAstDTO.isDefaultValue(), is(componentParameterAst.isDefaultValue()));
  }

  @Test
  public void testBuildReturnsComponentParameterAstDTOWithStringListValue_WhenBuildingFromComponentParameterAstWithStringListAsValue() {
    // Given
    this.unresolvedValue = "some unresolved value";
    this.resolvedRawValue = "some resolved raw value";
    this.value = right(asList("Aruba", "Jamaica", "Bermuda", "Bahamas", "Key Largo", "Montego", "Kokomo"));
    this.defaultValue = true;

    // When
    componentParameterAstDTO = astDTOFactoryProvider.getComponentParameterAstDTOFactory().from(componentParameterAst);

    // Then
    assertThat(componentParameterAstDTO.getRawValue(), is(componentParameterAst.getRawValue()));
    assertThat(componentParameterAstDTO.getResolvedRawValue(), is(componentParameterAst.getRawValue()));
    assertThat(((List<String>) componentParameterAstDTO.getValue().getRight()).size(),
               is(((List<String>) componentParameterAst.getValue().getRight()).size()));
    assertThat(componentParameterAstDTO.isDefaultValue(), is(componentParameterAst.isDefaultValue()));
  }

  @Test
  public void testBuildReturnsComponentParameterAstDTOWithLinkedTreeMapValue_WhenBuildingFromComponentParameterAstWithLinkedTreeMapAsValue() {
    // Given
    this.unresolvedValue = "some unresolved value";
    this.resolvedRawValue = "some resolved raw value";
    LinkedHashMap<Object, Object> map = new LinkedHashMap<>();
    map.put("some key", "some value");
    this.value = right(map);
    this.defaultValue = true;

    // When
    componentParameterAstDTO = astDTOFactoryProvider.getComponentParameterAstDTOFactory().from(componentParameterAst);

    // Then
    assertThat(componentParameterAstDTO.getRawValue(), is(componentParameterAst.getRawValue()));
    assertThat(componentParameterAstDTO.getResolvedRawValue(), is(componentParameterAst.getRawValue()));
    assertThat(((Map<Object, Object>) componentParameterAstDTO.getValue().getRight()).get("some key"),
               is(((Map<Object, Object>) componentParameterAst.getValue().getRight()).get("some key")));
    assertThat(((Map<Object, Object>) componentParameterAstDTO.getValue().getRight()).get("some key"), is("some value"));
    assertThat(componentParameterAstDTO.isDefaultValue(), is(componentParameterAst.isDefaultValue()));
  }

  @Test
  public void testBuildReturnsComponentParameterAstDTOWithMetadataDTO_WhenBuildingFromComponentParameterAstWithMetadata() {
    // Given
    // This will be returned by the ComponentParameterAst.getMetadata()
    componentMetadataAst = mock(ComponentMetadataAst.class);

    // Configure the metadata builder mock that is returned by the AstDtoBuilderProvider
    ComponentMetadataAstDTO componentMetadataAstDTO =
        new ComponentMetadataAstDTO(null, null, null, null, null, null, null, null, null, null);
    componentMetadataAstDTOFactory = mock(ComponentMetadataAstDTOFactory.class);
    when(componentMetadataAstDTOFactory.from(componentMetadataAst)).thenReturn(componentMetadataAstDTO);

    // When
    componentParameterAstDTO = astDTOFactoryProvider.getComponentParameterAstDTOFactory().from(componentParameterAst);

    // Then
    assertThat(componentParameterAstDTO.getMetadata(), is(of(componentMetadataAstDTO)));
  }

  private ComponentAstDTO testComponentAstDTO() {
    return new ComponentAstDTO(
                               emptyList(),
                               UNKNOWN.toString(),
                               ComponentIdentifier.builder().name("a name").namespace("prefix").namespaceUri("uri").build(),
                               mock(ComponentLocation.class),
                               null,
                               emptyList(),
                               null,
                               null,
                               null,
                               null,
                               null);
  }

  private class TestComponentParameterAst implements ComponentParameterAst {

    @Override
    public ParameterModel getModel() {
      return parameterModel;
    }

    @Override
    public ParameterGroupModel getGroupModel() {
      return null;
    }

    @Override
    public <T> Either<String, T> getValue() {
      return (Either<String, T>) value;
    }

    @Override
    public <T> Either<String, Either<ParameterResolutionException, T>> getValueOrResolutionError() {
      try {
        return getValue().mapRight(fixedValue -> right((T) fixedValue));
      } catch (ParameterResolutionException e) {
        return right(left(e));
      }
    }

    @Override
    public String getRawValue() {
      return unresolvedValue;
    }

    @Override
    public String getResolvedRawValue() {
      return resolvedRawValue;
    }

    @Override
    public Optional<ComponentMetadataAst> getMetadata() {
      return ofNullable(componentMetadataAst);
    }

    @Override
    public ComponentGenerationInformation getGenerationInformation() {
      return null;
    }

    @Override
    public boolean isDefaultValue() {
      return defaultValue;
    }
  }
}
