/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.api.serialization;

import org.mule.runtime.ast.internal.serialization.ArtifactAstSerializerFactory;
import org.mule.runtime.ast.internal.serialization.DefaultArtifactAstDeserializer;

import java.util.concurrent.Executor;

/**
 * This class provides serializers and deserializers instances for {@link org.mule.runtime.ast.api.ArtifactAst}s.
 */
public final class ArtifactAstSerializerProvider {

  private final ArtifactAstSerializerFactory artifactAstSerializerFactory;

  public ArtifactAstSerializerProvider() {
    this(new ArtifactAstSerializerFactory());
  }

  public ArtifactAstSerializerProvider(Executor executor) {
    this(new ArtifactAstSerializerFactory(executor, true));
  }

  ArtifactAstSerializerProvider(ArtifactAstSerializerFactory artifactAstSerializerFactory) {
    this.artifactAstSerializerFactory = artifactAstSerializerFactory;
  }

  /**
   * This method returns a serializer for a given desired output format. The format is a string that identifies the serializer in
   * the {@link ArtifactAstSerializerFactory}. Valid values for format strings are defined as static constants in
   * {@link ArtifactAstSerializerFactory} class.
   * <p>
   * A serializer instance returned by this provider may not be thread-safe so it should no be used for serializing many
   * {@link org.mule.runtime.ast.api.ArtifactAst}s at once.
   *
   * @param format the id of the serialization format, defined as constants in {@link ArtifactAstSerializerFactory}
   * @return A serializer for the given format
   * @throws IllegalArgumentException if the specified format is null or if no matching serializer for the given format is found
   *                                  in the {@link ArtifactAstSerializerFactory}
   *
   * @deprecated since 1.1, use {@link #getSerializer(String, String)} instead.
   */
  @Deprecated
  public final ArtifactAstSerializer getSerializer(String format) throws IllegalArgumentException {
    return getSerializer(format, "1.0");
  }

  /**
   * This method returns a serializer for a given desired output format. The format is a string that identifies the serializer in
   * the {@link ArtifactAstSerializerFactory}. Valid values for format strings are defined as static constants in
   * {@link ArtifactAstSerializerFactory} class.
   * <p>
   * A serializer instance returned by this provider may not be thread-safe so it should no be used for serializing many
   * {@link org.mule.runtime.ast.api.ArtifactAst}s at once.
   *
   * @param format  the id of the serialization format, defined as constants in {@link ArtifactAstSerializerFactory}
   * @param version the desired serializer version
   * @return A serializer for the given format
   * @throws IllegalArgumentException if the specified format is null or if no matching serializer for the given format is found
   *                                  in the {@link ArtifactAstSerializerFactory}
   *
   * @since 1.1
   */
  public final ArtifactAstSerializer getSerializer(String format, String version) throws IllegalArgumentException {
    return artifactAstSerializerFactory.getSerializer(format, version);
  }

  /**
   * This method returns an {@link ArtifactAstDeserializer} able to handle all formats known by the
   * {@link ArtifactAstSerializerFactory}.
   * <p>
   * A deserializer instance returned by this provider may not be thread-safe so it should no be used for deserializing many
   * {@link org.mule.runtime.ast.api.ArtifactAst}s at once.
   *
   * @return an {@link ArtifactAstDeserializer}
   */
  public final ArtifactAstDeserializer getDeserializer() {
    return getDeserializer(true);
  }

  /**
   * This method returns an {@link ArtifactAstDeserializer} able to handle all formats known by the
   * {@link ArtifactAstSerializerFactory}.
   * <p>
   * A deserializer instance returned by this provider may not be thread-safe so it should no be used for deserializing many
   * {@link org.mule.runtime.ast.api.ArtifactAst}s at once.
   *
   * @return an {@link ArtifactAstDeserializer}
   */
  public final ArtifactAstDeserializer getDeserializer(boolean populateGenerationInformation) {
    return new DefaultArtifactAstDeserializer(populateGenerationInformation);
  }
}
