/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.xml;

import static java.util.Arrays.asList;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.is;
import static org.junit.rules.ExpectedException.none;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static org.mule.runtime.ast.api.xml.AstXmlParser.builder;

import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.XmlDslModel;
import org.mule.runtime.api.meta.model.construct.ConstructModel;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ImportedResource;
import org.mule.runtime.ast.api.xml.AstXmlParser;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import io.qameta.allure.Issue;
import org.junit.Rule;
import org.junit.rules.ExpectedException;
import org.junit.Before;
import org.junit.Test;

public class DefaultAstXmlParserTestCase {

  private static final XmlDslModel XML_DSL_MODEL = XmlDslModel.builder()
      .setPrefix("mule")
      .setNamespace("http://mockns")
      .build();

  private ClassLoader classLoader;
  private AstXmlParser parser;

  private final Map<String, String> properties = new HashMap<>();

  @Rule
  public ExpectedException expectedException = none();

  private final ExtensionModel mockedExtensionModel = mock(ExtensionModel.class);

  @Before
  public void before() {
    properties.clear();
    classLoader = DefaultAstXmlParserTestCase.class.getClassLoader();

    ConstructModel topLevelSimpleConstruct = mock(ConstructModel.class);
    when(topLevelSimpleConstruct.getName()).thenReturn("top-level-simple");

    ConstructModel topLevelSimpleOtherConstruct = mock(ConstructModel.class);
    when(topLevelSimpleOtherConstruct.getName()).thenReturn("top-level-simple-other");

    ConstructModel topLevelImportConstruct = mock(ConstructModel.class);
    when(topLevelImportConstruct.getName()).thenReturn("top-level-import");

    when(mockedExtensionModel.getName()).thenReturn("Mock Extension Model");
    when(mockedExtensionModel.getXmlDslModel()).thenReturn(XML_DSL_MODEL);
    when(mockedExtensionModel.getConstructModels())
        .thenReturn(asList(topLevelSimpleConstruct, topLevelImportConstruct, topLevelSimpleOtherConstruct));

    parser = builder()
        .withSchemaValidationsDisabled()
        .withExtensionModel(mockedExtensionModel)
        .withPropertyResolver(propertyKey -> properties.getOrDefault(propertyKey, propertyKey))
        .build();
  }

  @Test
  public void simple() {
    final ArtifactAst simpleAst = parser.parse(classLoader.getResource("simple.xml"));

    final List<String> components = simpleAst.recursiveStream().map(c -> c.getIdentifier().toString()).collect(toList());
    assertThat(components, contains("top-level-simple"));
  }

  @Test
  public void imports() {
    final ArtifactAst importAst = parser.parse(classLoader.getResource("import.xml"));

    final List<String> components = importAst.recursiveStream().map(c -> c.getIdentifier().toString()).collect(toList());
    assertThat(components, contains("top-level-import", "top-level-simple"));

    final List<String> imports =
        importAst.getImportedResources().stream().map(ImportedResource::getResourceLocation).collect(toList());
    assertThat(imports, contains("simple.xml"));
  }

  @Test
  public void importsWithProperty() {
    properties.put("${prop}", "simple.xml");
    final ArtifactAst importAst = parser.parse(classLoader.getResource("import-property.xml"));

    final List<String> components = importAst.recursiveStream().map(c -> c.getIdentifier().toString()).collect(toList());
    assertThat(components, contains("top-level-import", "top-level-simple"));

    final List<String> imports =
        importAst.getImportedResources().stream().map(ImportedResource::getResourceLocation).collect(toList());
    assertThat(imports, contains("simple.xml"));
  }

  @Test
  public void importsWithPropertyChangingValue() {
    properties.put("${prop}", "simple.xml");
    final ArtifactAst importAst = parser.parse(classLoader.getResource("import-property.xml"));

    final List<String> components = importAst.recursiveStream().map(c -> c.getIdentifier().toString()).collect(toList());
    assertThat(components, contains("top-level-import", "top-level-simple"));

    final List<String> imports =
        importAst.getImportedResources().stream().map(ImportedResource::getResourceLocation).collect(toList());
    assertThat(imports, contains("simple.xml"));

    importAst.updatePropertiesResolver(key -> {
      if (key.equals("${prop}")) {
        return "alternativeValue";
      } else {
        return null;
      }
    });
    final List<String> changedImports =
        importAst.getImportedResources().stream().map(ImportedResource::getResourceLocation).collect(toList());
    assertThat(changedImports, contains("alternativeValue"));
  }

  @Test
  public void twoConfigs() {
    final ArtifactAst importAst = parser.parse(classLoader.getResource("simple.xml"),
                                               classLoader.getResource("simple-other.xml"));

    final List<String> components = importAst.recursiveStream().map(c -> c.getIdentifier().toString()).collect(toList());
    assertThat(components, contains("top-level-simple", "top-level-simple-other"));

    final List<String> imports =
        importAst.getImportedResources().stream().map(ImportedResource::getResourceLocation).collect(toList());
    assertThat(imports, is(empty()));
  }

  @Test
  public void twoConfigsAndImport() {
    final ArtifactAst importAst = parser.parse(classLoader.getResource("import.xml"),
                                               classLoader.getResource("simple-other.xml"));

    final List<String> components = importAst.recursiveStream().map(c -> c.getIdentifier().toString()).collect(toList());
    assertThat(components, contains("top-level-import", "top-level-simple-other", "top-level-simple"));

    final List<String> imports =
        importAst.getImportedResources().stream().map(ImportedResource::getResourceLocation).collect(toList());
    assertThat(imports, contains("simple.xml"));
  }

  @Test
  public void twoConfigsAndImportSame() {
    final ArtifactAst importAst = parser.parse(classLoader.getResource("import.xml"),
                                               classLoader.getResource("simple.xml"));

    final List<String> components = importAst.recursiveStream().map(c -> c.getIdentifier().toString()).collect(toList());
    assertThat(components, contains("top-level-import", "top-level-simple"));

    final List<String> imports =
        importAst.getImportedResources().stream().map(ImportedResource::getResourceLocation).collect(toList());
    assertThat(imports, contains("simple.xml"));
  }

  @Test
  public void twoConfigsAndImportSameInverse() {
    final ArtifactAst importAst = parser.parse(classLoader.getResource("simple.xml"),
                                               classLoader.getResource("import.xml"));

    final List<String> components = importAst.recursiveStream().map(c -> c.getIdentifier().toString()).collect(toList());
    assertThat(components, contains("top-level-simple", "top-level-import"));

    final List<String> imports =
        importAst.getImportedResources().stream().map(ImportedResource::getResourceLocation).collect(toList());
    assertThat(imports, contains("simple.xml"));
  }

  @Test
  @Issue("MULE-19534")
  public void legacyFailStrategy() {
    expectedException.expect(MuleRuntimeException.class);
    expectedException
        .expectMessage(containsString("Invalid content was found"));
    parser = builder()
        .withPropertyResolver(propertyKey -> properties.getOrDefault(propertyKey, propertyKey))
        .withLegacyFailStrategy()
        .build();

    parser.parse(classLoader.getResource("mule-config-schema-missing.xml"));
  }

  @Test
  @Issue("MULE-19534")
  public void newFailStrategy() {
    expectedException.expect(MuleRuntimeException.class);
    expectedException
        .expectMessage(containsString("Can't resolve http://www.mulesoft.org/schema/mule/invalid-namespace/current/invalid-schema.xsd, A dependency or plugin might be missing"));
    parser = builder()
        .withPropertyResolver(propertyKey -> properties.getOrDefault(propertyKey, propertyKey))
        .build();

    parser.parse(classLoader.getResource("mule-config-schema-missing.xml"));
  }
}
