/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.xml;

import static java.lang.String.format;
import static java.lang.Thread.currentThread;
import static java.util.Arrays.asList;
import static java.util.Collections.singletonList;
import static java.util.Collections.unmodifiableSet;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static java.util.stream.Stream.concat;
import static java.util.stream.StreamSupport.stream;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.ast.api.util.MuleAstUtils.emptyArtifact;
import static org.mule.runtime.ast.internal.xml.XmlNamespaceInfoProviderSupplier.createFromExtensionModels;
import static org.mule.runtime.dsl.api.xml.parser.XmlConfigurationDocumentLoader.noValidationDocumentLoader;
import static org.mule.runtime.dsl.api.xml.parser.XmlConfigurationDocumentLoader.schemaValidatingDocumentLoader;
import static org.mule.runtime.dsl.internal.xerces.xni.parser.DefaultXmlGrammarPoolManager.getGrammarPool;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.apache.xerces.xni.grammars.XMLGrammarPool;
import org.mule.runtime.api.dsl.DslResolvingContext;
import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.api.util.Pair;
import org.mule.runtime.api.util.xmlsecurity.XMLSecureFactories;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ImportedResource;
import org.mule.runtime.ast.api.NamespaceDefinition;
import org.mule.runtime.ast.api.builder.ArtifactAstBuilder;
import org.mule.runtime.ast.api.util.BaseArtifactAst;
import org.mule.runtime.ast.api.xml.AstXmlParser;
import org.mule.runtime.ast.internal.error.CompositeErrorTypeRepository;
import org.mule.runtime.ast.internal.xml.reader.ComponentAstReader;
import org.mule.runtime.ast.internal.xml.resolver.CachingExtensionSchemaGenerator;
import org.mule.runtime.ast.internal.xml.resolver.ModuleDelegatingEntityResolver;
import org.mule.runtime.ast.internal.xml.resolver.ResolveEntityFailStrategy;
import org.mule.runtime.dsl.api.ConfigResource;
import org.mule.runtime.dsl.api.xml.XmlNamespaceInfoProvider;
import org.mule.runtime.dsl.api.xml.parser.ParsingPropertyResolver;
import org.mule.runtime.dsl.api.xml.parser.XmlApplicationParser;
import org.mule.runtime.dsl.api.xml.parser.XmlConfigurationDocumentLoader;
import org.mule.runtime.extension.api.dsl.syntax.resources.spi.ExtensionSchemaGenerator;

import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.ServiceLoader;
import java.util.Set;
import java.util.function.UnaryOperator;
import java.util.stream.Stream;

import javax.xml.parsers.SAXParserFactory;

import org.slf4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.EntityResolver;
import org.xml.sax.helpers.DefaultHandler;

public class DefaultAstXmlParser implements AstXmlParser {

  private static final Logger LOGGER = getLogger(DefaultAstXmlParser.class);

  private static final SAXParserFactory SAX_PARSER_FACTORY = XMLSecureFactories.createDefault().getSAXParserFactory();

  private static final ComponentAstReader CONFIG_READER = new ComponentAstReader();

  private final XmlConfigurationDocumentLoader xmlConfigurationDocumentLoader;
  private final XMLGrammarPool grammarPool;
  private final EntityResolver entityResolver;
  private final XmlApplicationParser xmlApplicationParser;
  private final Set<ExtensionModel> extensionModels;
  private final ArtifactAst parentArtifact;
  private final ParsingPropertyResolver propertyResolver;

  public DefaultAstXmlParser(Set<ExtensionModel> extensionModels, ArtifactAst parentArtifact,
                             Optional<ExtensionSchemaGenerator> schemaGenerator,
                             ParsingPropertyResolver propertyResolver,
                             ResolveEntityFailStrategy resolveEntityFailStrategy) {
    this.extensionModels = unmodifiableSet(new HashSet<>(extensionModels));
    this.parentArtifact = parentArtifact;

    final DslResolvingContext dslResolvingContext = DslResolvingContext.getDefault(this.extensionModels);
    this.xmlConfigurationDocumentLoader = schemaGenerator.isPresent()
        ? schemaValidatingDocumentLoader()
        : noValidationDocumentLoader();
    this.grammarPool = schemaGenerator.isPresent()
        ? getGrammarPool().orElse(null)
        : null;

    this.entityResolver = schemaGenerator
        .map(sg -> (EntityResolver) new ModuleDelegatingEntityResolver(extensionModels, new CachingExtensionSchemaGenerator(sg),
                                                                       dslResolvingContext, resolveEntityFailStrategy))
        .orElseGet(() -> new DefaultHandler());
    this.propertyResolver = propertyResolver;

    // How will this resolve the stuff from the runtime when running in 'light' mode?
    List<XmlNamespaceInfoProvider> namespaceInfoProviders = createFromExtensionModels(extensionModels,
                                                                                      of(cl -> stream(ServiceLoader
                                                                                          .load(XmlNamespaceInfoProvider.class,
                                                                                                cl)
                                                                                          .spliterator(), false)
                                                                                              .collect(toList())));
    xmlApplicationParser = new XmlApplicationParser(namespaceInfoProviders);
  }

  @Override
  public ArtifactAst parse(List<Pair<String, InputStream>> appXmlConfigInputStreams) {
    if (appXmlConfigInputStreams.isEmpty()) {
      throw new IllegalArgumentException("At least one 'appXmlConfigInputStream' must be provided");
    }

    LOGGER.debug("About to parse AST from inputStreams {}", appXmlConfigInputStreams);

    final List<ConfigResource> baseAstResources = appXmlConfigInputStreams.stream()
        .map(is -> inputStreamToConfigResource(is.getFirst(), is.getSecond()))
        .collect(toList());

    return resolveImports(aggregateAsts(baseAstResources.stream()
        .map(this::doParse)
        .collect(toList())), baseAstResources);
  }

  @Override
  public ArtifactAst parse(String resourceName, InputStream appXmlConfigInputStream) {
    LOGGER.debug("About to parse AST from inputStream {}, {}", resourceName, appXmlConfigInputStream);
    final ConfigResource baseAstResource = inputStreamToConfigResource(resourceName, appXmlConfigInputStream);
    return resolveImports(doParse(baseAstResource), singletonList(baseAstResource));
  }

  @Override
  public ArtifactAst parse(URI... appXmlConfigUris) {
    if (appXmlConfigUris.length == 0) {
      throw new IllegalArgumentException("At least one 'appXmlConfigUri' must be provided");
    }

    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("About to parse AST from uris {}", asList(appXmlConfigUris));
    }

    final List<ConfigResource> baseAstResources = Stream.of(appXmlConfigUris)
        .map(this::uriToConfigResource)
        .collect(toList());

    return resolveImports(aggregateAsts(baseAstResources.stream()
        .map(this::doParse)
        .collect(toList())), baseAstResources);
  }

  @Override
  public ArtifactAst parse(URL... appXmlConfigUrls) {
    if (appXmlConfigUrls.length == 0) {
      throw new IllegalArgumentException("At least one 'appXmlConfigUrl' must be provided");
    }

    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("About to parse AST from urls {}", asList(appXmlConfigUrls));
    }

    final List<ConfigResource> baseAstResources = Stream.of(appXmlConfigUrls)
        .map(this::urlToConfigResource)
        .collect(toList());

    return resolveImports(aggregateAsts(baseAstResources.stream()
        .map(this::doParse)
        .collect(toList())), baseAstResources);
  }

  private ArtifactAst aggregateAsts(final List<ArtifactAst> parsedAstsList) {
    ArtifactAst aggregatedAst;
    if (parsedAstsList.size() == 1) {
      aggregatedAst = parsedAstsList.get(0);
    } else {
      aggregatedAst = new BaseArtifactAst() {

        final LazyValue<Set<ExtensionModel>> dependencies = new LazyValue<>(() -> parsedAstsList.stream()
            .flatMap(ast -> ast.dependencies().stream())
            .collect(toSet()));

        final LazyValue<ErrorTypeRepository> errorTypeRepo = new LazyValue<>(() -> new CompositeErrorTypeRepository(parsedAstsList
            .stream()
            .map(ArtifactAst::getErrorTypeRepository)
            .collect(toList())));

        final LazyValue<List<ComponentAst>> topLevelComponents = new LazyValue<>(() -> parsedAstsList
            .stream()
            .flatMap(ArtifactAst::topLevelComponentsStream)
            .collect(toList()));

        final LazyValue<List<ImportedResource>> importedResources = new LazyValue<>(() -> parsedAstsList.stream()
            .flatMap(ast -> ast.getImportedResources().stream())
            .collect(toList()));

        @Override
        public Set<ExtensionModel> dependencies() {
          return dependencies.get();
        }

        @Override
        public Optional<ArtifactAst> getParent() {
          return ofNullable(parentArtifact);
        }

        @Override
        public List<ComponentAst> topLevelComponents() {
          return topLevelComponents.get();
        }

        @Override
        public void updatePropertiesResolver(UnaryOperator<String> newPropertiesResolver) {
          parsedAstsList.forEach(ast -> ast.updatePropertiesResolver(newPropertiesResolver));
        }

        @Override
        public ErrorTypeRepository getErrorTypeRepository() {
          return errorTypeRepo.get();
        }

        @Override
        public Collection<ImportedResource> getImportedResources() {
          return importedResources.get();
        }
      };
    }
    return aggregatedAst;
  }

  private ArtifactAst resolveImports(ArtifactAst base, List<ConfigResource> baseAstResources) {
    final Map<ConfigResource, ArtifactAst> importedConfigs = new LinkedHashMap<>();
    baseAstResources.forEach(r -> importedConfigs.put(r, emptyArtifact()));
    resolveImports(base, importedConfigs);

    if (importedConfigs.isEmpty()) {
      return base;
    } else {
      return new BaseArtifactAst() {

        final LazyValue<Set<ExtensionModel>> dependencies =
            new LazyValue<>(() -> concat(base.dependencies().stream(),
                                         importedConfigs.values().stream()
                                             .flatMap(ast -> ast.dependencies().stream()))
                                                 .collect(toSet()));

        final LazyValue<ErrorTypeRepository> errorTypeRepo = new LazyValue<>(() -> {
          final List<ErrorTypeRepository> allErrorTypeRepos =
              importedConfigs.values().stream().map(ArtifactAst::getErrorTypeRepository).collect(toList());
          allErrorTypeRepos.add(0, base.getErrorTypeRepository());

          return new CompositeErrorTypeRepository(allErrorTypeRepos);
        });

        @Override
        public Set<ExtensionModel> dependencies() {
          return dependencies.get();
        }

        @Override
        public Optional<ArtifactAst> getParent() {
          return ofNullable(parentArtifact);
        }

        @Override
        public Stream<ComponentAst> topLevelComponentsStream() {
          return concat(base.topLevelComponentsStream(),
                        importedConfigs.values().stream().flatMap(ArtifactAst::topLevelComponentsStream));
        }

        @Override
        public List<ComponentAst> topLevelComponents() {
          return topLevelComponentsStream().collect(toList());
        }

        @Override
        public void updatePropertiesResolver(UnaryOperator<String> newPropertiesResolver) {
          base.updatePropertiesResolver(newPropertiesResolver);
          importedConfigs.values().forEach(ast -> ast.updatePropertiesResolver(newPropertiesResolver));
        }

        @Override
        public ErrorTypeRepository getErrorTypeRepository() {
          return errorTypeRepo.get();
        }

        @Override
        public NamespaceDefinition namespaceDefinition() {
          return base.namespaceDefinition();
        }

        @Override
        public Collection<ImportedResource> getImportedResources() {
          return concat(base.getImportedResources().stream(),
                        importedConfigs.values().stream().flatMap(ast -> ast.getImportedResources().stream())).collect(toList());
        }
      };
    }
  }

  private ArtifactAst doParse(ConfigResource resource) {
    try (InputStream inputStream = resource.getInputStream()) {
      Document document = xmlConfigurationDocumentLoader.loadDocument(() -> SAX_PARSER_FACTORY,
                                                                      entityResolver,
                                                                      resource.getResourceName(),
                                                                      inputStream,
                                                                      grammarPool);

      final ArtifactAstBuilder astBuilder = ArtifactAstBuilder.builder(extensionModels, ofNullable(parentArtifact),
                                                                       propertyResolver::resolveProperty);

      final Element rootElement = document.getDocumentElement();

      final String identifier = xmlApplicationParser.parseIdentifier(rootElement);
      if (identifier.equals(MULE_ROOT_IDENTIFIER.getName())
          || identifier.equals(DOMAIN_ROOT_IDENTIFIER.getName())) {
        NodeList children = rootElement.getChildNodes();
        for (int i = 0; i < children.getLength(); i++) {
          Node child = children.item(i);
          if (child instanceof Element) {
            CONFIG_READER.processAttributes(astBuilder, (Element) child);
            if (IMPORT_IDENTIFIER.getName().equals(child.getNodeName())) {
              CONFIG_READER.extractImport(astBuilder, (Element) child, resource.getResourceName(),
                                          astBuilder::createMetadataBuilder);
            } else {
              try {
                CONFIG_READER.extractComponentDefinitionModel(xmlApplicationParser, (Element) child, resource.getResourceName(),
                                                              astBuilder.addTopLevelComponent(),
                                                              astBuilder::createMetadataBuilder);
              } catch (Exception e) {
                throw new MuleRuntimeException(createStaticMessage("Exception extracting definition model for component {}.",
                                                                   ((Element) child).getTagName()),
                                               e);
              }
            }
          }
        }
      } else {
        // XML SDK 1 needs to access the parameter on the root component
        // TODO MULE-19548 : remove Vendor, Name, Doc:* and Category from the namespaces' map!
        CONFIG_READER.processAttributes(astBuilder, rootElement);

        try {
          CONFIG_READER.extractComponentDefinitionModel(xmlApplicationParser, rootElement, resource.getResourceName(),
                                                        astBuilder.addTopLevelComponent(), astBuilder::createMetadataBuilder);
        } catch (Exception e) {
          throw new MuleRuntimeException(createStaticMessage("Exception extracting definition model for component {}.",
                                                             rootElement.getTagName()),
                                         e);
        }
      }

      return astBuilder.build();
    } catch (IOException e) {
      throw new MuleRuntimeException(e);
    }

  }

  private void resolveImports(ArtifactAst parsedConfig, Map<ConfigResource, ArtifactAst> alreadyResolvedConfigFiles) {
    parsedConfig.getImportedResources().stream()
        .map(ImportedResource::getResourceLocation)
        .map(propertyResolver::resolveProperty)
        .map(importFile -> ofNullable(currentThread().getContextClassLoader().getResourceAsStream(importFile))
            .map(stream -> new ConfigResource(importFile, stream))
            .orElseThrow(() -> new MuleRuntimeException(createStaticMessage(format("Could not find imported resource '%s'",
                                                                                   importFile)))))
        .filter(importedConfigResource -> !alreadyResolvedConfigFiles.containsKey(importedConfigResource))
        .forEach(importedConfigResource -> {
          LOGGER.debug("Resolving import {}", importedConfigResource.getResourceName());

          final ArtifactAst parsedImportedConfig = doParse(importedConfigResource);
          alreadyResolvedConfigFiles.put(importedConfigResource, parsedImportedConfig);
          resolveImports(parsedImportedConfig, alreadyResolvedConfigFiles);
        });
  }

  private ConfigResource uriToConfigResource(URI appXmlConfig) {
    final URL appXmlConfigUrl;
    try {
      appXmlConfigUrl = appXmlConfig.toURL();
    } catch (MalformedURLException e) {
      throw new MuleRuntimeException(e);
    }

    return urlToConfigResource(appXmlConfigUrl);
  }

  private ConfigResource urlToConfigResource(URL appXmlConfigUrl) {
    final ConfigResource configResource = new ConfigResource(appXmlConfigUrl);

    // Try to just initialize the inputStream here so if there is any issue there it is caught in this early phase.
    try {
      configResource.getInputStream();
    } catch (IOException e) {
      throw new MuleRuntimeException(e);
    }

    return configResource;
  }

  private ConfigResource inputStreamToConfigResource(String resourceName, InputStream appXmlConfigInputStream) {
    return new ConfigResource(resourceName, appXmlConfigInputStream);
  }
}
