/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.xml.reader;

import static java.lang.Boolean.TRUE;
import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.isEmpty;
import static org.mule.runtime.api.component.Component.NS_MULE_DOCUMENTATION;
import static org.mule.runtime.api.component.Component.NS_MULE_PARSER_METADATA;
import static org.mule.runtime.api.component.Component.Annotations.NAME_ANNOTATION_KEY;
import static org.mule.runtime.api.component.ComponentIdentifier.builder;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.ast.api.xml.AstXmlParser.ANNOTATIONS_IDENTIFIER;
import static org.mule.runtime.dsl.api.xml.parser.XmlApplicationParser.DECLARED_PREFIX;
import static org.mule.runtime.dsl.api.xml.parser.XmlApplicationParser.IS_CDATA;
import static org.mule.runtime.dsl.api.xml.parser.XmlApplicationParser.isTextContent;
import static org.mule.runtime.dsl.internal.xml.parser.XmlMetadataAnnotations.METADATA_ANNOTATIONS_KEY;
import static org.mule.runtime.internal.dsl.DslConstants.CORE_NAMESPACE;
import static org.mule.runtime.internal.dsl.DslConstants.CORE_PREFIX;
import static org.w3c.dom.Node.CDATA_SECTION_NODE;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.ast.api.ComponentMetadataAst;
import org.mule.runtime.ast.api.builder.ArtifactAstBuilder;
import org.mule.runtime.ast.api.builder.ComponentAstBuilder;
import org.mule.runtime.ast.api.builder.NamespaceDefinitionBuilder;
import org.mule.runtime.ast.internal.DefaultComponentMetadataAst;
import org.mule.runtime.ast.internal.DefaultComponentMetadataAst.Builder;
import org.mule.runtime.ast.internal.builder.ImportedResourceBuilder;
import org.mule.runtime.dsl.api.xml.parser.XmlApplicationParser;
import org.mule.runtime.dsl.internal.xml.parser.XmlMetadataAnnotations;

import java.util.HashMap;
import java.util.Map;
import java.util.function.Consumer;
import java.util.function.Predicate;

import javax.xml.namespace.QName;

import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * This code was moved from {@code mule} project.
 */
public class ComponentAstReader {

  private static final QName DECLARED_PREFIX_QNAME = QName.valueOf(DECLARED_PREFIX);
  private static final String SCHEMA_LOCATION = "xsi:schemaLocation";
  private static final String MODULE_PREFIX_ATTRIBUTE = "prefix";
  private static final String MODULE_NAMESPACE_ATTRIBUTE = "namespace";

  public void extractComponentDefinitionModel(XmlApplicationParser parser, Element element, String configFileName,
                                              ComponentAstBuilder componentAstBuilder) {
    componentAstBuilder.withIdentifier(buildIdentifier(parser, element));

    final Builder metadata = extractMetadata(element, configFileName);

    processAttributes(componentAstBuilder, element);
    processAnnotations(componentAstBuilder, element);
    processChildNodesRecursively(componentAstBuilder, metadata, parser, element, configFileName);

    componentAstBuilder.withMetadata(metadata.build());
  }

  private ComponentIdentifier buildIdentifier(XmlApplicationParser parser, Element element) {
    String namespace = parser.parseNamespace(element);
    String namespaceUri = parser.parseNamespaceUri(element);

    final ComponentIdentifier identifier = builder()
        .namespace(namespace == null ? CORE_PREFIX : namespace)
        .namespaceUri(namespaceUri == null ? CORE_NAMESPACE : namespaceUri)
        .name(parser.parseIdentifier(element))
        .build();
    return identifier;
  }

  private Builder extractMetadata(Element element, String configFileName) {
    Builder metadataBuilder = DefaultComponentMetadataAst.builder()
        .setFileName(configFileName);

    XmlMetadataAnnotations userData = (XmlMetadataAnnotations) element.getUserData(METADATA_ANNOTATIONS_KEY);
    if (userData != null) {
      metadataBuilder = metadataBuilder.setStartLine(userData.getLineNumber())
          .setEndLine(userData.getLineNumber())
          .setStartColumn(userData.getColumnNumber())
          .setEndColumn(userData.getColumnNumber())
          .setSourceCode(userData.getElementString());
    }

    Node nameAttribute = element.getAttributes()
        .getNamedItemNS(NAME_ANNOTATION_KEY.getNamespaceURI(), NAME_ANNOTATION_KEY.getLocalPart());
    if (nameAttribute != null) {
      addCustomAttribute(metadataBuilder, NAME_ANNOTATION_KEY, nameAttribute.getNodeValue());
    }
    if (element.getPrefix() != null) {
      addCustomAttribute(metadataBuilder, DECLARED_PREFIX_QNAME, element.getPrefix());
    }
    for (int i = 0; i < element.getAttributes().getLength(); i++) {
      Node attributeNode = element.getAttributes().item(i);
      if (attributeNode.getNamespaceURI() != null) {
        addCustomAttribute(metadataBuilder,
                           new QName(attributeNode.getNamespaceURI(), attributeNode.getLocalName()),
                           attributeNode.getNodeValue());
      }
    }

    return metadataBuilder;
  }

  private void processAttributes(ComponentAstBuilder componentAstBuilder, Element element) {
    processAttributes(element,
                      n -> isEmpty(n.getNamespaceURI()),
                      node -> componentAstBuilder.withRawParameter(node.getNodeName(), node.getNodeValue()));
  }

  private void processAnnotations(ComponentAstBuilder componentAstBuilder, Element element) {
    processAttributes(element,
                      n -> !isEmpty(n.getNamespaceURI()),
                      node -> componentAstBuilder
                          .withAnnotation(new QName(node.getNamespaceURI(), node.getLocalName()).toString(),
                                          node.getNodeValue()));
  }

  private Map<String, String> calculateSchemaLocations(String schLoc) {
    Map<String, String> schemaLocations = new HashMap<>();
    String[] pairs = schLoc.trim().split("\\s+");
    for (int i = 0; i < pairs.length; i = i + 2) {
      schemaLocations.put(pairs[i], pairs[i + 1]);
    }
    return schemaLocations;
  }


  public void processAttributes(ArtifactAstBuilder astBuilder, Element element) {
    final NamespaceDefinitionBuilder namespaceDefinitionBuilder = NamespaceDefinitionBuilder.builder();
    processAttributes(element, n -> true, node -> {
      String name = node.getNodeName();
      String value = node.getNodeValue();

      if (name.equals(SCHEMA_LOCATION)) {
        calculateSchemaLocations(value).forEach((schLoc, loc) -> namespaceDefinitionBuilder.withSchemaLocation(schLoc, loc));
      } else if (name.equals(MODULE_NAMESPACE_ATTRIBUTE)) {
        namespaceDefinitionBuilder.withNamespace(value);
      } else if (name.equals(MODULE_PREFIX_ATTRIBUTE)) {
        namespaceDefinitionBuilder.withPrefix(value);
      } else {
        namespaceDefinitionBuilder.withUnresolvedNamespace(name, value);
      }
    });
    astBuilder.withNamespaceDefinition(namespaceDefinitionBuilder.build());
  }

  private void processAttributes(Element element, Predicate<Node> filter, Consumer<Node> consumer) {
    NamedNodeMap attributes = element.getAttributes();
    if (element.hasAttributes()) {
      for (int i = 0; i < attributes.getLength(); i++) {
        Node attribute = attributes.item(i);
        if (filter.test(attribute)) {
          consumer.accept(attribute);
        }
      }
    }
  }

  private void processChildNodesRecursively(ComponentAstBuilder componentAstBuilder, Builder metadata,
                                            XmlApplicationParser parser, Element element,
                                            String configFileName) {
    if (element.hasChildNodes()) {
      NodeList children = element.getChildNodes();
      for (int i = 0; i < children.getLength(); i++) {
        Node child = children.item(i);
        if (isTextContent(child)) {
          componentAstBuilder.withBodyParameter(child.getNodeValue());
          if (child.getNodeType() == CDATA_SECTION_NODE) {
            final Builder metadataBuilder = metadata;
            metadataBuilder.putParserAttribute(IS_CDATA, TRUE);
            break;
          }
        } else {
          if (child instanceof Element) {
            if (child.getNamespaceURI().equals(CORE_NAMESPACE)
                && child.getLocalName().endsWith(ANNOTATIONS_IDENTIFIER.getName())) {
              processNestedAnnotations((Element) child, metadata);
            } else {
              extractComponentDefinitionModel(parser, (Element) child, configFileName, componentAstBuilder.addChildComponent());
            }
          }
        }
      }
    }
  }

  private void processNestedAnnotations(Element element, Builder metadata) {
    if (element.hasChildNodes()) {
      NodeList children = element.getChildNodes();
      for (int i = 0; i < children.getLength(); i++) {
        Node child = children.item(i);

        if (child instanceof Element) {
          addCustomAttribute(metadata, new QName(child.getNamespaceURI(), child.getLocalName(), child.getPrefix()),
                             child.getTextContent().trim());
        }
      }
    }
  }

  private void addCustomAttribute(final Builder metadataBuilder, QName qname, Object value) {
    if (isEmpty(qname.getNamespaceURI()) || NS_MULE_PARSER_METADATA.equals(qname.getNamespaceURI())) {
      metadataBuilder.putParserAttribute(qname.getLocalPart(), value);
    } else {
      if (NS_MULE_DOCUMENTATION.equals(qname.getNamespaceURI())) {
        // This is added for compatibility,
        // since in previous versions the doc attributes were looked up without the namespace.
        metadataBuilder.putDocAttribute(qname.getLocalPart(), value.toString());
      }
    }
  }

  public void extractImport(ArtifactAstBuilder astBuilder, Element importNode, String configFileName) {
    final ComponentMetadataAst metadata = extractMetadata(importNode, configFileName).build();

    if (importNode.hasAttribute("file")) {
      astBuilder.withImportedResource(ImportedResourceBuilder.builder()
          .withResourceLocation(importNode.getAttribute("file"))
          .withMetadata(metadata)
          .build());
    } else {
      throw new MuleRuntimeException(createStaticMessage(format("<import> does not have a file attribute defined. At file '%s', at line %s",
                                                                metadata.getFileName(), metadata.getStartLine())));
    }
  }
}
