/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.xml.metadata;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.core.Is.is;
import static org.mule.runtime.ast.AllureConstants.ArtifactAst.ARTIFACT_AST;
import static org.mule.runtime.ast.AllureConstants.ArtifactAst.Metadata.SOURCE_LOCATION;
import static org.mule.runtime.ast.api.xml.AstXmlParser.builder;

import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentMetadataAst;
import org.mule.runtime.ast.api.xml.AstXmlParser;
import org.mule.runtime.ast.internal.xml.DefaultAstXmlParserTestCase;

import org.junit.Before;
import org.junit.Test;

import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;

@Feature(ARTIFACT_AST)
@Story(SOURCE_LOCATION)
public class LocationMetadataTestCase {

  private ClassLoader classLoader;
  private AstXmlParser parser;

  @Before
  public void before() {
    classLoader = DefaultAstXmlParserTestCase.class.getClassLoader();

    parser = builder()
        .withSchemaValidationsDisabled()
        .withPropertyResolver(propertyKey -> propertyKey)
        .build();
  }

  @Test
  @Issue("MULE-19634")
  public void propertyLocationMetadata() {
    final ArtifactAst simpleAst = parser.parse(classLoader.getResource("metadata/flow-with-source-and-operations.xml"));

    ComponentAst globalProperty = simpleAst.topLevelComponents().get(0);

    ComponentMetadataAst metadata = globalProperty.getMetadata();
    assertThat(metadata.getStartLine().getAsInt(), is(11));
    assertThat(metadata.getStartColumn().getAsInt(), is(5));
  }

  @Test
  public void configLocationMetadata() {
    final ArtifactAst simpleAst = parser.parse(classLoader.getResource("metadata/flow-with-source-and-operations.xml"));

    ComponentAst httpConfig = simpleAst.topLevelComponents().get(1);

    ComponentMetadataAst metadata = httpConfig.getMetadata();
    assertThat(metadata.getStartLine().getAsInt(), is(12));
    assertThat(metadata.getStartColumn().getAsInt(), is(5));
  }

  @Test
  public void flowLocationMetadata() {
    final ArtifactAst simpleAst = parser.parse(classLoader.getResource("metadata/flow-with-source-and-operations.xml"));

    ComponentAst flow = simpleAst.topLevelComponents().get(2);

    ComponentMetadataAst metadata = flow.getMetadata();
    assertThat(metadata.getStartLine().getAsInt(), is(16));
    assertThat(metadata.getStartColumn().getAsInt(), is(5));
  }

  @Test
  @Issue("MULE-19634")
  public void sourceLocationMetadata() {
    final ArtifactAst simpleAst = parser.parse(classLoader.getResource("metadata/flow-with-source-and-operations.xml"));

    ComponentAst flow = simpleAst.topLevelComponents().get(2);
    ComponentAst source = flow.directChildren().get(0);

    ComponentMetadataAst metadata = source.getMetadata();
    assertThat(metadata.getStartLine().getAsInt(), is(17));
    assertThat(metadata.getStartColumn().getAsInt(), is(9));
  }

  @Test
  public void operationTransformLocationMetadata() {
    final ArtifactAst simpleAst = parser.parse(classLoader.getResource("metadata/flow-with-source-and-operations.xml"));

    ComponentAst flow = simpleAst.topLevelComponents().get(2);
    ComponentAst transform = flow.directChildren().get(1);

    ComponentMetadataAst metadata = transform.getMetadata();
    assertThat(metadata.getStartLine().getAsInt(), is(18));
    assertThat(metadata.getStartColumn().getAsInt(), is(9));
  }

  @Test
  public void operationLoggerLocationMetadata() {
    final ArtifactAst simpleAst = parser.parse(classLoader.getResource("metadata/flow-with-source-and-operations.xml"));

    ComponentAst flow = simpleAst.topLevelComponents().get(2);
    ComponentAst logger = flow.directChildren().get(2);

    ComponentMetadataAst metadata = logger.getMetadata();
    assertThat(metadata.getStartLine().getAsInt(), is(28));
    assertThat(metadata.getStartColumn().getAsInt(), is(9));
  }

  @Test
  public void choiceRouterLocationMetadata() {
    final ArtifactAst simpleAst = parser.parse(classLoader.getResource("metadata/flow-with-source-and-operations.xml"));

    ComponentAst flow = simpleAst.topLevelComponents().get(2);
    ComponentAst choice = flow.directChildren().get(4);

    ComponentMetadataAst metadata = choice.getMetadata();
    assertThat(metadata.getStartLine().getAsInt(), is(30));
    assertThat(metadata.getStartColumn().getAsInt(), is(9));
  }

  @Test
  public void operationRouteLocationMetadata() {
    final ArtifactAst simpleAst = parser.parse(classLoader.getResource("metadata/flow-with-source-and-operations.xml"));

    ComponentAst flow = simpleAst.topLevelComponents().get(2);
    ComponentAst choice = flow.directChildren().get(4);
    ComponentAst firstRoute = choice.directChildren().get(0);

    ComponentMetadataAst metadata = firstRoute.getMetadata();
    assertThat(metadata.getStartLine().getAsInt(), is(31));
    assertThat(metadata.getStartColumn().getAsInt(), is(13));
  }


  @Test
  public void operationInChoiceRouteLocationMetadata() {
    final ArtifactAst simpleAst = parser.parse(classLoader.getResource("metadata/flow-with-source-and-operations.xml"));

    ComponentAst flow = simpleAst.topLevelComponents().get(2);
    ComponentAst choice = flow.directChildren().get(4);
    ComponentAst firstRoute = choice.directChildren().get(0);
    ComponentAst componentInChoiceRoute = firstRoute.directChildren().get(0);

    ComponentMetadataAst metadata = componentInChoiceRoute.getMetadata();
    assertThat(metadata.getStartLine().getAsInt(), is(32));
    assertThat(metadata.getStartColumn().getAsInt(), is(17));
  }

}
