/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.xml.reader;

import static java.util.Collections.emptyList;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.aMapWithSize;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.nullValue;
import static org.hamcrest.collection.IsMapContaining.hasEntry;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static org.mockito.MockitoAnnotations.initMocks;
import static org.mule.runtime.ast.internal.builder.BaseComponentAstBuilder.BODY_RAW_PARAM_NAME;
import static org.mule.runtime.dsl.api.xml.parser.XmlApplicationParser.IS_CDATA;

import org.mockito.ArgumentCaptor;
import org.mockito.Captor;
import org.mockito.Mock;
import org.mule.runtime.ast.api.NamespaceDefinition;
import org.mule.runtime.ast.api.builder.ArtifactAstBuilder;
import org.mule.runtime.ast.internal.builder.DefaultComponentAstBuilder;
import org.mule.runtime.ast.internal.builder.PropertiesResolver;
import org.mule.runtime.ast.internal.model.ExtensionModelHelper;
import org.mule.runtime.dsl.api.xml.parser.XmlApplicationParser;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.junit.Before;
import org.junit.Test;
import org.w3c.dom.Document;
import org.xml.sax.SAXException;

public class ComponentAstReaderTestCase {

  private final ComponentAstReader reader = new ComponentAstReader();

  private DocumentBuilder builder;
  private XmlApplicationParser parser;

  @Mock
  private ArtifactAstBuilder mockBuilder;

  @Captor
  private ArgumentCaptor<NamespaceDefinition> namespaceCaptor;

  @Before
  public void before() throws ParserConfigurationException {
    DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
    builder = factory.newDocumentBuilder();

    parser = mock(XmlApplicationParser.class);
    initMocks(this);
  }

  @Test
  public void emptyBody() throws ParserConfigurationException, SAXException, IOException {
    Document doc = createDoc("<mule></mule>");

    when(parser.parseIdentifier(any())).thenReturn("comp");

    final DefaultComponentAstBuilder compBuilder = new DefaultComponentAstBuilder(new PropertiesResolver(),
                                                                                  mock(ExtensionModelHelper.class), emptyList(),
                                                                                  0);

    reader.extractComponentDefinitionModel(parser, doc.getDocumentElement(), "app.xml", compBuilder);

    assertThat(compBuilder.getRawParameters().get(BODY_RAW_PARAM_NAME), is(nullValue()));
    assertThat(compBuilder.getMetadata().getParserAttributes().getOrDefault(IS_CDATA, false), is(false));
  }

  @Test
  public void emptyBodyFormatted() throws ParserConfigurationException, SAXException, IOException {
    Document doc = createDoc("<mule>\n</mule>");

    when(parser.parseIdentifier(any())).thenReturn("comp");

    final DefaultComponentAstBuilder compBuilder = new DefaultComponentAstBuilder(new PropertiesResolver(),
                                                                                  mock(ExtensionModelHelper.class), emptyList(),
                                                                                  0);

    reader.extractComponentDefinitionModel(parser, doc.getDocumentElement(), "app.xml", compBuilder);

    assertThat(compBuilder.getRawParameters().get(BODY_RAW_PARAM_NAME), is("\n"));
    assertThat(compBuilder.getMetadata().getParserAttributes().getOrDefault(IS_CDATA, false), is(false));
  }

  @Test
  public void emptyCdataBody() throws ParserConfigurationException, SAXException, IOException {
    Document doc = createDoc("<mule><![CDATA[]]></mule>");

    when(parser.parseIdentifier(any())).thenReturn("comp");

    final DefaultComponentAstBuilder compBuilder = new DefaultComponentAstBuilder(new PropertiesResolver(),
                                                                                  mock(ExtensionModelHelper.class), emptyList(),
                                                                                  0);

    reader.extractComponentDefinitionModel(parser, doc.getDocumentElement(), "app.xml", compBuilder);

    assertThat(compBuilder.getRawParameters().get(BODY_RAW_PARAM_NAME), is(""));
    assertThat(compBuilder.getMetadata().getParserAttributes().getOrDefault(IS_CDATA, false), is(true));
  }

  @Test
  public void emptyCdataBodyFormatted() throws ParserConfigurationException, SAXException, IOException {
    Document doc = createDoc("<mule>\n  <![CDATA[]]>\n</mule>");

    when(parser.parseIdentifier(any())).thenReturn("comp");

    final DefaultComponentAstBuilder compBuilder = new DefaultComponentAstBuilder(new PropertiesResolver(),
                                                                                  mock(ExtensionModelHelper.class), emptyList(),
                                                                                  0);

    reader.extractComponentDefinitionModel(parser, doc.getDocumentElement(), "app.xml", compBuilder);

    assertThat(compBuilder.getRawParameters().get(BODY_RAW_PARAM_NAME), is(""));
    assertThat(compBuilder.getMetadata().getParserAttributes().getOrDefault(IS_CDATA, false), is(true));
  }

  private Document createDoc(final String content) throws UnsupportedEncodingException, SAXException, IOException {
    StringBuilder xmlStringBuilder = new StringBuilder();
    xmlStringBuilder.append("<?xml version=\"1.0\"?> " + content);
    ByteArrayInputStream input = new ByteArrayInputStream(xmlStringBuilder.toString().getBytes("UTF-8"));

    return builder.parse(input);
  }

  @Test
  public void creatingNamespaceDefinition() throws SAXException, IOException {
    Document doc = createDoc("<module name=\"module-calling-operations-within-module\"\n" +
        "        prefix=\"modified-prefix\"\n" +
        "        namespace=\"http://www.mulesoft.org/schema/a/different/path/mule/module-calling-operations-within-module\"\n" +
        "        xmlns:tns=\"http://www.mulesoft.org/schema/a/different/path/mule/module-calling-operations-within-module\"\n" +
        "        xmlns=\"http://www.mulesoft.org/schema/mule/module\"\n" +
        "        xmlns:mule=\"http://www.mulesoft.org/schema/mule/core\"\n" +
        "        xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"\n" +
        "        xsi:schemaLocation=\"\n" +
        "           http://www.mulesoft.org/schema/mule/module http://www.mulesoft.org/schema/mule/module/current/mule-module.xsd\n"
        +
        "           http://www.mulesoft.org/schema/mule/core http://www.mulesoft.org/schema/mule/core/current/mule.xsd\n" +
        "           http://www.mulesoft.org/schema/a/different/path/mule/module-calling-operations-within-module http://www.mulesoft.org/schema/a/different/path/mule/module-calling-operations-within-module/current/mule-modified-prefix.xsd\"/>");
    reader.processAttributes(mockBuilder, doc.getDocumentElement());

    verify(mockBuilder).withNamespaceDefinition(namespaceCaptor.capture());

    NamespaceDefinition definition = namespaceCaptor.getValue();
    assertThat(definition.getNamespace(),
               is("http://www.mulesoft.org/schema/a/different/path/mule/module-calling-operations-within-module"));
    assertThat(definition.getPrefix(), is("modified-prefix"));
    assertThat(definition.getSchemaLocations(), is(aMapWithSize(3)));
    assertThat(definition.getSchemaLocations(), hasEntry("http://www.mulesoft.org/schema/mule/module",
                                                         "http://www.mulesoft.org/schema/mule/module/current/mule-module.xsd"));
    assertThat(definition.getSchemaLocations(),
               hasEntry("http://www.mulesoft.org/schema/mule/core", "http://www.mulesoft.org/schema/mule/core/current/mule.xsd"));
    assertThat(definition.getSchemaLocations(),
               hasEntry("http://www.mulesoft.org/schema/a/different/path/mule/module-calling-operations-within-module",
                        "http://www.mulesoft.org/schema/a/different/path/mule/module-calling-operations-within-module/current/mule-modified-prefix.xsd"));

    // TODO MULE-19548 assert size 4 (now is setting size 5 because name is being included)
    assertThat(definition.getUnresovedNamespaces(), hasEntry("xmlns:mule", "http://www.mulesoft.org/schema/mule/core"));
    assertThat(definition.getUnresovedNamespaces(), hasEntry("xmlns", "http://www.mulesoft.org/schema/mule/module"));
    assertThat(definition.getUnresovedNamespaces(),
               hasEntry("xmlns:tns",
                        "http://www.mulesoft.org/schema/a/different/path/mule/module-calling-operations-within-module"));
    assertThat(definition.getUnresovedNamespaces(), hasEntry("xmlns:xsi", "http://www.w3.org/2001/XMLSchema-instance"));
  }

}
