/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.xml.resolver;

import static org.mule.runtime.dsl.internal.util.SchemaMappingsUtils.CUSTOM_SCHEMA_MAPPINGS_LOCATION;
import static org.mule.runtime.dsl.internal.util.SchemaMappingsUtils.getMuleSchemasMappings;
import static org.mule.runtime.dsl.internal.util.SchemaMappingsUtils.getSchemaMappings;
import static org.mule.runtime.dsl.internal.util.SchemaMappingsUtils.getSpringSchemasMappings;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLConnection;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;

/**
 * Custom entity resolver based on Spring's schema resolver.
 * <p>
 * This code was moved from mule project
 *
 * @since 1.0
 */
public class MuleCustomEntityResolver implements EntityResolver {

  private static final Logger LOGGER = LoggerFactory.getLogger(MuleCustomEntityResolver.class);

  private final ClassLoader classLoader;
  private final Map<String, String> muleSchemaMappings;
  private final Map<String, String> appPluginsSchemaMappings;

  MuleCustomEntityResolver(ClassLoader classLoader) {
    this.classLoader = classLoader;
    this.muleSchemaMappings = getMuleSchemaMappings();
    this.appPluginsSchemaMappings = getAppPluginsSchemaMappings();
  }

  @Override
  public InputSource resolveEntity(String publicId, String systemId) {
    if (systemId != null) {
      // Runtime takes precedence over plugins, to avoid a misbehaving plugin to override something from the runtime
      InputSource source =
          resolveEntityInClassloader(muleSchemaMappings, publicId, systemId, MuleCustomEntityResolver.class.getClassLoader());

      if (source == null) {
        source = resolveEntityInClassloader(appPluginsSchemaMappings, publicId, systemId, this.classLoader);
      }

      return source;
    }
    return null;
  }

  private static InputSource resolveEntityInClassloader(Map<String, String> schemaMappings, String publicId, String systemId,
                                                        ClassLoader cl) {
    String resourceLocation = schemaMappings.get(systemId);
    if (resourceLocation != null) {
      URL resource = cl.getResource(resourceLocation);
      if (resource == null) {
        LOGGER.debug("Couldn't find XML schema [" + systemId + "]: " + resourceLocation);
        return null;
      }
      // The caller expects the stream in the InputSource to be open, so this cannot be closed before returning.
      try {
        URLConnection connection = resource.openConnection();
        connection.setUseCaches(false);
        InputStream is = connection.getInputStream();

        InputSource source = new InputSource(is);
        source.setPublicId(publicId);
        source.setSystemId(systemId);
        if (LOGGER.isDebugEnabled()) {
          LOGGER.debug("Found XML schema [" + systemId + "] in classpath: " + resourceLocation);
        }
        return source;
      } catch (IOException e) {
        LOGGER.warn("Error resolving entity [" + systemId + "]: " + resourceLocation, e);
      }
    }
    return null;
  }

  /**
   * Load the specified schema mappings.
   */
  private Map<String, String> getMuleSchemaMappings() {
    Map<String, String> schemaMappings = getMuleSchemasMappings();
    Map<String, String> springMappings = getSpringSchemasMappings();
    schemaMappings.putAll(springMappings);
    return schemaMappings;
  }

  private ClassLoader getClassLoaderToUse(ClassLoader classLoader) {
    ClassLoader classLoaderToUse = classLoader;
    if (classLoaderToUse == null) {
      classLoaderToUse = getDefaultClassLoader();
    }

    return classLoaderToUse;
  }

  public static ClassLoader getDefaultClassLoader() {
    ClassLoader cl = null;
    try {
      cl = Thread.currentThread().getContextClassLoader();
    } catch (Throwable ex) {
      // Cannot access thread context ClassLoader - falling back...
    }
    if (cl == null) {
      // No thread context class loader -> use class loader of this class.
      cl = MuleCustomEntityResolver.class.getClassLoader();
      if (cl == null) {
        // getClassLoader() returning null indicates the bootstrap ClassLoader
        try {
          cl = ClassLoader.getSystemClassLoader();
        } catch (Throwable ex) {
          // Cannot access system ClassLoader - oh well, maybe the caller can live with null...
        }
      }
    }
    return cl;
  }

  /**
   * Load the specified schema mappings.
   */
  private Map<String, String> getAppPluginsSchemaMappings() {
    return getSchemaMappings(CUSTOM_SCHEMA_MAPPINGS_LOCATION, () -> getClassLoaderToUse(this.classLoader));
  }

}
