/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.xml;

import static org.mule.runtime.ast.AllureConstants.ArtifactAst.ARTIFACT_AST;
import static org.mule.runtime.ast.AllureXmlParserConstants.DslParsing.IMPORT_HANDLING;
import static org.mule.runtime.ast.AllureXmlParserConstants.DslParsing.XML_DOM_PROCESSING;
import static org.mule.runtime.ast.api.ImportedResource.COULD_NOT_FIND_IMPORTED_RESOURCE;
import static org.mule.runtime.ast.api.ImportedResource.COULD_NOT_RESOLVE_IMPORTED_RESOURCE;
import static org.mule.runtime.ast.api.xml.AstXmlParser.builder;
import static org.mule.runtime.dsl.api.xml.parser.XmlConfigurationDocumentLoader.noValidationDocumentLoader;

import static java.lang.Thread.currentThread;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.allOf;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.endsWith;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.startsWith;
import static org.hamcrest.collection.IsMapContaining.hasEntry;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.XmlDslModel;
import org.mule.runtime.api.meta.model.construct.ConstructModel;
import org.mule.runtime.api.util.Pair;
import org.mule.runtime.api.util.xmlsecurity.XMLSecureFactories;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ImportedResource;
import org.mule.runtime.ast.api.exception.PropertyNotFoundException;
import org.mule.runtime.ast.api.xml.AstXmlParser;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.w3c.dom.Document;
import org.xml.sax.helpers.DefaultHandler;

import org.junit.Before;
import org.junit.Test;

import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;

@Feature(ARTIFACT_AST)
@Story(IMPORT_HANDLING)
public class DefaultAstXmlParserWithImportsTestCase {

  private static final XmlDslModel XML_DSL_MODEL = XmlDslModel.builder()
      .setPrefix("mule")
      .setNamespace("http://mockns")
      .build();

  private ClassLoader classLoader;
  private AstXmlParser parser;

  private final Map<String, String> properties = new HashMap<>();

  private final ExtensionModel mockedExtensionModel = mock(ExtensionModel.class);

  @Before
  public void before() {
    properties.clear();
    classLoader = DefaultAstXmlParserWithImportsTestCase.class.getClassLoader();

    ConstructModel topLevelSimpleConstruct = mock(ConstructModel.class);
    when(topLevelSimpleConstruct.getName()).thenReturn("top-level-simple");

    ConstructModel topLevelSimpleOtherConstruct = mock(ConstructModel.class);
    when(topLevelSimpleOtherConstruct.getName()).thenReturn("top-level-simple-other");

    ConstructModel topLevelImportConstruct = mock(ConstructModel.class);
    when(topLevelImportConstruct.getName()).thenReturn("top-level-import");

    when(mockedExtensionModel.getName()).thenReturn("Mock Extension Model");
    when(mockedExtensionModel.getXmlDslModel()).thenReturn(XML_DSL_MODEL);
    when(mockedExtensionModel.getConstructModels())
        .thenReturn(asList(topLevelSimpleConstruct, topLevelImportConstruct, topLevelSimpleOtherConstruct));

    parser = builder()
        .withSchemaValidationsDisabled()
        .withExtensionModel(mockedExtensionModel)
        .withPropertyResolver(propertyKey -> properties.getOrDefault(propertyKey, propertyKey))
        .build();
  }

  @Test
  public void imports() {
    final ArtifactAst importAst = parser.parse(classLoader.getResource("import.xml"));

    final List<String> components = importAst.recursiveStream().map(c -> c.getIdentifier().toString()).collect(toList());
    assertThat(components, contains("top-level-import", "top-level-simple"));

    final List<String> imports =
        importAst.getImportedResources().stream().map(ImportedResource::getResourceLocation).collect(toList());
    assertThat(imports, contains("simple.xml"));

    Map<String, List<String>> filesToImportHierarchy = obtainFilesToImportHierarchy(importAst);

    assertThat(filesToImportHierarchy,
               hasEntry(is("simple.xml"), contains(allOf(startsWith("file:/"), endsWith("/import.xml")))));
    assertThat(filesToImportHierarchy,
               hasEntry("import.xml", emptyList()));
  }

  @Test
  @Issue("MULE-19940")
  public void importsAlreadyParsed() {
    currentThread().setContextClassLoader(new ClassLoader(currentThread().getContextClassLoader()) {

      @Override
      public URL getResource(String name) {
        if (name.equals("simple.xml")) {
          try {
            // Make that the resource is not detected as a resource within the classpath
            return new File("src/test/resources/simple.xml").toURI().toURL();
          } catch (MalformedURLException e) {
            e.printStackTrace();
            return null;
          }
        } else {
          return super.getResource(name);
        }
      }
    });

    try {
      final ArtifactAst importAst = parser.parse(classLoader.getResource("import.xml"),
                                                 classLoader.getResource("simple.xml"));

      final List<String> components = importAst.recursiveStream().map(c -> c.getIdentifier().toString()).collect(toList());
      assertThat(components, contains("top-level-import", "top-level-simple"));

      final List<String> imports =
          importAst.getImportedResources().stream().map(ImportedResource::getResourceLocation).collect(toList());
      assertThat(imports, contains("simple.xml"));

      Map<String, List<String>> filesToImportHierarchy = obtainFilesToImportHierarchy(importAst);

      assertThat(filesToImportHierarchy,
                 hasEntry("simple.xml", emptyList()));
      assertThat(filesToImportHierarchy,
                 hasEntry("import.xml", emptyList()));
    } finally {
      currentThread().setContextClassLoader(currentThread().getContextClassLoader().getParent());
    }
  }

  @Test
  @Story(XML_DOM_PROCESSING)
  public void importsDocuments() {
    Document document = noValidationDocumentLoader().loadDocument(() -> XMLSecureFactories.createDefault().getSAXParserFactory(),
                                                                  new DefaultHandler(),
                                                                  "import.xml",
                                                                  classLoader.getResourceAsStream("import.xml"),
                                                                  null);

    final ArtifactAst importAst = parser.parseDocument(asList(new Pair<>("import.xml", document)));

    final List<String> components = importAst.recursiveStream().map(c -> c.getIdentifier().toString()).collect(toList());
    assertThat(components, contains("top-level-import", "top-level-simple"));

    final List<String> imports =
        importAst.getImportedResources().stream().map(ImportedResource::getResourceLocation).collect(toList());
    assertThat(imports, contains("simple.xml"));

    Map<String, List<String>> filesToImportHierarchy = obtainFilesToImportHierarchy(importAst);

    assertThat(filesToImportHierarchy,
               hasEntry(is("simple.xml"), contains("<no_file_uri>")));
    assertThat(filesToImportHierarchy,
               hasEntry("import.xml", emptyList()));
  }

  @Test
  public void importsFromDependency() {
    final ArtifactAst importAst = parser.parse(classLoader.getResource("import-from-dependency.xml"));

    final List<String> components = importAst.recursiveStream().map(c -> c.getIdentifier().toString()).collect(toList());
    assertThat(components, contains("top-level-import", "flow", "logger"));

    final List<String> imports =
        importAst.getImportedResources().stream().map(ImportedResource::getResourceLocation).collect(toList());
    assertThat(imports, contains("reusable-flows.xml"));

    Map<String, List<String>> filesToImportHierarchy = obtainFilesToImportHierarchy(importAst);

    assertThat(filesToImportHierarchy,
               hasEntry("import-from-dependency.xml", emptyList()));
    assertThat(filesToImportHierarchy,
               hasEntry(is("reusable-flows.xml"),
                        contains(allOf(startsWith("file:/"), endsWith("/import-from-dependency.xml")))));
  }

  @Test
  public void importsFromDependencyIndirect() {
    final ArtifactAst importAst = parser.parse(classLoader.getResource("import-from-dependency-indirect.xml"));

    final List<String> components = importAst.recursiveStream().map(c -> c.getIdentifier().toString()).collect(toList());
    assertThat(components, contains("top-level-import", "flow", "logger"));

    final List<String> imports =
        importAst.getImportedResources().stream().map(ImportedResource::getResourceLocation).collect(toList());
    assertThat(imports, contains("import-reusable-flows.xml", "reusable-flows.xml"));

    Map<String, List<String>> filesToImportHierarchy = obtainFilesToImportHierarchy(importAst);

    assertThat(filesToImportHierarchy,
               hasEntry("import-from-dependency-indirect.xml", emptyList()));
    assertThat(filesToImportHierarchy,
               hasEntry(is("reusable-flows.xml"),
                        contains(allOf(startsWith("file:/"), endsWith("/import-from-dependency-indirect.xml")),
                                 allOf(startsWith("jar:file:/"),
                                       endsWith("/mule-artifact-ast-mule-reusable-flows-1.1.0-SNAPSHOT.jar!/import-reusable-flows.xml")))));
  }

  @Test
  public void importsWithProperty() {
    properties.put("${prop}", "simple.xml");
    final ArtifactAst importAst = parser.parse(classLoader.getResource("import-property.xml"));

    final List<String> components = importAst.recursiveStream().map(c -> c.getIdentifier().toString()).collect(toList());
    assertThat(components, contains("top-level-import", "top-level-simple"));

    final List<String> imports =
        importAst.getImportedResources().stream().map(ImportedResource::getResourceLocation).collect(toList());
    assertThat(imports, contains("simple.xml"));
  }

  @Test
  @Issue("W-12143338")
  public void importsWithUnresolvableProperty() {
    parser = builder()
        .withSchemaValidationsDisabled()
        .withExtensionModel(mockedExtensionModel)
        .withPropertyResolver(propertyKey -> {
          throw new PropertyNotFoundException(new Pair<>("hardcoded", "prop"));
        })
        .build();

    final ArtifactAst importAst = parser.parse(classLoader.getResource("import-property.xml"));

    ImportedResource unresolvableImport = importAst.getImportedResources().iterator().next();

    assertThat(unresolvableImport.getResolutionFailure().get(),
               is(COULD_NOT_RESOLVE_IMPORTED_RESOURCE
                   + "'${prop}': Couldn't find configuration property value for key ${prop}"));
  }

  @Test
  public void importsWithPropertyChangingValue() {
    properties.put("${prop}", "simple.xml");
    final ArtifactAst importAst = parser.parse(classLoader.getResource("import-property.xml"));

    final List<String> components = importAst.recursiveStream().map(c -> c.getIdentifier().toString()).collect(toList());
    assertThat(components, contains("top-level-import", "top-level-simple"));

    final List<String> imports =
        importAst.getImportedResources().stream().map(ImportedResource::getResourceLocation).collect(toList());
    assertThat(imports, contains("simple.xml"));

    importAst.updatePropertiesResolver(key -> {
      if (key.equals("${prop}")) {
        return "alternativeValue";
      } else {
        return null;
      }
    });
    final List<String> changedImports =
        importAst.getImportedResources().stream().map(ImportedResource::getResourceLocation).collect(toList());
    assertThat(changedImports, contains("alternativeValue"));
  }

  @Test
  public void importsTwoLevels() {
    properties.put("${prop}", "simple.xml");
    final ArtifactAst importAst = parser.parse(classLoader.getResource("import-two-levels.xml"));

    final List<String> components = importAst.recursiveStream().map(c -> c.getIdentifier().toString()).collect(toList());
    assertThat(components,
               contains("top-level-import", "top-level-import", "top-level-simple", "top-level-import", "flow", "logger",
                        "top-level-import"));

    final List<String> imports =
        importAst.getImportedResources().stream().map(ImportedResource::getResourceLocation).collect(toList());
    assertThat(imports,
               contains("import.xml", "simple.xml", "import-from-dependency.xml", "reusable-flows.xml", "import-property.xml",
                        "simple.xml"));

    Map<String, List<String>> filesToImportHierarchy = obtainFilesToImportHierarchy(importAst);

    assertThat(filesToImportHierarchy,
               hasEntry(is("simple.xml"), contains(allOf(startsWith("file:/"), endsWith("/import-two-levels.xml")),
                                                   allOf(startsWith("file:/"), endsWith("/import.xml")))));
    assertThat(filesToImportHierarchy,
               hasEntry(is("import-property.xml"), contains(allOf(startsWith("file:/"), endsWith("/import-two-levels.xml")))));
    assertThat(filesToImportHierarchy,
               hasEntry("import-two-levels.xml", emptyList()));
    assertThat(filesToImportHierarchy,
               hasEntry(is("import.xml"), contains(allOf(startsWith("file:/"), endsWith("/import-two-levels.xml")))));
    assertThat(filesToImportHierarchy,
               hasEntry(is("import-from-dependency.xml"),
                        contains(allOf(startsWith("file:/"), endsWith("/import-two-levels.xml")))));
    assertThat(filesToImportHierarchy,
               hasEntry(is("reusable-flows.xml"),
                        contains(allOf(startsWith("file:/"), endsWith("/import-two-levels.xml")),
                                 allOf(startsWith("file:/"), endsWith("/import-from-dependency.xml")))));
  }

  @Test
  @Story(XML_DOM_PROCESSING)
  public void importsTwoLevelsDocuments() {
    Document document = noValidationDocumentLoader().loadDocument(() -> XMLSecureFactories.createDefault().getSAXParserFactory(),
                                                                  new DefaultHandler(),
                                                                  "import-two-levels.xml",
                                                                  classLoader.getResourceAsStream("import-two-levels.xml"),
                                                                  null);

    properties.put("${prop}", "simple.xml");
    final ArtifactAst importAst = parser.parseDocument(asList(new Pair<>("import-two-levels.xml", document)));

    final List<String> components = importAst.recursiveStream().map(c -> c.getIdentifier().toString()).collect(toList());
    assertThat(components,
               contains("top-level-import", "top-level-import", "top-level-simple", "top-level-import", "flow", "logger",
                        "top-level-import"));

    final List<String> imports =
        importAst.getImportedResources().stream().map(ImportedResource::getResourceLocation).collect(toList());
    assertThat(imports,
               contains("import.xml", "simple.xml", "import-from-dependency.xml", "reusable-flows.xml", "import-property.xml",
                        "simple.xml"));

    Map<String, List<String>> filesToImportHierarchy = obtainFilesToImportHierarchy(importAst);

    assertThat(filesToImportHierarchy,
               hasEntry(is("simple.xml"), contains(is("<no_file_uri>"),
                                                   allOf(startsWith("file:/"), endsWith("/import.xml")))));
    assertThat(filesToImportHierarchy,
               hasEntry(is("import-property.xml"), contains("<no_file_uri>")));
    assertThat(filesToImportHierarchy,
               hasEntry("import-two-levels.xml", emptyList()));
    assertThat(filesToImportHierarchy,
               hasEntry(is("import.xml"), contains("<no_file_uri>")));
    assertThat(filesToImportHierarchy,
               hasEntry(is("import-from-dependency.xml"), contains("<no_file_uri>")));
    assertThat(filesToImportHierarchy,
               hasEntry(is("reusable-flows.xml"), contains(is("<no_file_uri>"),
                                                           allOf(startsWith("file:/"),
                                                                 endsWith("/import-from-dependency.xml")))));
  }

  private Map<String, List<String>> obtainFilesToImportHierarchy(final ArtifactAst importAst) {
    Map<String, List<String>> filesToImportHierarchy = importAst.recursiveStream()
        .collect(toMap(c -> c.getMetadata().getFileName().get(),
                       c -> c.getMetadata().getImportChain().stream()
                           .map(imp -> imp.getMetadata().getFileUri().map(Object::toString).orElse("<no_file_uri>"))
                           .collect(toList()),
                       (k, u) -> u,
                       HashMap::new));
    return filesToImportHierarchy;
  }

  @Test
  public void twoConfigsAndImport() {
    final ArtifactAst importAst = parser.parse(classLoader.getResource("import.xml"),
                                               classLoader.getResource("simple-other.xml"));

    final List<String> components = importAst.recursiveStream().map(c -> c.getIdentifier().toString()).collect(toList());
    assertThat(components, contains("top-level-import", "top-level-simple-other", "top-level-simple"));

    final List<String> imports =
        importAst.getImportedResources().stream().map(ImportedResource::getResourceLocation).collect(toList());
    assertThat(imports, contains("simple.xml"));
  }

  @Test
  public void twoConfigsAndImportSame() {
    final ArtifactAst importAst = parser.parse(classLoader.getResource("import.xml"),
                                               classLoader.getResource("simple.xml"));

    final List<String> components = importAst.recursiveStream().map(c -> c.getIdentifier().toString()).collect(toList());
    assertThat(components, contains("top-level-import", "top-level-simple"));

    final List<String> imports =
        importAst.getImportedResources().stream().map(ImportedResource::getResourceLocation).collect(toList());
    assertThat(imports, contains("simple.xml"));
  }

  @Test
  public void twoConfigsAndImportSameInverse() {
    final ArtifactAst importAst = parser.parse(classLoader.getResource("simple.xml"),
                                               classLoader.getResource("import.xml"));

    final List<String> components = importAst.recursiveStream().map(c -> c.getIdentifier().toString()).collect(toList());
    assertThat(components, contains("top-level-simple", "top-level-import"));

    final List<String> imports =
        importAst.getImportedResources().stream().map(ImportedResource::getResourceLocation).collect(toList());
    assertThat(imports, contains("simple.xml"));
  }

  @Test
  public void twoConfigsAndImportSameWithProperty() {
    properties.put("${prop}", "simple.xml");
    final ArtifactAst importAst = parser.parse(classLoader.getResource("import.xml"),
                                               classLoader.getResource("import-property.xml"));

    final List<String> components = importAst.recursiveStream().map(c -> c.getIdentifier().toString()).collect(toList());
    // the file containing top-level-simple is imported twice, but the second is ignored, so only ine such component ends up in
    // the AST.
    assertThat(components, contains("top-level-import", "top-level-import", "top-level-simple"));

    final List<String> imports =
        importAst.getImportedResources().stream().map(ImportedResource::getResourceLocation).collect(toList());
    assertThat(imports, contains("simple.xml", "simple.xml"));
  }

  @Test
  public void unresolvableImport() {
    final ArtifactAst importAst = parser.parse(classLoader.getResource("import-unresolvable.xml"));
    ImportedResource unresolvableImport = importAst.getImportedResources().iterator().next();

    assertThat(unresolvableImport.getResolutionFailure().get(),
               is(COULD_NOT_FIND_IMPORTED_RESOURCE + "'doesnt_exist.xml'"));
  }

  @Test
  public void scrambledImportTarget() {
    final ArtifactAst importAst = parser.parse(classLoader.getResource("import-scrambled.xml"));
    ImportedResource scrambledImport = importAst.getImportedResources().iterator().next();

    assertThat(scrambledImport.getResolutionFailure().get(),
               is("Error loading: mule-config-scrambled.xml, Content is not allowed in prolog."));
  }

}
