/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.xml.resolver;

import static java.lang.String.format;
import static java.nio.charset.StandardCharsets.UTF_8;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.runtime.api.dsl.DslResolvingContext;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.XmlDslModel;
import org.mule.runtime.ast.api.xml.AstXmlParser;
import org.mule.runtime.ast.internal.xml.DefaultAstXmlParserTestCase;
import org.mule.runtime.extension.api.dsl.syntax.resources.spi.ExtensionSchemaGenerator;

import java.io.IOException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.io.IOUtils;
import org.junit.Before;
import org.junit.Test;

public class CachingExtensionSchemaGeneratorTestCase {

  private static int schemaGenerateCounter = 0;

  private ClassLoader classLoader;
  private AstXmlParser parser;

  private final Map<String, String> properties = new HashMap<>();

  @Before
  public void before() {
    schemaGenerateCounter = 0;
    properties.clear();
    classLoader = DefaultAstXmlParserTestCase.class.getClassLoader();

    parser = AstXmlParser.builder()
        // .withSchemaValidationsDisabled()
        .withExtensionModel(getMockHeisenbergModel())
        .withPropertyResolver(propertyKey -> properties.getOrDefault(propertyKey, propertyKey))
        .build();
  }

  private ExtensionModel getMockHeisenbergModel() {
    return mockExtensionModel("Heisenberg");
  }

  private ExtensionModel mockExtensionModel(String name) {
    final ExtensionModel model = mock(ExtensionModel.class);
    final String prefix = name.toLowerCase();

    when(model.getName()).thenReturn(name);
    when(model.getXmlDslModel())
        .thenReturn(XmlDslModel.builder()
            .setNamespace("http://www.mulesoft.org/schema/mule/" + prefix)
            .setSchemaLocation(format("http://www.mulesoft.org/schema/mule/%s/current/mule-%s.xsd", prefix, prefix))
            .setPrefix(prefix)
            .build());

    return model;
  }

  @Test
  public void generatedSchemaIsCached() {
    parser.parse(classLoader.getResource("mule-config-with-extension.xml"));
    parser.parse(classLoader.getResource("mule-config-with-extension.xml"));
    assertThat(schemaGenerateCounter, is(1));
  }

  @Test
  public void applicationSchemaNotCached() {
    parser = AstXmlParser.builder()
        .withSchemaValidationsDisabled()
        .withExtensionModels(Arrays.asList(getMockHeisenbergModel(), mockExtensionModel("this")))
        .build();

    parser.parse(classLoader.getResource("hello-world-operation.xml"));
    parser.parse(classLoader.getResource("hello-world-operation.xml"));

    assertThat(schemaGenerateCounter, is(0));
  }

  public static final class TestExtensionSchemaGenerator implements ExtensionSchemaGenerator {

    @Override
    public String generate(ExtensionModel extensionModel, DslResolvingContext context) {
      schemaGenerateCounter++;
      if (extensionModel.getName().equals("Heisenberg")) {
        try {
          return IOUtils.toString(DefaultAstXmlParserTestCase.class.getClassLoader().getResource("META-INF/heisenberg.xsd"),
                                  UTF_8);
        } catch (IOException e) {
          throw new MuleRuntimeException(e);
        }
      } else {
        return "";
      }
    }
  }
}
