/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.api.xml;

import static org.mule.runtime.ast.api.ArtifactType.APPLICATION;
import static org.mule.runtime.dsl.api.xml.XmlDslConstants.IMPORT_ELEMENT;
import static org.mule.runtime.internal.dsl.DslConstants.CORE_PREFIX;

import static java.util.Optional.empty;

import org.mule.api.annotation.NoImplement;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.util.Pair;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ArtifactType;
import org.mule.runtime.ast.internal.xml.DefaultAstXmlParser;
import org.mule.runtime.ast.internal.xml.resolver.FailAfterTenErrorsResolveEntityFailStrategy;
import org.mule.runtime.ast.internal.xml.resolver.FailOnFirstErrorResolveEntityFailStrategy;
import org.mule.runtime.ast.internal.xml.resolver.ResolveEntityFailStrategy;
import org.mule.runtime.dsl.api.ConfigResource;
import org.mule.runtime.dsl.api.xml.parser.ParsingPropertyResolver;
import org.mule.runtime.dsl.internal.xml.parser.XmlMetadataAnnotations;
import org.mule.runtime.extension.api.dsl.syntax.resources.spi.ExtensionSchemaGenerator;
import org.mule.runtime.extension.internal.spi.ExtensionsApiSpiUtils;

import java.io.InputStream;
import java.net.URI;
import java.net.URL;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import org.w3c.dom.Document;
import org.w3c.dom.Node;

/**
 * Provides a mechanism to parse a XML config for a deployable mule artifact into an {@link ArtifactAst}.
 * <p>
 * This parser needs to know the extension models that are used by the artifact in order to generate the {@link ArtifactAst}
 * correctly.
 * <p>
 * A single parser instance may be used to generate more than one {@link ArtifactAst} from different configs, since no state
 * specific to a parsing process is kept by this parser. The same parser may be used to parse different configs concurrently, as
 * well.
 *
 * @since 1.0
 */
@NoImplement
public interface AstXmlParser {

  ComponentIdentifier IMPORT_IDENTIFIER = ComponentIdentifier.builder().namespace(CORE_PREFIX).name(IMPORT_ELEMENT).build();
  ComponentIdentifier MULE_ROOT_IDENTIFIER = ComponentIdentifier.builder().namespace(CORE_PREFIX).name("mule").build();
  ComponentIdentifier DOMAIN_ROOT_IDENTIFIER = ComponentIdentifier.builder().namespace(CORE_PREFIX).name("mule-domain").build();
  ComponentIdentifier ANNOTATIONS_IDENTIFIER = ComponentIdentifier.builder().namespace(CORE_PREFIX).name("annotations").build();

  /**
   * Creates a builder for an {@link AstXmlParser}.
   *
   * @return a builder for an {@link AstXmlParser}.
   */
  static Builder builder() {
    return new Builder();
  }

  /**
   * Generate a new {@link ArtifactAst} representation from the XML DOM tree.
   * <p>
   * The parsing process will resolve references in {@link #IMPORT_IDENTIFIER import} tags. Relative paths will be resolved using
   * the currently set {@link Thread#getContextClassLoader() ThreadContextClassLoader}.
   * <p>
   * Note: The component metadata that is generated relies on data populated at the time of parsing the config XML into its
   * {@link Document}. In the case of this method, the provided {@link Document} must be of
   * <a href="https://www.w3.org/TR/DOM-Level-3-Core/">level 3</a> and have that data populated for the component metadata about
   * the underlying DSL to be accurate. The required data is a {@link Node#getUserData(String) userData} with key
   * {@link XmlMetadataAnnotations#METADATA_ANNOTATIONS_KEY} and value of type {@link XmlMetadataAnnotations} on every node of the
   * DOM tree.
   *
   * @param artifactName          the name of the artifact to parse
   * @param appXmlConfigDocuments the XML content to be parsed. The first part of each pair is the name to be used as fileName in
   *                              the metadata of the generates {@link ArtifactAst}.
   * @return the {@link ArtifactAst} representation of the provided XML config.
   */
  ArtifactAst parseDocument(String artifactName, List<Pair<String, Document>> appXmlConfigDocuments);

  /**
   * Generate a new {@link ArtifactAst} representation form the XML DOM tree.
   * <p>
   * The parsing process will resolve references in {@link #IMPORT_IDENTIFIER import} tags. Relative paths will be resolved using
   * the currently set {@link Thread#getContextClassLoader() ThreadContextClassLoader}.
   * <p>
   * Note: The component metadata that is generated relies on data populated at the time of parsing the config XML into its
   * {@link Document}. In the case of this method, the provided {@link Document} must be of
   * <a href="https://www.w3.org/TR/DOM-Level-3-Core/">level 3</a> and have that data populated for the component metadata about
   * the underlying DSL to be accurate. The required data is a {@link Node#getUserData(String) userData} with key
   * {@link XmlMetadataAnnotations#METADATA_ANNOTATIONS_KEY} and value of type {@link XmlMetadataAnnotations} on every node of the
   * DOM tree.
   *
   * @param appXmlConfigDocuments the XML content to be parsed. The first part of each pair is the name to be used as fileName in
   *                              the metadata of the generates {@link ArtifactAst}.
   * @return the {@link ArtifactAst} representation of the provided XML config.
   * 
   * @since 1.1
   */
  ArtifactAst parseDocument(List<Pair<String, Document>> appXmlConfigDocuments);

  /**
   * Generate a new {@link ArtifactAst} representation form the XML read from the provided inputStreams.
   * <p>
   * The parsing process will resolve references in {@link #IMPORT_IDENTIFIER import} tags. Relative paths will be resolved using
   * the currently set {@link Thread#getContextClassLoader() ThreadContextClassLoader}.
   *
   * @param artifactName             the name of the artifact to parse
   * @param appXmlConfigInputStreams the streams that will provide the XML content to be parsed. The first part of each pair is
   *                                 the name to be used as fileName in the metadata of the generates {@link ArtifactAst}.
   * @return the {@link ArtifactAst} representation of the provided XML config.
   *
   * @since 1.1
   */
  ArtifactAst parse(String artifactName, List<Pair<String, InputStream>> appXmlConfigInputStreams);

  /**
   * Generate a new {@link ArtifactAst} representation form the XML read from the provided inputStreams.
   * <p>
   * The parsing process will resolve references in {@link #IMPORT_IDENTIFIER import} tags. Relative paths will be resolved using
   * the currently set {@link Thread#getContextClassLoader() ThreadContextClassLoader}.
   *
   * @param appXmlConfigInputStreams the streams that will provide the XML content to be parsed. The first part of each pair is
   *                                 the name to be used as fileName in the metadata of the generates {@link ArtifactAst}.
   * @return the {@link ArtifactAst} representation of the provided XML config.
   */
  ArtifactAst parse(List<Pair<String, InputStream>> appXmlConfigInputStreams);

  /**
   * Generate a new {@link ArtifactAst} representation form the XML read from the provided inputStreams.
   * <p>
   * The parsing process will resolve references in {@link #IMPORT_IDENTIFIER import} tags. Relative paths will be resolved using
   * the currently set {@link Thread#getContextClassLoader() ThreadContextClassLoader}.
   *
   * @param artifactName          the name of the artifact to parse
   * @param appXmlConfigResources the resources that will provide the XML content to be parsed.
   * @return the {@link ArtifactAst} representation of the provided XML config.
   */
  ArtifactAst parse(String artifactName, ConfigResource... appXmlConfigResources);

  /**
   * Generate a new {@link ArtifactAst} representation form the XML read from the provided inputStreams.
   * <p>
   * The parsing process will resolve references in {@link #IMPORT_IDENTIFIER import} tags. Relative paths will be resolved using
   * the currently set {@link Thread#getContextClassLoader() ThreadContextClassLoader}.
   *
   * @param appXmlConfigResources the resources that will provide the XML content to be parsed.
   * @return the {@link ArtifactAst} representation of the provided XML config.
   */
  ArtifactAst parse(ConfigResource... appXmlConfigResources);

  /**
   * Generate a new {@link ArtifactAst} representation form the XML read from the provided inputStream.
   * <p>
   * The parsing process will resolve references in {@link #IMPORT_IDENTIFIER import} tags. Relative paths will be resolved using
   * the currently set {@link Thread#getContextClassLoader() ThreadContextClassLoader}.
   *
   * @param artifactName            the name of the artifact to parse
   * @param resourceName            the name to be used as fileName in the metadata of the generates {@link ArtifactAst}.
   * @param appXmlConfigInputStream the stream that will provide the XML content to be parsed.
   * @return the {@link ArtifactAst} representation of the provided XML config.
   */
  ArtifactAst parse(String artifactName, String resourceName, InputStream appXmlConfigInputStream);

  /**
   * Generate a new {@link ArtifactAst} representation form the XML read from the provided inputStream.
   * <p>
   * The parsing process will resolve references in {@link #IMPORT_IDENTIFIER import} tags. Relative paths will be resolved using
   * the currently set {@link Thread#getContextClassLoader() ThreadContextClassLoader}.
   *
   * @param resourceName            the name to be used as fileName in the metadata of the generates {@link ArtifactAst}.
   * @param appXmlConfigInputStream the stream that will provide the XML content to be parsed.
   * @return the {@link ArtifactAst} representation of the provided XML config.
   */
  ArtifactAst parse(String resourceName, InputStream appXmlConfigInputStream);

  /**
   * Generate a new {@link ArtifactAst} representation form the XML read from the provided uris.
   * <p>
   * The parsing process will resolve references in {@link #IMPORT_IDENTIFIER import} tags. Relative paths will be resolved using
   * the currently set {@link Thread#getContextClassLoader() ThreadContextClassLoader}.
   *
   * @param artifactName     the name of the artifact to parse
   * @param appXmlConfigUris the uris that will provide the XML content to be parsed.
   * @return the {@link ArtifactAst} representation of the provided XML config.
   */
  ArtifactAst parse(String artifactName, URI... appXmlConfigUris);

  /**
   * Generate a new {@link ArtifactAst} representation form the XML read from the provided uris.
   * <p>
   * The parsing process will resolve references in {@link #IMPORT_IDENTIFIER import} tags. Relative paths will be resolved using
   * the currently set {@link Thread#getContextClassLoader() ThreadContextClassLoader}.
   *
   * @param appXmlConfigUris the uris that will provide the XML content to be parsed.
   * @return the {@link ArtifactAst} representation of the provided XML config.
   */
  ArtifactAst parse(URI... appXmlConfigUris);

  /**
   * Generate a new {@link ArtifactAst} representation form the XML read from the provided urls.
   * <p>
   * The parsing process will resolve references in {@link #IMPORT_IDENTIFIER import} tags. Relative paths will be resolved using
   * the currently set {@link Thread#getContextClassLoader() ThreadContextClassLoader}.
   *
   * @param artifactName     the name of the artifact to parse
   * @param appXmlConfigUrls the urls that will provide the XML content to be parsed.
   * @return the {@link ArtifactAst} representation of the provided XML config.
   */
  ArtifactAst parse(String artifactName, URL... appXmlConfigUrls);

  /**
   * Generate a new {@link ArtifactAst} representation form the XML read from the provided urls.
   * <p>
   * The parsing process will resolve references in {@link #IMPORT_IDENTIFIER import} tags. Relative paths will be resolved using
   * the currently set {@link Thread#getContextClassLoader() ThreadContextClassLoader}.
   *
   * @param appXmlConfigUrls the urls that will provide the XML content to be parsed.
   * @return the {@link ArtifactAst} representation of the provided XML config.
   */
  ArtifactAst parse(URL... appXmlConfigUrls);

  /**
   * A builder for creating {@link AstXmlParser} instances.
   */
  final class Builder {

    private static Optional<ExtensionSchemaGenerator> SCHEMA_GENERATOR;

    static {
      SCHEMA_GENERATOR = ExtensionsApiSpiUtils.loadExtensionSchemaGenerators()
          .findFirst();
    }

    private final Set<ExtensionModel> extensionModels = new HashSet<>();
    private ArtifactType artifactType = APPLICATION;
    private ArtifactAst parentArtifact;
    private boolean schemaValidations = true;
    private ParsingPropertyResolver propertyResolver = k -> k;
    private ResolveEntityFailStrategy failStrategy = new FailOnFirstErrorResolveEntityFailStrategy();

    /**
     * Add an extension model for the parser.
     * <p>
     * The target parser must contain the extension models for every namespace that is used in the XML config for the
     * {@link ArtifactAst}s to be created correctly.
     *
     * @param extensionModel the extension model to add to the target parser
     * @return the updated builder.
     */
    public Builder withExtensionModel(ExtensionModel extensionModel) {
      this.extensionModels.add(extensionModel);
      return this;
    }

    /**
     * Add an extension model for the parser.
     * <p>
     * The target parser must contain the extension models for every namespace that is used in the XML config for the
     * {@link ArtifactAst}s to be created correctly.
     *
     * @param extensionModels the extension model to add to the target parser
     * @return the updated builder.
     */
    public Builder withExtensionModels(Collection<ExtensionModel> extensionModels) {
      this.extensionModels.addAll(extensionModels);
      return this;
    }

    /**
     * Set the {@link ArtifactType} of the target {@link ArtifactAst}
     * 
     * @param artifactType the type pf the target artifact.
     * @return the updated builder.
     */
    public Builder withArtifactType(ArtifactType artifactType) {
      this.artifactType = artifactType;
      return this;
    }

    /**
     * Set the {@link ArtifactAst} from the parent artifact (i.e.: the parent domain)
     *
     * @param parentArtifact the {@link ArtifactAst} that is the parent of the artifacts generated with this parser.
     * @return the updated builder.
     */
    public Builder withParentArtifact(ArtifactAst parentArtifact) {
      this.parentArtifact = parentArtifact;
      return this;
    }

    /**
     * Disables schema validations during the parse process.
     * <p>
     * This will speed up the parsing process, but may provide an invalid {@link ArtifactAst} for invalid XMLs.
     *
     * @return the updated builder.
     */
    public Builder withSchemaValidationsDisabled() {
      this.schemaValidations = false;
      return this;
    }

    /**
     * Sets the {@code propertyResolver} to be used to resolve property placeholders ({@code ${...}) found in the config.
     *
     * @param propertyResolver the property resolver to be used by the created {@link ArtifactAst}s.
     *
     * @return the updated builder.
     */
    public Builder withPropertyResolver(ParsingPropertyResolver propertyResolver) {
      this.propertyResolver = propertyResolver;
      return this;
    }

    public Builder withLegacyFailStrategy() {
      this.failStrategy = new FailAfterTenErrorsResolveEntityFailStrategy();
      return this;
    }

    /**
     * Creates a new {@link ArtifactAst} according to the state set to this builder.
     *
     * @return a newly built {@link ArtifactAst}.
     */
    public AstXmlParser build() {
      return new DefaultAstXmlParser(artifactType, extensionModels, parentArtifact,
                                     schemaValidations ? SCHEMA_GENERATOR : empty(),
                                     propertyResolver,
                                     failStrategy);
    }
  }
}
