/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.xml;

import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.ast.api.ImportedResource.COULD_NOT_FIND_IMPORTED_RESOURCE;
import static org.mule.runtime.ast.api.ImportedResource.COULD_NOT_RESOLVE_IMPORTED_RESOURCE;
import static org.mule.runtime.ast.api.util.MuleAstUtils.emptyArtifact;
import static org.mule.runtime.ast.internal.xml.XmlNamespaceInfoProviderSupplier.createFromExtensionModels;
import static org.mule.runtime.dsl.api.xml.parser.XmlConfigurationDocumentLoader.noValidationDocumentLoader;
import static org.mule.runtime.dsl.api.xml.parser.XmlConfigurationDocumentLoader.schemaValidatingDocumentLoader;
import static org.mule.runtime.dsl.internal.xerces.xni.parser.DefaultXmlGrammarPoolManager.getGrammarPool;
import static org.mule.runtime.extension.internal.dsl.xml.XmlDslConstants.MODULE_DSL_NAMESPACE;
import static org.mule.runtime.extension.internal.dsl.xml.XmlDslConstants.MODULE_DSL_NAMESPACE_URI;
import static org.mule.runtime.extension.internal.dsl.xml.XmlDslConstants.MODULE_ROOT_NODE_NAME;

import static java.lang.String.format;
import static java.lang.Thread.currentThread;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static java.util.Collections.unmodifiableSet;
import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toList;

import static org.slf4j.LoggerFactory.getLogger;

import org.mule.apache.xerces.xni.grammars.XMLGrammarPool;
import org.mule.runtime.api.dsl.DslResolvingContext;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.util.Pair;
import org.mule.runtime.api.util.xmlsecurity.XMLSecureFactories;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ArtifactType;
import org.mule.runtime.ast.api.ImportedResource;
import org.mule.runtime.ast.api.builder.ArtifactAstBuilder;
import org.mule.runtime.ast.api.exception.PropertyNotFoundException;
import org.mule.runtime.ast.api.xml.AstXmlParser;
import org.mule.runtime.ast.internal.builder.ImportedResourceBuilder;
import org.mule.runtime.ast.internal.xml.model.AggregatedImportsArtifactAst;
import org.mule.runtime.ast.internal.xml.model.AggregatedMultiConfigsArtifactAst;
import org.mule.runtime.ast.internal.xml.reader.ComponentAstReader;
import org.mule.runtime.ast.internal.xml.resolver.CachingExtensionSchemaGenerator;
import org.mule.runtime.ast.internal.xml.resolver.ModuleDelegatingEntityResolver;
import org.mule.runtime.ast.internal.xml.resolver.ResolveEntityFailStrategy;
import org.mule.runtime.dsl.api.ConfigResource;
import org.mule.runtime.dsl.api.xml.XmlNamespaceInfoProvider;
import org.mule.runtime.dsl.api.xml.parser.ParsingPropertyResolver;
import org.mule.runtime.dsl.api.xml.parser.XmlApplicationParser;
import org.mule.runtime.dsl.api.xml.parser.XmlConfigurationDocumentLoader;
import org.mule.runtime.extension.api.dsl.syntax.resources.spi.ExtensionSchemaGenerator;

import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Stream;

import javax.xml.parsers.SAXParserFactory;

import com.google.common.collect.ImmutableList;

import org.slf4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.EntityResolver;
import org.xml.sax.helpers.DefaultHandler;

public class DefaultAstXmlParser implements AstXmlParser {

  private static final Logger LOGGER = getLogger(DefaultAstXmlParser.class);

  private static final SAXParserFactory SAX_PARSER_FACTORY = XMLSecureFactories.createDefault().getSAXParserFactory();

  private static final ComponentAstReader CONFIG_READER = new ComponentAstReader();

  private final XmlConfigurationDocumentLoader xmlConfigurationDocumentLoader;
  private final XMLGrammarPool grammarPool;
  private final EntityResolver entityResolver;
  private final XmlApplicationParser xmlApplicationParser;
  private final ArtifactType artifactType;
  private final Set<ExtensionModel> extensionModels;
  private final ArtifactAst parentArtifact;
  private final ParsingPropertyResolver propertyResolver;

  public DefaultAstXmlParser(ArtifactType artifactType,
                             Set<ExtensionModel> extensionModels, ArtifactAst parentArtifact,
                             Optional<ExtensionSchemaGenerator> schemaGenerator,
                             ParsingPropertyResolver propertyResolver,
                             ResolveEntityFailStrategy resolveEntityFailStrategy) {
    this.artifactType = artifactType;
    this.extensionModels = unmodifiableSet(new HashSet<>(extensionModels));
    this.parentArtifact = parentArtifact;

    final DslResolvingContext dslResolvingContext = DslResolvingContext.getDefault(this.extensionModels);
    this.xmlConfigurationDocumentLoader = schemaGenerator.isPresent()
        ? schemaValidatingDocumentLoader()
        : noValidationDocumentLoader();
    this.grammarPool = schemaGenerator.isPresent()
        ? getGrammarPool().orElse(null)
        : null;

    this.entityResolver = schemaGenerator
        .map(sg -> (EntityResolver) new ModuleDelegatingEntityResolver(extensionModels, new CachingExtensionSchemaGenerator(sg),
                                                                       dslResolvingContext, resolveEntityFailStrategy))
        .orElseGet(() -> new DefaultHandler());
    this.propertyResolver = propertyResolver;

    // How will this resolve the stuff from the runtime when running in 'light' mode?
    List<XmlNamespaceInfoProvider> namespaceInfoProviders = createFromExtensionModels(extensionModels);
    xmlApplicationParser = new XmlApplicationParser(namespaceInfoProviders);
  }

  @Override
  public ArtifactAst parseDocument(String artifactName, List<Pair<String, Document>> appXmlConfigDocuments) {
    if (appXmlConfigDocuments.isEmpty()) {
      throw new IllegalArgumentException("At least one 'appXmlConfigInputStream' must be provided");
    }

    LOGGER.debug("About to parse AST from XML Documents {}", appXmlConfigDocuments);

    final List<ConfigResource> baseAstResources = appXmlConfigDocuments.stream()
        .map(is -> {
          try {
            return new ConfigResource(is.getFirst());
          } catch (IOException e) {
            throw new MuleRuntimeException(e);
          }
        })
        .collect(toList());

    final Map<ConfigResource, ArtifactAst> importedConfigs = new LinkedHashMap<>();
    baseAstResources.forEach(r -> importedConfigs.put(r, emptyArtifact()));

    return resolveImports(aggregateAsts(appXmlConfigDocuments.stream()
        .map(resource -> doParse(artifactName, resource.getSecond(), resource.getFirst(), null, emptyList()))
        .collect(toList())), importedConfigs);
  }

  @Override
  public ArtifactAst parseDocument(List<Pair<String, Document>> appXmlConfigDocuments) {
    return parseDocument("artifact", appXmlConfigDocuments);
  }

  @Override
  public ArtifactAst parse(String artifactName, List<Pair<String, InputStream>> appXmlConfigInputStreams) {
    if (appXmlConfigInputStreams.isEmpty()) {
      throw new IllegalArgumentException("At least one 'appXmlConfigInputStream' must be provided");
    }

    LOGGER.debug("About to parse AST from inputStreams {}", appXmlConfigInputStreams);

    final List<ConfigResource> baseAstResources = appXmlConfigInputStreams.stream()
        .map(is -> inputStreamToConfigResource(is.getFirst(), is.getSecond()))
        .collect(toList());

    return resolveImports(aggregateAsts(baseAstResources.stream()
        .map(resource -> doParse(artifactName, resource))
        .collect(toList())), baseAstResources);
  }

  @Override
  public ArtifactAst parse(List<Pair<String, InputStream>> appXmlConfigInputStreams) {
    return parse("artifact", appXmlConfigInputStreams);
  }

  @Override
  public ArtifactAst parse(String artifactName, ConfigResource... appXmlConfigResources) {
    if (appXmlConfigResources.length == 0) {
      throw new IllegalArgumentException("At least one 'appXmlConfigResources' must be provided");
    }

    final List<ConfigResource> baseAstResources = asList(appXmlConfigResources);

    LOGGER.debug("About to parse AST from config resources {}", baseAstResources);

    return resolveImports(aggregateAsts(baseAstResources.stream()
        .map(resource -> doParse(artifactName, resource))
        .collect(toList())), baseAstResources);
  }

  @Override
  public ArtifactAst parse(ConfigResource... appXmlConfigResources) {
    return parse("artifact", appXmlConfigResources);
  }

  @Override
  public ArtifactAst parse(String artifactName, String resourceName, InputStream appXmlConfigInputStream) {
    LOGGER.debug("About to parse AST from inputStream {}, {}", resourceName, appXmlConfigInputStream);
    final ConfigResource baseAstResource = inputStreamToConfigResource(resourceName, appXmlConfigInputStream);
    return resolveImports(doParse(artifactName, baseAstResource), singletonList(baseAstResource));
  }

  @Override
  public ArtifactAst parse(String resourceName, InputStream appXmlConfigInputStream) {
    return parse("artifact", resourceName, appXmlConfigInputStream);
  }

  @Override
  public ArtifactAst parse(String artifactName, URI... appXmlConfigUris) {
    if (appXmlConfigUris.length == 0) {
      throw new IllegalArgumentException("At least one 'appXmlConfigUri' must be provided");
    }

    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("About to parse AST from uris {}", asList(appXmlConfigUris));
    }

    final List<ConfigResource> baseAstResources = Stream.of(appXmlConfigUris)
        .map(this::uriToConfigResource)
        .collect(toList());

    return resolveImports(aggregateAsts(baseAstResources.stream()
        .map(resource -> doParse(artifactName, resource))
        .collect(toList())), baseAstResources);
  }

  @Override
  public ArtifactAst parse(URI... appXmlConfigUris) {
    return parse("artifact", appXmlConfigUris);
  }

  @Override
  public ArtifactAst parse(String artifactName, URL... appXmlConfigUrls) {
    if (appXmlConfigUrls.length == 0) {
      throw new IllegalArgumentException("At least one 'appXmlConfigUrl' must be provided");
    }

    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("About to parse AST from urls {}", asList(appXmlConfigUrls));
    }

    final List<ConfigResource> baseAstResources = Stream.of(appXmlConfigUrls)
        .map(this::urlToConfigResource)
        .collect(toList());

    return resolveImports(aggregateAsts(baseAstResources.stream()
        .map(resource -> doParse(artifactName, resource))
        .collect(toList())), baseAstResources);
  }

  @Override
  public ArtifactAst parse(URL... appXmlConfigUrls) {
    return parse("artifact", appXmlConfigUrls);
  }

  private ArtifactAst aggregateAsts(final List<ArtifactAst> parsedAstsList) {
    if (parsedAstsList.size() == 1) {
      return parsedAstsList.get(0);
    } else {
      return new AggregatedMultiConfigsArtifactAst(parentArtifact, parsedAstsList);
    }
  }

  private ArtifactAst resolveImports(ArtifactAst base, List<ConfigResource> baseAstResources) {
    final Map<ConfigResource, ArtifactAst> importedConfigs = new LinkedHashMap<>();
    baseAstResources.forEach(r -> importedConfigs.put(r, emptyArtifact()));
    return resolveImports(base, importedConfigs);
  }

  protected ArtifactAst resolveImports(ArtifactAst base, final Map<ConfigResource, ArtifactAst> importedConfigs) {
    Collection<ImportedResource> importedResources = new ArrayList<>();
    resolveImports(base, importedConfigs, importedResources, emptyList());

    if (importedResources.isEmpty()) {
      return base;
    } else {
      return new AggregatedImportsArtifactAst(parentArtifact, importedResources, importedConfigs.values(), base);
    }
  }

  private ArtifactAst doParse(String artifactName, ConfigResource resource) {
    return doParse(artifactName, resource, emptyList());
  }

  private ArtifactAst doParse(String artifactName, ConfigResource resource, List<ImportedResource> importChain) {
    try (InputStream inputStream = resource.getInputStream()) {
      Document document = xmlConfigurationDocumentLoader.loadDocument(() -> SAX_PARSER_FACTORY,
                                                                      entityResolver,
                                                                      resource.getResourceName(),
                                                                      inputStream,
                                                                      grammarPool);

      return doParse(artifactName, document, resource.getResourceName(), resource.getUrl(), importChain);
    } catch (IOException e) {
      throw new MuleRuntimeException(e);
    }
  }

  protected ArtifactAst doParse(String artifactName, Document document,
                                String resourceName, URL resourceUrl,
                                List<ImportedResource> importChain) {
    final ArtifactAstBuilder astBuilder = ArtifactAstBuilder.builder(artifactName, artifactType,
                                                                     extensionModels, ofNullable(parentArtifact),
                                                                     propertyResolver::resolveProperty);

    final Element rootElement = document.getDocumentElement();

    if (isXmlSdkModule(rootElement)) {
      // XML SDK 1 needs to access the parameter on the root component
      CONFIG_READER.processAttributes(astBuilder, rootElement);

      try {
        CONFIG_READER.extractComponentDefinitionModel(xmlApplicationParser, rootElement, resourceName, resourceUrl, importChain,
                                                      astBuilder::addTopLevelComponent, astBuilder::createMetadataBuilder);
      } catch (Exception e) {
        throw new MuleRuntimeException(createStaticMessage("Exception extracting definition model for component {}.",
                                                           rootElement.getTagName()),
                                       e);
      }
    } else {
      NodeList children = rootElement.getChildNodes();
      for (int i = 0; i < children.getLength(); i++) {
        Node child = children.item(i);
        if (child instanceof Element) {
          CONFIG_READER.processAttributes(astBuilder, (Element) child);
          if (IMPORT_IDENTIFIER.getName().equals(child.getNodeName())) {
            CONFIG_READER.extractImport(astBuilder, (Element) child, resourceName, resourceUrl, importChain,
                                        astBuilder::createMetadataBuilder);
          } else {
            try {
              CONFIG_READER.extractComponentDefinitionModel(xmlApplicationParser, (Element) child, resourceName, resourceUrl,
                                                            importChain,
                                                            astBuilder::addTopLevelComponent, astBuilder::createMetadataBuilder);
            } catch (Exception e) {
              throw new MuleRuntimeException(createStaticMessage("Exception extracting definition model for component {}.",
                                                                 ((Element) child).getTagName()),
                                             e);
            }
          }
        }
      }
    }

    return astBuilder.build();
  }

  private boolean isXmlSdkModule(Element rootElement) {
    // TODO W-12020311: once we have a specific ArtifactType for modules, we could just use that info
    final String identifier = xmlApplicationParser.parseIdentifier(rootElement);
    final String namespace = xmlApplicationParser.parseNamespace(rootElement);
    final String namespaceUri = xmlApplicationParser.parseNamespaceUri(rootElement);
    return namespace != null && namespace.equals(MODULE_DSL_NAMESPACE)
        && namespaceUri != null && namespaceUri.equals(MODULE_DSL_NAMESPACE_URI)
        && identifier != null && identifier.equals(MODULE_ROOT_NODE_NAME);
  }

  private void resolveImports(ArtifactAst parsedConfig,
                              Map<ConfigResource, ArtifactAst> alreadyResolvedConfigFiles,
                              Collection<ImportedResource> importedResources,
                              List<ImportedResource> importChain) {
    parsedConfig.getImportedResources().stream()
        .forEach(importedResource -> {
          String importFile;
          try {
            importFile = importedResource.getResourceLocation();
          } catch (PropertyNotFoundException pnfe) {
            importedResources.add(ImportedResourceBuilder.builder()
                .withResourceLocation(importedResource.getRawResourceLocation())
                .withMetadata(importedResource.getMetadata())
                .withResolutionFailure(format(COULD_NOT_RESOLVE_IMPORTED_RESOURCE + "'%s': %s",
                                              importedResource.getRawResourceLocation(), pnfe.getMessage()))
                .build());
            return;
          }

          URL importedResourceUrl = currentThread().getContextClassLoader().getResource(importFile);

          if (importedResourceUrl == null) {
            importedResources.add(ImportedResourceBuilder.builder()
                .withResourceLocation(importedResource.getResourceLocation())
                .withMetadata(importedResource.getMetadata())
                .withResolutionFailure(format(COULD_NOT_FIND_IMPORTED_RESOURCE + "'%s'", importFile))
                .build());
            return;
          }

          ConfigResource importedConfigResource;
          try {
            importedConfigResource = new ConfigResource(importFile, importedResourceUrl);
          } catch (IOException e) {
            // Still need the metadata of the import tag available
            importedResources.add(importedResource);
            return;
          }

          if (alreadyResolvedConfigFiles.containsKey(importedConfigResource)) {
            // Still need the metadata of the import tag available
            importedResources.add(importedResource);
            return;
          }

          LOGGER.debug("Resolving import {}", importedConfigResource.getResourceName());

          ImmutableList<ImportedResource> updatedImportChain = ImmutableList.<ImportedResource>builder()
              .addAll(importChain)
              .add(importedResource).build();
          ArtifactAst parsedImportedConfig;
          try {
            parsedImportedConfig = doParse(parsedConfig.getArtifactName(), importedConfigResource, updatedImportChain);
          } catch (Exception e) {
            importedResources.add(ImportedResourceBuilder.builder()
                .withResourceLocation(importedResource.getResourceLocation())
                .withMetadata(importedResource.getMetadata())
                .withResolutionFailure(e.getMessage())
                .build());
            return;
          }
          importedResources.add(importedResource);
          alreadyResolvedConfigFiles.put(importedConfigResource, parsedImportedConfig);
          resolveImports(parsedImportedConfig, alreadyResolvedConfigFiles, importedResources, updatedImportChain);
        });
  }

  private ConfigResource uriToConfigResource(URI appXmlConfig) {
    final URL appXmlConfigUrl;
    try {
      appXmlConfigUrl = appXmlConfig.toURL();
    } catch (MalformedURLException e) {
      throw new MuleRuntimeException(e);
    }

    return urlToConfigResource(appXmlConfigUrl);
  }

  private ConfigResource urlToConfigResource(URL appXmlConfigUrl) {
    final ConfigResource configResource = new ConfigResource(appXmlConfigUrl);

    // Try to just initialize the inputStream here so if there is any issue there it is caught in this early phase.
    try {
      configResource.getInputStream();
    } catch (IOException e) {
      throw new MuleRuntimeException(e);
    }

    return configResource;
  }

  private ConfigResource inputStreamToConfigResource(String resourceName, InputStream appXmlConfigInputStream) {
    return new ConfigResource(resourceName, appXmlConfigInputStream);
  }
}
