/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.xml.reader;

import static org.mule.runtime.api.component.Component.NS_MULE_DOCUMENTATION;
import static org.mule.runtime.api.component.Component.NS_MULE_PARSER_METADATA;
import static org.mule.runtime.api.component.Component.Annotations.NAME_ANNOTATION_KEY;
import static org.mule.runtime.api.component.ComponentIdentifier.builder;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.ast.api.xml.AstXmlParser.ANNOTATIONS_IDENTIFIER;
import static org.mule.runtime.ast.api.xml.AstXmlParserAttribute.CLOSING_TAG_END_COLUMN;
import static org.mule.runtime.ast.api.xml.AstXmlParserAttribute.CLOSING_TAG_END_LINE;
import static org.mule.runtime.ast.api.xml.AstXmlParserAttribute.CLOSING_TAG_START_COLUMN;
import static org.mule.runtime.ast.api.xml.AstXmlParserAttribute.CLOSING_TAG_START_LINE;
import static org.mule.runtime.ast.api.xml.AstXmlParserAttribute.IS_CDATA;
import static org.mule.runtime.ast.api.xml.AstXmlParserAttribute.IS_SELF_CLOSING;
import static org.mule.runtime.ast.api.xml.AstXmlParserAttribute.OPENING_TAG_END_COLUMN;
import static org.mule.runtime.ast.api.xml.AstXmlParserAttribute.OPENING_TAG_END_LINE;
import static org.mule.runtime.ast.api.xml.AstXmlParserAttribute.OPENING_TAG_START_COLUMN;
import static org.mule.runtime.ast.api.xml.AstXmlParserAttribute.OPENING_TAG_START_LINE;
import static org.mule.runtime.dsl.api.xml.parser.XmlApplicationParser.DECLARED_PREFIX;
import static org.mule.runtime.dsl.api.xml.parser.XmlApplicationParser.isTextContent;
import static org.mule.runtime.dsl.internal.xml.parser.XmlMetadataAnnotations.METADATA_ANNOTATIONS_KEY;
import static org.mule.runtime.internal.dsl.DslConstants.CORE_NAMESPACE;
import static org.mule.runtime.internal.dsl.DslConstants.CORE_PREFIX;

import static java.lang.Boolean.TRUE;
import static java.lang.String.format;

import static org.apache.commons.lang3.StringUtils.isEmpty;
import static org.w3c.dom.Node.CDATA_SECTION_NODE;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.ast.api.ComponentMetadataAst;
import org.mule.runtime.ast.api.ImportedResource;
import org.mule.runtime.ast.api.builder.ArtifactAstBuilder;
import org.mule.runtime.ast.api.builder.ComponentAstBuilder;
import org.mule.runtime.ast.api.builder.ComponentMetadataAstBuilder;
import org.mule.runtime.ast.api.builder.NamespaceDefinitionBuilder;
import org.mule.runtime.ast.internal.builder.ImportedResourceBuilder;
import org.mule.runtime.dsl.api.xml.parser.XmlApplicationParser;
import org.mule.runtime.dsl.internal.xml.parser.XmlMetadataAnnotations;

import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.function.Supplier;

import javax.xml.namespace.QName;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * This code was moved from {@code mule} project.
 */
public class ComponentAstReader {

  private static final Logger LOGGER = LoggerFactory.getLogger(ComponentAstReader.class);

  private static final QName DECLARED_PREFIX_QNAME = QName.valueOf(DECLARED_PREFIX);
  private static final String SCHEMA_LOCATION = "xsi:schemaLocation";
  private static final String MODULE_PREFIX_ATTRIBUTE = "prefix";
  private static final String MODULE_NAMESPACE_ATTRIBUTE = "namespace";

  private static final ComponentIdentifier DESCRIPTION_IDENTIFIER = builder()
      .namespace(CORE_PREFIX)
      .namespaceUri(CORE_NAMESPACE)
      .name("description")
      .build();

  public void extractComponentDefinitionModel(XmlApplicationParser parser, Element element,
                                              String resourceName, URL resourceUrl, List<ImportedResource> importChain,
                                              Supplier<ComponentAstBuilder> childFactory,
                                              Supplier<ComponentMetadataAstBuilder> childMetadataFactory) {
    extractComponentDefinitionModel(parser, element, resourceName, resourceUrl, importChain, childFactory, childMetadataFactory,
                                    null);
  }

  public void extractComponentDefinitionModel(XmlApplicationParser parser, Element element,
                                              String resourceName, URL resourceUrl, List<ImportedResource> importChain,
                                              Supplier<ComponentAstBuilder> childFactory,
                                              Supplier<ComponentMetadataAstBuilder> childMetadataFactory,
                                              ComponentMetadataAstBuilder parentMetadata) {
    ComponentIdentifier elementIdentifier = buildIdentifier(parser, element);
    if (DESCRIPTION_IDENTIFIER.equals(elementIdentifier)) {
      if (parentMetadata != null) {
        parentMetadata.putDocAttribute("description", element.getTextContent());
      }
      return;
    }

    ComponentAstBuilder componentAstBuilder = childFactory.get();
    componentAstBuilder.withIdentifier(elementIdentifier);

    final ComponentMetadataAstBuilder metadata =
        extractMetadata(childMetadataFactory.get(), element, resourceName, resourceUrl, importChain);

    processAttributes(componentAstBuilder, element);
    processAnnotations(componentAstBuilder, element);
    processChildNodesRecursively(componentAstBuilder, metadata, childMetadataFactory, parser, element, resourceName, resourceUrl,
                                 importChain);

    componentAstBuilder.withMetadata(metadata.build());
  }

  private ComponentIdentifier buildIdentifier(XmlApplicationParser parser, Element element) {
    String namespace = parser.parseNamespace(element);
    String namespaceUri = parser.parseNamespaceUri(element);

    final ComponentIdentifier identifier = builder()
        .namespace(namespace == null ? CORE_PREFIX : namespace)
        .namespaceUri(namespaceUri == null ? CORE_NAMESPACE : namespaceUri)
        .name(parser.parseIdentifier(element))
        .build();
    return identifier;
  }

  private ComponentMetadataAstBuilder extractMetadata(ComponentMetadataAstBuilder metadataBuilder, Element element,
                                                      String resourceName, URL resourceUrl, List<ImportedResource> importChain) {
    metadataBuilder
        .setFileName(resourceName)
        .setImportChain(importChain);

    // extracts the URI from the configuration resource, only if it is available
    if (resourceUrl != null) {
      final URI fileUri;
      try {
        fileUri = resourceUrl.toURI();
      } catch (URISyntaxException e) {
        throw new MuleRuntimeException(createStaticMessage(format("File URI is not RFC 2396 compliant: %s",
                                                                  resourceUrl)),
                                       e);
      }
      metadataBuilder.setFileUri(fileUri);
    }

    XmlMetadataAnnotations userData = (XmlMetadataAnnotations) element.getUserData(METADATA_ANNOTATIONS_KEY);
    if (userData != null) {
      metadataBuilder.setStartLine(userData.getOpeningTagBoundaries().getStartLineNumber())
          .setEndLine(userData.getClosingTagBoundaries().getEndLineNumber())
          .setStartColumn(userData.getOpeningTagBoundaries().getStartColumnNumber())
          .setEndColumn(userData.getClosingTagBoundaries().getEndColumnNumber())
          .setSourceCode(userData.getElementString())
          .putParserAttribute(OPENING_TAG_START_LINE, userData.getOpeningTagBoundaries().getStartLineNumber())
          .putParserAttribute(OPENING_TAG_START_COLUMN, userData.getOpeningTagBoundaries().getStartColumnNumber())
          .putParserAttribute(OPENING_TAG_END_LINE, userData.getOpeningTagBoundaries().getEndLineNumber())
          .putParserAttribute(OPENING_TAG_END_COLUMN, userData.getOpeningTagBoundaries().getEndColumnNumber())
          .putParserAttribute(CLOSING_TAG_START_LINE, userData.getClosingTagBoundaries().getStartLineNumber())
          .putParserAttribute(CLOSING_TAG_START_COLUMN, userData.getClosingTagBoundaries().getStartColumnNumber())
          .putParserAttribute(CLOSING_TAG_END_LINE, userData.getClosingTagBoundaries().getEndLineNumber())
          .putParserAttribute(CLOSING_TAG_END_COLUMN, userData.getClosingTagBoundaries().getEndColumnNumber());

      if (userData.isSelfClosing()) {
        metadataBuilder.putParserAttribute(IS_SELF_CLOSING, true);
      }
    }

    Node nameAttribute = element.getAttributes()
        .getNamedItemNS(NAME_ANNOTATION_KEY.getNamespaceURI(), NAME_ANNOTATION_KEY.getLocalPart());
    if (nameAttribute != null) {
      addCustomAttribute(metadataBuilder, NAME_ANNOTATION_KEY, nameAttribute.getNodeValue());
    }
    if (element.getPrefix() != null) {
      addCustomAttribute(metadataBuilder, DECLARED_PREFIX_QNAME, element.getPrefix());
    }
    for (int i = 0; i < element.getAttributes().getLength(); i++) {
      Node attributeNode = element.getAttributes().item(i);
      if (attributeNode.getNamespaceURI() != null) {
        addCustomAttribute(metadataBuilder,
                           new QName(attributeNode.getNamespaceURI(), attributeNode.getLocalName()),
                           attributeNode.getNodeValue());
      }
    }

    return metadataBuilder;
  }

  private void processAttributes(ComponentAstBuilder componentAstBuilder, Element element) {
    processAttributes(element,
                      n -> isEmpty(n.getNamespaceURI()),
                      node -> componentAstBuilder.withRawParameter(node.getNodeName(), node.getNodeValue()));
  }

  private void processAnnotations(ComponentAstBuilder componentAstBuilder, Element element) {
    processAttributes(element,
                      n -> !isEmpty(n.getNamespaceURI()),
                      node -> componentAstBuilder
                          .withAnnotation(new QName(node.getNamespaceURI(), node.getLocalName()).toString(),
                                          node.getNodeValue()));
  }

  private Map<String, String> calculateSchemaLocations(String schLoc) {
    LOGGER.debug("calculateSchemaLocations: {}", schLoc);
    Map<String, String> schemaLocations = new HashMap<>();
    String[] pairs = schLoc.trim().split("\\s+");
    for (int i = 0; i < pairs.length; i = i + 2) {
      if (i + 1 < pairs.length) {
        schemaLocations.put(pairs[i], pairs[i + 1]);
      } else {
        LOGGER.warn("namespaceUri without location: {}", pairs[i]);
      }
    }
    return schemaLocations;
  }


  public void processAttributes(ArtifactAstBuilder astBuilder, Element element) {
    final NamespaceDefinitionBuilder namespaceDefinitionBuilder = NamespaceDefinitionBuilder.builder();
    processAttributes(element, n -> true, node -> {
      String name = node.getNodeName();
      String value = node.getNodeValue();

      if (name.equals(SCHEMA_LOCATION)) {
        calculateSchemaLocations(value).forEach((schLoc, loc) -> namespaceDefinitionBuilder.withSchemaLocation(schLoc, loc));
      } else if (name.equals(MODULE_NAMESPACE_ATTRIBUTE)) {
        namespaceDefinitionBuilder.withNamespace(value);
      } else if (name.equals(MODULE_PREFIX_ATTRIBUTE)) {
        namespaceDefinitionBuilder.withPrefix(value);
      } else {
        namespaceDefinitionBuilder.withUnresolvedNamespace(name, value);
      }
    });
    astBuilder.withNamespaceDefinition(namespaceDefinitionBuilder.build());
  }

  private void processAttributes(Element element, Predicate<Node> filter, Consumer<Node> consumer) {
    NamedNodeMap attributes = element.getAttributes();
    if (element.hasAttributes()) {
      for (int i = 0; i < attributes.getLength(); i++) {
        Node attribute = attributes.item(i);
        if (filter.test(attribute)) {
          consumer.accept(attribute);
        }
      }
    }
  }

  private void processChildNodesRecursively(ComponentAstBuilder componentAstBuilder, ComponentMetadataAstBuilder metadata,
                                            Supplier<ComponentMetadataAstBuilder> childMetadataFactory,
                                            XmlApplicationParser parser, Element element,
                                            String resourceName, URL resourceUrl,
                                            List<ImportedResource> importChain) {
    if (element.hasChildNodes()) {
      NodeList children = element.getChildNodes();
      for (int i = 0; i < children.getLength(); i++) {
        Node child = children.item(i);
        if (isTextContent(child)) {
          componentAstBuilder.withBodyParameter(child.getNodeValue());
          if (child.getNodeType() == CDATA_SECTION_NODE) {
            metadata.putParserAttribute(IS_CDATA, TRUE);
            break;
          }
        } else {
          if (child instanceof Element) {
            if (child.getNamespaceURI().equals(CORE_NAMESPACE)
                && child.getLocalName().endsWith(ANNOTATIONS_IDENTIFIER.getName())) {
              processNestedAnnotations((Element) child, metadata);
            } else {
              extractComponentDefinitionModel(parser, (Element) child, resourceName, resourceUrl, importChain,
                                              componentAstBuilder::addChildComponent, childMetadataFactory, metadata);
            }
          }
        }
      }
    }
  }

  private void processNestedAnnotations(Element element, ComponentMetadataAstBuilder metadata) {
    if (element.hasChildNodes()) {
      NodeList children = element.getChildNodes();
      for (int i = 0; i < children.getLength(); i++) {
        Node child = children.item(i);

        if (child instanceof Element) {
          addCustomAttribute(metadata, new QName(child.getNamespaceURI(), child.getLocalName(), child.getPrefix()),
                             child.getTextContent().trim());
        }
      }
    }
  }

  private void addCustomAttribute(final ComponentMetadataAstBuilder metadataBuilder, QName qname, Object value) {
    if (isEmpty(qname.getNamespaceURI()) || NS_MULE_PARSER_METADATA.equals(qname.getNamespaceURI())) {
      metadataBuilder.putParserAttribute(qname.getLocalPart(), value);
    } else {
      if (NS_MULE_DOCUMENTATION.equals(qname.getNamespaceURI())) {
        // This is added for compatibility,
        // since in previous versions the doc attributes were looked up without the namespace.
        metadataBuilder.putDocAttribute(qname.getLocalPart(), value.toString());
      }
    }
  }

  public void extractImport(ArtifactAstBuilder astBuilder, Element importNode,
                            String resourceName, URL resourceUrl, List<ImportedResource> importChain,
                            Supplier<ComponentMetadataAstBuilder> metadataFactory) {
    final ComponentMetadataAst metadata =
        extractMetadata(metadataFactory.get(), importNode, resourceName, resourceUrl, importChain).build();

    if (importNode.hasAttribute("file")) {
      astBuilder.withImportedResource(ImportedResourceBuilder.builder()
          .withResourceLocation(importNode.getAttribute("file"))
          .withMetadata(metadata)
          .build());
    } else {
      throw new MuleRuntimeException(createStaticMessage(format("<import> does not have a file attribute defined. At file '%s', at line %s",
                                                                metadata.getFileName(), metadata.getStartLine())));
    }
  }
}
