/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.xml.model;

import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static java.util.stream.Stream.concat;

import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ArtifactType;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ImportedResource;
import org.mule.runtime.ast.api.NamespaceDefinition;
import org.mule.runtime.ast.api.util.BaseArtifactAst;
import org.mule.runtime.ast.internal.error.CompositeErrorTypeRepository;

import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.UnaryOperator;
import java.util.stream.Stream;

public final class AggregatedImportsArtifactAst extends BaseArtifactAst {

  private final Collection<ImportedResource> importedResources;
  private final Collection<ArtifactAst> importedConfigs;
  private final ArtifactAst parentArtifact;
  private final ArtifactAst base;
  private final LazyValue<Set<ExtensionModel>> dependencies;
  private final LazyValue<ErrorTypeRepository> errorTypeRepo;

  public AggregatedImportsArtifactAst(ArtifactAst parentArtifact,
                                      Collection<ImportedResource> importedResources,
                                      Collection<ArtifactAst> importedConfigs, ArtifactAst base) {
    this.parentArtifact = parentArtifact;
    this.importedResources = importedResources;
    this.importedConfigs = importedConfigs;
    this.base = base;

    dependencies = new LazyValue<>(() -> concat(base.dependencies().stream(),
                                                importedConfigs.stream()
                                                    .flatMap(ast -> ast.dependencies().stream()))
                                                        .collect(toSet()));

    errorTypeRepo = new LazyValue<>(() -> {
      final List<ErrorTypeRepository> allErrorTypeRepos =
          importedConfigs.stream().map(ArtifactAst::getErrorTypeRepository).collect(toList());
      allErrorTypeRepos.add(0, base.getErrorTypeRepository());

      return new CompositeErrorTypeRepository(allErrorTypeRepos);
    });
  }

  @Override
  public String getArtifactName() {
    return base.getArtifactName();
  }

  @Override
  public ArtifactType getArtifactType() {
    return base.getArtifactType();
  }

  @Override
  public Set<ExtensionModel> dependencies() {
    return dependencies.get();
  }

  @Override
  public Optional<ArtifactAst> getParent() {
    return ofNullable(parentArtifact);
  }

  @Override
  public Stream<ComponentAst> topLevelComponentsStream() {
    return concat(base.topLevelComponentsStream(),
                  importedConfigs.stream().flatMap(ArtifactAst::topLevelComponentsStream));
  }

  @Override
  public List<ComponentAst> topLevelComponents() {
    return topLevelComponentsStream().collect(toList());
  }

  @Override
  public void updatePropertiesResolver(UnaryOperator<String> newPropertiesResolver) {
    base.updatePropertiesResolver(newPropertiesResolver);
    importedConfigs.forEach(ast -> ast.updatePropertiesResolver(newPropertiesResolver));
  }

  @Override
  public ErrorTypeRepository getErrorTypeRepository() {
    return errorTypeRepo.get();
  }

  @Override
  public NamespaceDefinition namespaceDefinition() {
    return base.namespaceDefinition();
  }

  @Override
  public Collection<ImportedResource> getImportedResources() {
    return importedResources;
  }
}
