/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.xml.model;

import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;

import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ArtifactType;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ImportedResource;
import org.mule.runtime.ast.api.util.BaseArtifactAst;
import org.mule.runtime.ast.internal.error.CompositeErrorTypeRepository;

import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.UnaryOperator;

public final class AggregatedMultiConfigsArtifactAst extends BaseArtifactAst {

  private final ArtifactAst parentArtifact;
  private final List<ArtifactAst> parsedAstsList;
  private final LazyValue<Set<ExtensionModel>> dependencies;
  private final LazyValue<ErrorTypeRepository> errorTypeRepo;
  private final LazyValue<List<ComponentAst>> topLevelComponents;
  private final LazyValue<List<ImportedResource>> importedResources;

  public AggregatedMultiConfigsArtifactAst(ArtifactAst parentArtifact, List<ArtifactAst> parsedAstsList) {
    this.parentArtifact = parentArtifact;
    this.parsedAstsList = parsedAstsList;

    dependencies = new LazyValue<>(() -> parsedAstsList.stream()
        .flatMap(ast -> ast.dependencies().stream())
        .collect(toSet()));

    errorTypeRepo = new LazyValue<>(() -> new CompositeErrorTypeRepository(parsedAstsList
        .stream()
        .map(ArtifactAst::getErrorTypeRepository)
        .collect(toList())));

    topLevelComponents = new LazyValue<>(() -> parsedAstsList
        .stream()
        .flatMap(ArtifactAst::topLevelComponentsStream)
        .collect(toList()));

    importedResources = new LazyValue<>(() -> parsedAstsList.stream()
        .flatMap(ast -> ast.getImportedResources().stream())
        .collect(toList()));
  }

  @Override
  public String getArtifactName() {
    return parsedAstsList.get(0).getArtifactName();
  }

  @Override
  public ArtifactType getArtifactType() {
    return parsedAstsList.get(0).getArtifactType();
  }

  @Override
  public Set<ExtensionModel> dependencies() {
    return dependencies.get();
  }

  @Override
  public Optional<ArtifactAst> getParent() {
    return ofNullable(parentArtifact);
  }

  @Override
  public List<ComponentAst> topLevelComponents() {
    return topLevelComponents.get();
  }

  @Override
  public void updatePropertiesResolver(UnaryOperator<String> newPropertiesResolver) {
    parsedAstsList.forEach(ast -> ast.updatePropertiesResolver(newPropertiesResolver));
  }

  @Override
  public ErrorTypeRepository getErrorTypeRepository() {
    return errorTypeRepo.get();
  }

  @Override
  public Collection<ImportedResource> getImportedResources() {
    return importedResources.get();
  }
}
