/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.xml.reader;

import static org.mule.runtime.ast.api.ArtifactType.APPLICATION;
import static org.mule.runtime.ast.api.ArtifactType.DOMAIN;
import static org.mule.runtime.extension.internal.dsl.xml.XmlDslConstants.MODULE_DSL_NAMESPACE;
import static org.mule.runtime.extension.internal.dsl.xml.XmlDslConstants.MODULE_DSL_NAMESPACE_URI;
import static org.mule.runtime.extension.internal.dsl.xml.XmlDslConstants.MODULE_ROOT_NODE_NAME;

import static java.lang.String.format;

import org.mule.runtime.ast.api.ArtifactType;
import org.mule.runtime.dsl.api.xml.parser.XmlApplicationParser;

import org.w3c.dom.Element;

/**
 * Models the root tag of a mule config.
 * 
 * @since 1.4
 */
public class ArtifactRootElement {

  /**
   * Format mask for the default location of a schema
   */
  private static final String DEFAULT_NAMESPACE_URI_MASK = "http://www.mulesoft.org/schema/mule/%s";
  private static final String DEFAULT_EE_NAMESPACE_URI_MASK = "http://www.mulesoft.org/schema/mule/ee/%s";

  /**
   * Namespace for Mule core elements
   */
  private static final String CORE_NAMESPACE = format(DEFAULT_NAMESPACE_URI_MASK, "core");

  /**
   * Namespace for Mule core domain elements
   */
  private static final String DOMAIN_NAMESPACE = format(DEFAULT_NAMESPACE_URI_MASK, "domain");

  /**
   * Namespace for Mule core domain elements
   */
  private static final String DOMAIN_NAMESPACE_EE = format(DEFAULT_EE_NAMESPACE_URI_MASK, "domain");

  public static ArtifactRootElement from(XmlApplicationParser parser, Element rootElement) {
    return new ArtifactRootElement(parser.parseIdentifier(rootElement),
                                   parser.parseNamespace(rootElement),
                                   parser.parseNamespaceUri(rootElement));

  }

  private final String identifier;
  private final String namespace;
  private final String namespaceUri;

  private ArtifactRootElement(String identifier, String namespace, String namespaceUri) {
    this.identifier = identifier;
    this.namespace = namespace;
    this.namespaceUri = namespaceUri;
  }

  public boolean isXmlSdkModule() {
    // TODO W-12020311: once we have a specific ArtifactType for modules, we could just use that info
    return namespace != null && namespace.equals(MODULE_DSL_NAMESPACE)
        && namespaceUri != null && namespaceUri.equals(MODULE_DSL_NAMESPACE_URI)
        && identifier != null && identifier.equals(MODULE_ROOT_NODE_NAME);
  }

  public ArtifactType resolveType() {
    if (namespaceUri != null && namespaceUri.equals(DOMAIN_NAMESPACE)
        && identifier != null && identifier.equals("mule-domain")) {
      return DOMAIN;
    } else if (namespaceUri != null && namespaceUri.equals(DOMAIN_NAMESPACE_EE)
        && identifier != null && identifier.equals("mule-domain")) {
      return DOMAIN;
    } else if (namespaceUri != null && namespaceUri.equals(CORE_NAMESPACE)
        && identifier != null && identifier.equals("mule")) {
      return APPLICATION;
    } else {
      return null;
    }
  }

}
