/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.test.internal.xml;

import static org.mule.runtime.ast.api.xml.AstXmlParser.builder;
import static org.mule.runtime.ast.test.AllureConstants.ArtifactAst.ARTIFACT_AST;
import static org.mule.runtime.ast.test.internal.xml.AllureXmlParserConstants.DslParsing.XML_DOM_PROCESSING;
import static org.mule.runtime.ast.test.internal.xml.AllureXmlParserConstants.DslParsing.XML_PARSING;
import static org.mule.runtime.dsl.api.xml.parser.XmlConfigurationDocumentLoader.noValidationDocumentLoader;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.nullValue;
import static org.junit.Assert.assertThrows;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.XmlDslModel;
import org.mule.runtime.api.meta.model.construct.ConstructModel;
import org.mule.runtime.api.util.Pair;
import org.mule.runtime.api.util.xmlsecurity.XMLSecureFactories;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ImportedResource;
import org.mule.runtime.ast.api.xml.AstXmlParser;
import org.mule.runtime.ast.test.internal.xml.resolver.CachingExtensionSchemaGeneratorTestCase.TestExtensionSchemaGenerator;
import org.mule.runtime.dsl.api.ConfigResource;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Field;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.w3c.dom.Document;
import org.xml.sax.helpers.DefaultHandler;

import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;

@Feature(ARTIFACT_AST)
@Story(XML_PARSING)
public class DefaultAstXmlParserTestCase {

  private static final XmlDslModel XML_DSL_MODEL = XmlDslModel.builder()
      .setPrefix("mule")
      .setNamespace("http://mockns")
      .build();

  private ClassLoader classLoader;
  private AstXmlParser parser;

  private final Map<String, String> properties = new HashMap<>();

  private final ExtensionModel mockedExtensionModel = mock(ExtensionModel.class);

  @BeforeClass
  public static void configureTestSchemaLoader()
      throws IllegalArgumentException, IllegalAccessException, NoSuchFieldException, SecurityException {
    final Field schemaGeneratorField = AstXmlParser.Builder.class.getDeclaredField("SCHEMA_GENERATOR");
    schemaGeneratorField.setAccessible(true);
    schemaGeneratorField.set(null, of(new TestExtensionSchemaGenerator()));
  }

  @Before
  public void before() {
    properties.clear();
    classLoader = DefaultAstXmlParserTestCase.class.getClassLoader();

    ConstructModel topLevelSimpleConstruct = mock(ConstructModel.class);
    when(topLevelSimpleConstruct.getName()).thenReturn("top-level-simple");

    ConstructModel topLevelSimpleOtherConstruct = mock(ConstructModel.class);
    when(topLevelSimpleOtherConstruct.getName()).thenReturn("top-level-simple-other");

    ConstructModel topLevelImportConstruct = mock(ConstructModel.class);
    when(topLevelImportConstruct.getName()).thenReturn("top-level-import");

    when(mockedExtensionModel.getName()).thenReturn("Mock Extension Model");
    when(mockedExtensionModel.getXmlDslModel()).thenReturn(XML_DSL_MODEL);
    when(mockedExtensionModel.getConstructModels())
        .thenReturn(asList(topLevelSimpleConstruct, topLevelImportConstruct, topLevelSimpleOtherConstruct));

    parser = builder()
        .withSchemaValidationsDisabled()
        .withExtensionModel(mockedExtensionModel)
        .withPropertyResolver(propertyKey -> properties.getOrDefault(propertyKey, propertyKey))
        .build();
  }

  @Test
  public void simple() {
    final ArtifactAst simpleAst = parser.parse(classLoader.getResource("simple.xml"));

    final List<String> components = simpleAst.recursiveStream().map(c -> c.getIdentifier().toString()).collect(toList());
    assertThat(components, contains("top-level-simple"));
  }

  @Test
  public void twoConfigs() {
    final ArtifactAst twoConfigsAst = parser.parse(classLoader.getResource("simple.xml"),
                                                   classLoader.getResource("simple-other.xml"));

    final List<String> components = twoConfigsAst.recursiveStream().map(c -> c.getIdentifier().toString()).collect(toList());
    assertThat(components, contains("top-level-simple", "top-level-simple-other"));

    final List<String> imports =
        twoConfigsAst.getImportedResources().stream().map(ImportedResource::getResourceLocation).collect(toList());
    assertThat(imports, is(empty()));
  }

  @Test
  public void noConfigs() {
    IllegalArgumentException thrown = assertThrows(IllegalArgumentException.class, () -> parser.parse(new URL[] {}));
    assertThat(thrown.getMessage(), is("At least one 'appXmlConfigUrl' must be provided"));
  }

  @Test
  @Story(XML_DOM_PROCESSING)
  public void simpleDocuments() {
    Document document = noValidationDocumentLoader().loadDocument(() -> XMLSecureFactories.createDefault().getSAXParserFactory(),
                                                                  new DefaultHandler(),
                                                                  "simple.xml",
                                                                  classLoader.getResourceAsStream("simple.xml"),
                                                                  null);

    final ArtifactAst simpleAst = parser.parseDocument(asList(new Pair<>("simple.xml", document)));

    final List<String> components = simpleAst.recursiveStream().map(c -> c.getIdentifier().toString()).collect(toList());
    assertThat(components, contains("top-level-simple"));
  }

  @Test
  @Story(XML_DOM_PROCESSING)
  public void twoConfigsDocuments() {
    Document simpleDocument =
        noValidationDocumentLoader().loadDocument(() -> XMLSecureFactories.createDefault().getSAXParserFactory(),
                                                  new DefaultHandler(),
                                                  "simple.xml",
                                                  classLoader.getResourceAsStream("simple.xml"),
                                                  null);
    Document otherDocument =
        noValidationDocumentLoader().loadDocument(() -> XMLSecureFactories.createDefault().getSAXParserFactory(),
                                                  new DefaultHandler(),
                                                  "simple-other.xml",
                                                  classLoader.getResourceAsStream("simple-other.xml"),
                                                  null);

    final ArtifactAst twoConfigsAst = parser.parseDocument(asList(new Pair<>("simple.xml", simpleDocument),
                                                                  new Pair<>("simple-other.xml", otherDocument)));

    final List<String> components = twoConfigsAst.recursiveStream().map(c -> c.getIdentifier().toString()).collect(toList());
    assertThat(components, contains("top-level-simple", "top-level-simple-other"));

    final List<String> imports =
        twoConfigsAst.getImportedResources().stream().map(ImportedResource::getResourceLocation).collect(toList());
    assertThat(imports, is(empty()));
  }

  @Test
  @Story(XML_DOM_PROCESSING)
  public void noDocuments() {
    IllegalArgumentException thrown = assertThrows(IllegalArgumentException.class, () -> parser.parseDocument(emptyList()));
    assertThat(thrown.getMessage(), is("At least one 'appXmlConfigInputStream' must be provided"));
  }

  @Test
  public void parseFromConfigResource() {
    final URL resourceURL = classLoader.getResource("simple.xml");
    assertThat(resourceURL, is(not(nullValue())));

    final ConfigResource configResource = new ConfigResource(resourceURL);
    final ArtifactAst simpleAst = parser.parse(configResource);

    final List<String> components = simpleAst.recursiveStream().map(c -> c.getIdentifier().toString()).collect(toList());
    assertThat(components, contains("top-level-simple"));
  }

  @Test
  public void parseFromNoConfigResource() {
    IllegalArgumentException thrown = assertThrows(IllegalArgumentException.class, () -> parser.parse(new ConfigResource[] {}));
    assertThat(thrown.getMessage(), is("At least one 'appXmlConfigResources' must be provided"));
  }

  @Test
  public void parseFromInputStreams() {
    final InputStream resourceStream = classLoader.getResourceAsStream("simple.xml");
    assertThat(resourceStream, is(not(nullValue())));

    final ArtifactAst simpleAst = parser.parse(singletonList(new Pair<>("simple.xml", resourceStream)));

    final List<String> components = simpleAst.recursiveStream().map(c -> c.getIdentifier().toString()).collect(toList());
    assertThat(components, contains("top-level-simple"));
  }

  @Test
  public void parseFromNoInputStreams() {
    IllegalArgumentException thrown = assertThrows(IllegalArgumentException.class, () -> parser.parse(emptyList()));
    assertThat(thrown.getMessage(), is("At least one 'appXmlConfigInputStream' must be provided"));
  }

  @Test
  @Issue("MULE-19534")
  public void legacyFailStrategy() {
    parser = builder()
        .withPropertyResolver(propertyKey -> properties.getOrDefault(propertyKey, propertyKey))
        .withLegacyFailStrategy()
        .build();

    MuleRuntimeException thrown = assertThrows(MuleRuntimeException.class,
                                               () -> parser.parse(classLoader.getResource("mule-config-schema-missing.xml")));
    assertThat(thrown.getMessage(), containsString("Invalid content was found"));
  }

  @Test
  @Issue("MULE-19534")
  public void newFailStrategy() {
    parser = builder()
        .withPropertyResolver(propertyKey -> properties.getOrDefault(propertyKey, propertyKey))
        .build();

    MuleRuntimeException thrown = assertThrows(MuleRuntimeException.class,
                                               () -> parser.parse(classLoader.getResource("mule-config-schema-missing.xml")));
    assertThat(thrown.getMessage(),
               containsString("Can't resolve http://www.mulesoft.org/schema/mule/invalid-namespace/current/invalid-schema.xsd, A dependency or plugin might be missing"));
  }

  @Test
  public void parseXmlWithUtf8Bom() {
    final ArtifactAst ast = parser.parse(classLoader.getResource("xml-with-utf8-bom.xml"));

    final List<String> components = ast.recursiveStream()
        .map(c -> c.getIdentifier().toString())
        .collect(toList());
    assertThat(components, contains("top-level-simple"));
  }

  @Test
  public void parseXmlWithUtf8BomFromInputStream() throws IOException {
    // Create InputStream with BOM prepended
    byte[] bomBytes = new byte[] {(byte) 0xEF, (byte) 0xBB, (byte) 0xBF};
    String xmlContent = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
        + "<mule xmlns=\"http://www.mulesoft.org/schema/mule/core\"\n"
        + "      xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"\n"
        + "      xsi:schemaLocation=\"\n"
        + "        http://www.mulesoft.org/schema/mule/core http://www.mulesoft.org/schema/mule/core/current/mule.xsd\">\n"
        + "    <top-level-simple/>\n"
        + "</mule>";
    byte[] xmlBytes = xmlContent.getBytes(StandardCharsets.UTF_8);

    ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
    outputStream.write(bomBytes);
    outputStream.write(xmlBytes);

    ByteArrayInputStream inputStream = new ByteArrayInputStream(outputStream.toByteArray());

    final ArtifactAst ast = parser.parse(singletonList(new Pair<>("test-with-bom.xml", inputStream)));

    assertThat(ast, is(not(nullValue())));
    final List<String> components = ast.recursiveStream()
        .map(c -> c.getIdentifier().toString())
        .collect(toList());
    assertThat(components, contains("top-level-simple"));
  }

  @Test
  public void parseMultipleXmlsWithMixedBomPresence() {
    // Verify parser handles mix of BOM/no-BOM files correctly
    final ArtifactAst ast = parser.parse(
                                         classLoader.getResource("simple.xml"), // No BOM
                                         classLoader.getResource("xml-with-utf8-bom.xml") // With BOM
    );

    final List<String> components = ast.recursiveStream()
        .map(c -> c.getIdentifier().toString())
        .collect(toList());
    assertThat(components, contains("top-level-simple", "top-level-simple"));
  }

  @Test
  @Story(XML_DOM_PROCESSING)
  public void parseDocumentWithUtf8Bom() {
    Document document = noValidationDocumentLoader().loadDocument(() -> XMLSecureFactories.createDefault().getSAXParserFactory(),
                                                                  new DefaultHandler(),
                                                                  "xml-with-utf8-bom.xml",
                                                                  classLoader.getResourceAsStream("xml-with-utf8-bom.xml"),
                                                                  null);

    final ArtifactAst ast = parser.parseDocument(asList(new Pair<>("xml-with-utf8-bom.xml", document)));

    final List<String> components = ast.recursiveStream()
        .map(c -> c.getIdentifier().toString())
        .collect(toList());
    assertThat(components, contains("top-level-simple"));
  }
}
