/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.api.util;

import static java.util.stream.Collectors.toSet;

import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.meta.model.stereotype.StereotypeModel;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentParameterAst;

import java.util.Collections;
import java.util.Optional;
import java.util.Set;
import java.util.Spliterator;
import java.util.stream.Stream;

/**
 * Set of utility methods for working with {@link ArtifactAst}s and {@link ComponentAst}s.
 *
 * @since 1.0
 */
public final class MuleAstUtils {

  private static final ArtifactAst EMPTY_ARTIFACT = new ArtifactAst() {

    @Override
    public Stream<ComponentAst> topLevelComponentsStream() {
      return Collections.<ComponentAst>emptySet().stream();
    }

    @Override
    public Spliterator<ComponentAst> topLevelComponentsSpliterator() {
      return Collections.<ComponentAst>emptySet().spliterator();
    }

    @Override
    public Stream<ComponentAst> recursiveStream() {
      return Collections.<ComponentAst>emptySet().stream();
    }

    @Override
    public Spliterator<ComponentAst> recursiveSpliterator() {
      return Collections.<ComponentAst>emptySet().spliterator();
    }
  };

  private MuleAstUtils() {
    // Nothing to do
  }

  /**
   * Given an {@link ArtifactAst}, return all those components within it that are not accessible from a top-level component.
   *
   * @param appModel the model to determine orphan components from.
   * @return components within the provided {@code appModel} that are not accessible from a top-level component.
   */
  public static Set<ComponentAst> resolveOrphanComponents(ArtifactAst appModel) {
    final Set<ComponentAst> notOrphanComponents = appModel.topLevelComponentsStream()
        .flatMap(ComponentAst::recursiveStream)
        .collect(toSet());
    notOrphanComponents.addAll(appModel.recursiveStream()
        .filter(cm -> !notOrphanComponents.contains(cm))
        .flatMap(ComponentAst::directChildrenStream)
        .collect(toSet()));

    return appModel.recursiveStream()
        .filter(cm -> !notOrphanComponents.contains(cm))
        .collect(toSet());
  }

  /**
   * Provides a singleton empty {@link ArtifactAst} instance.
   *
   * @return an {@link ArtifactAst} without components.
   */
  public static ArtifactAst emptyArtifact() {
    return EMPTY_ARTIFACT;
  }

  /**
   * Given a {@link ComponentAst}, lookup for a parameter that references an object with a certain stereotype.
   *
   * @param referent the {@link ComponentAst} that will be checked for the reference.
   * @param referenceStereotype the {@link StereotypeModel} to check for.
   * @return a {@link ComponentParameterAst} representing a parameter of {@code referent} that references an object assignable to
   *         {@code referenceStereotype}, if it exists.
   */
  public static Optional<ComponentParameterAst> parameterOfType(ComponentAst referent, StereotypeModel referenceStereotype) {
    return referent.getModel(ParameterizedModel.class)
        .flatMap(parameterized -> parameterized.getAllParameterModels()
            .stream()
            .filter(paramModel -> paramModel.getAllowedStereotypes()
                .stream()
                .anyMatch(allowed -> allowed.isAssignableTo(referenceStereotype)))
            .map(paramModel -> referent.getParameter(paramModel.getName()))
            .findAny());
  }
}
