/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.api.util;

import static java.util.Collections.singletonList;
import static java.util.Optional.of;
import static org.hamcrest.CoreMatchers.sameInstance;
import static org.hamcrest.core.Is.is;
import static org.hamcrest.core.IsCollectionContaining.hasItem;
import static org.hamcrest.core.IsNot.not;
import static org.junit.Assert.assertThat;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static org.mule.runtime.ast.api.util.MuleAstUtils.parameterOfType;
import static org.mule.runtime.ast.api.util.MuleAstUtils.resolveOrphanComponents;

import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.meta.model.stereotype.ImmutableStereotypeModel;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentParameterAst;

import java.util.Set;
import java.util.stream.Stream;

import org.junit.Test;

public class MuleAstUtilsTestCase {

  @Test
  public void orphanComponentsTopLevelWithChildren() {
    final ArtifactAst artifact = mock(ArtifactAst.class);

    final ComponentAst processor = mock(ComponentAst.class);
    when(processor.directChildrenStream()).thenAnswer(inv -> Stream.empty());
    when(processor.recursiveStream()).thenAnswer(inv -> Stream.of(processor));

    final ComponentAst topLevelComponent = mock(ComponentAst.class);
    when(topLevelComponent.directChildrenStream()).thenAnswer(inv -> Stream.of(processor));
    when(topLevelComponent.recursiveStream()).thenAnswer(inv -> Stream.of(topLevelComponent, processor));

    when(artifact.topLevelComponentsStream()).thenAnswer(inv -> Stream.of(topLevelComponent));
    when(artifact.recursiveStream()).thenAnswer(inv -> Stream.of(topLevelComponent, processor));

    final Set<ComponentAst> orphanComponents = resolveOrphanComponents(artifact);

    assertThat(orphanComponents, not(hasItem(topLevelComponent)));
    assertThat(orphanComponents, not(hasItem(processor)));
  }

  @Test
  public void orphanComponentsSingleProcessor() {
    final ArtifactAst artifact = mock(ArtifactAst.class);

    final ComponentAst orphanProcessor = mock(ComponentAst.class);
    when(orphanProcessor.directChildrenStream()).thenAnswer(inv -> Stream.empty());
    when(orphanProcessor.recursiveStream()).thenAnswer(inv -> Stream.of(orphanProcessor));

    when(artifact.topLevelComponentsStream()).thenAnswer(inv -> Stream.empty());
    when(artifact.recursiveStream()).thenAnswer(inv -> Stream.of(orphanProcessor));

    final Set<ComponentAst> orphanComponents = resolveOrphanComponents(artifact);

    assertThat(orphanComponents, hasItem(orphanProcessor));
  }

  @Test
  public void orphanComponentsNestedProcessor() {
    final ArtifactAst artifact = mock(ArtifactAst.class);

    final ComponentAst nestedProcessor = mock(ComponentAst.class);
    when(nestedProcessor.directChildrenStream()).thenAnswer(inv -> Stream.empty());
    when(nestedProcessor.recursiveStream()).thenAnswer(inv -> Stream.of(nestedProcessor));

    final ComponentAst orphanRouter = mock(ComponentAst.class);
    when(orphanRouter.directChildrenStream()).thenAnswer(inv -> Stream.of(nestedProcessor));
    when(orphanRouter.recursiveStream()).thenAnswer(inv -> Stream.of(orphanRouter, nestedProcessor));

    when(artifact.topLevelComponentsStream()).thenAnswer(inv -> Stream.empty());
    when(artifact.recursiveStream()).thenAnswer(inv -> Stream.of(nestedProcessor, orphanRouter));

    final Set<ComponentAst> orphanComponents = resolveOrphanComponents(artifact);

    assertThat(orphanComponents, hasItem(orphanRouter));
    assertThat(orphanComponents, not(hasItem(nestedProcessor)));
  }

  @Test
  public void parameterOfTypeSameStereotype() {
    final ImmutableStereotypeModel myOwnStereotype = new ImmutableStereotypeModel("MY_OWN", "MINE", null);

    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes()).thenReturn(singletonList(myOwnStereotype));
    when(parameterModel.getName()).thenReturn("myOwnParam");

    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    when(parameterizedModel.getAllParameterModels()).thenReturn(singletonList(parameterModel));

    final ComponentParameterAst parameter = mock(ComponentParameterAst.class);

    final ComponentAst comp = mock(ComponentAst.class);
    when(comp.getModel(ParameterizedModel.class)).thenReturn(of(parameterizedModel));
    when(comp.getParameter("myOwnParam")).thenReturn(parameter);

    assertThat(parameterOfType(comp, myOwnStereotype).get(), sameInstance(parameter));
  }

  @Test
  public void parameterOfTypeParentStereotype() {
    final ImmutableStereotypeModel myOwnParentStereotype = new ImmutableStereotypeModel("MY_OWN", "MINE", null);
    final ImmutableStereotypeModel myOwnStereotype = new ImmutableStereotypeModel("MY_PRECIOUS", "MINE", myOwnParentStereotype);

    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes()).thenReturn(singletonList(myOwnStereotype));
    when(parameterModel.getName()).thenReturn("myOwnParam");

    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    when(parameterizedModel.getAllParameterModels()).thenReturn(singletonList(parameterModel));

    final ComponentParameterAst parameter = mock(ComponentParameterAst.class);

    final ComponentAst comp = mock(ComponentAst.class);
    when(comp.getModel(ParameterizedModel.class)).thenReturn(of(parameterizedModel));
    when(comp.getParameter("myOwnParam")).thenReturn(parameter);

    assertThat(parameterOfType(comp, myOwnParentStereotype).get(), sameInstance(parameter));
  }

  @Test
  public void parameterOfTypeNotMatchingStereotype() {
    final ImmutableStereotypeModel myOwnStereotype = new ImmutableStereotypeModel("MY_PRECIOUS", "MINE", null);
    final ImmutableStereotypeModel myOtherStereotype = new ImmutableStereotypeModel("A_SWORD", "MINE", null);

    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes()).thenReturn(singletonList(myOwnStereotype));
    when(parameterModel.getName()).thenReturn("myOwnParam");

    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    when(parameterizedModel.getAllParameterModels()).thenReturn(singletonList(parameterModel));

    final ComponentParameterAst parameter = mock(ComponentParameterAst.class);

    final ComponentAst comp = mock(ComponentAst.class);
    when(comp.getModel(ParameterizedModel.class)).thenReturn(of(parameterizedModel));
    when(comp.getParameter("myOwnParam")).thenReturn(parameter);

    assertThat(parameterOfType(comp, myOtherStereotype).isPresent(), is(false));
  }
}
