/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.api;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.meta.model.ComponentModel;
import org.mule.runtime.api.meta.model.config.ConfigurationModel;
import org.mule.runtime.api.meta.model.connection.ConnectionProviderModel;

import java.util.Collection;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.Spliterator;
import java.util.stream.Stream;

/**
 * Represents the user configuration of a component (flow, config, message processor, etc) defined in an artifact configuration
 * file.
 * <p>
 * Every {@code ComponentAst} represents the configuration of a core configuration or an extension configuration. Which
 * configuration element this object represents is identified by a {@link ComponentIdentifier} that can be retrieved using
 * {@code #getName()}.
 *
 * @since 1.0
 */
public interface ComponentAst {

  /**
   * @deprecated this will be removed for 1.0.0.
   */
  @Deprecated
  public static final String BODY_RAW_PARAM_NAME = "$%body%$";

  /**
   *
   * @return the identifier for the configuration element this object represents.
   */
  ComponentIdentifier getIdentifier();

  /**
   *
   * @return a general typification of the role of this component.
   */
  ComponentType getComponentType();

  /**
   *
   * @return the location of the component in the configuration.
   */
  ComponentLocation getLocation();

  /**
   *
   * @return the parser metadata for this component.
   */
  ComponentMetadataAst getMetadata();

  /**
   *
   * @return the id of this component if it has one defined, or {@link Optional#empty() empty} otherwise.
   */
  Optional<String> getComponentId();

  /**
   * A component may be represented by different kinds of models, depending on the actual type of this component. Possible values
   * may be, for instance, instances of {@link ConfigurationModel}, {@link ComponentModel} or {@link ConnectionProviderModel} (but
   * not necessarily limited to those).
   *
   * @param modelClass the class of the model this method should return.
   * @return the model that represents this component.
   */
  <M> Optional<M> getModel(Class<M> modelClass);

  /**
   * @param paramName the name of the parameter to get the raw value for.
   * @return the value of the parameter as defined in the DSL if present, or {@link Optional#empty()} if not present.
   *
   * @deprecated Use {@link #getParameter(String)} instead. This will removed before 1.0.0.
   */
  @Deprecated
  Optional<String> getRawParameterValue(String paramName);

  /**
   * @param paramName the name of the parameter to get AST for.
   * @return the AST of the parameter.
   * @throws NoSuchElementException if there is no parameter with the given {@code paramName}.
   */
  ComponentParameterAst getParameter(String paramName);

  /**
   * A parameter may have a value either because it is explicitly set in the DSL, or because the extension model defines a default
   * value for it.
   *
   * @return the ASTs of all the parameters that have a value.
   */
  Collection<ComponentParameterAst> getParameters();

  /**
   *
   * @return a {@link Stream} for this component and its children recursively that navigates the whole AST.
   */
  Stream<ComponentAst> recursiveStream();

  /**
   *
   * @return a {@link Spliterator} for this component and its children recursively that navigates the whole AST.
   */
  Spliterator<ComponentAst> recursiveSpliterator();

  /**
   *
   * @return a {@link Stream} for this component and its children recursively that navigates the whole AST.
   */
  Stream<ComponentAst> directChildrenStream();

  /**
   *
   * @return a {@link Spliterator} for this component and its children recursively that navigates the whole AST.
   */
  Spliterator<ComponentAst> directChildrenSpliterator();

}
