/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal;

import static java.util.Collections.unmodifiableMap;

import org.mule.runtime.ast.api.ComponentMetadataAst;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalInt;

public class DefaultComponentMetadataAst implements ComponentMetadataAst {

  private final Optional<String> fileName;

  private final OptionalInt startLine;
  private final OptionalInt startColumn;
  private final OptionalInt endLine;
  private final OptionalInt endColumn;

  private final Optional<String> sourceCode;

  private final Map<String, String> docAttributes;

  private final Map<String, Object> syntheticAttributes;

  private DefaultComponentMetadataAst(Optional<String> fileName,
                                      OptionalInt startLine,
                                      OptionalInt startColumn,
                                      OptionalInt endLine,
                                      OptionalInt endColumn,
                                      Optional<String> sourceCode,
                                      Map<String, String> docAttributes,
                                      Map<String, Object> syntheticAttributes) {
    this.fileName = fileName;
    this.startLine = startLine;
    this.startColumn = startColumn;
    this.endLine = endLine;
    this.endColumn = endColumn;
    this.sourceCode = sourceCode;
    this.docAttributes = docAttributes;
    this.syntheticAttributes = syntheticAttributes;
  }

  @Override
  public Optional<String> getFileName() {
    return fileName;
  }

  @Override
  public OptionalInt getStartLine() {
    return startLine;
  }

  @Override
  public OptionalInt getStartColumn() {
    return startColumn;
  }

  @Override
  public OptionalInt getEndLine() {
    return endLine;
  }

  @Override
  public OptionalInt getEndColumn() {
    return endColumn;
  }

  @Override
  public Optional<String> getSourceCode() {
    return sourceCode;
  }

  @Override
  public Map<String, String> getDocAttributes() {
    return docAttributes;
  }

  @Override
  public Map<String, Object> getParserAttributes() {
    return syntheticAttributes;
  }

  public static class Builder implements org.mule.runtime.ast.api.ComponentMetadataAst.Builder {

    private Optional<String> fileName = Optional.empty();

    private OptionalInt startLine = OptionalInt.empty();
    private OptionalInt startColumn = OptionalInt.empty();
    private OptionalInt endLine = OptionalInt.empty();
    private OptionalInt endColumn = OptionalInt.empty();

    private Optional<String> sourceCode = Optional.empty();

    private final Map<String, String> docAttributes = new HashMap<>();

    private final Map<String, Object> parserAttributes = new HashMap<>();

    @Override
    public Builder setFileName(String fileName) {
      this.fileName = Optional.of(fileName);
      return this;
    }

    @Override
    public Builder setStartLine(int startLine) {
      this.startLine = OptionalInt.of(startLine);
      return this;
    }

    @Override
    public Builder setStartColumn(int startColumn) {
      this.startColumn = OptionalInt.of(startColumn);
      return this;
    }

    @Override
    public Builder setEndLine(int endLine) {
      this.endLine = OptionalInt.of(endLine);
      return this;
    }

    @Override
    public Builder setEndColumn(int endColumn) {
      this.endColumn = OptionalInt.of(endColumn);
      return this;
    }

    @Override
    public Builder setSourceCode(String sourceCode) {
      this.sourceCode = Optional.of(sourceCode);
      return this;
    }

    @Override
    public Builder putDocAttribute(String key, String value) {
      this.docAttributes.put(key, value);
      return this;
    }

    @Override
    public Builder putParserAttribute(String key, Object value) {
      this.parserAttributes.put(key, value);
      return this;
    }

    @Override
    public ComponentMetadataAst build() {
      return new DefaultComponentMetadataAst(fileName, startLine, startColumn, endLine, endColumn, sourceCode,
                                             unmodifiableMap(docAttributes), unmodifiableMap(parserAttributes));
    }
  }
}
