/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal;

import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toList;
import static org.mule.runtime.ast.api.DependencyResolutionMode.getDependencyResolutionMode;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ImportedResource;
import org.mule.runtime.ast.api.NamespaceDefinition;
import org.mule.runtime.ast.api.util.BaseArtifactAst;
import org.mule.runtime.ast.internal.builder.PropertiesResolver;

import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.Supplier;
import java.util.function.UnaryOperator;
import java.util.stream.Collectors;

public final class DefaultArtifactAst extends BaseArtifactAst {

  private final List<ComponentAst> topLevelComponents;
  private final Supplier<Set<ExtensionModel>> dependenciesSupplier;
  private final ArtifactAst parentArtifact;
  private final PropertiesResolver propertiesResolver;
  private final NamespaceDefinition namespaceDefinition;
  private final List<ImportedResource> importedResources;

  public DefaultArtifactAst(List<ComponentAst> topLevelComponents,
                            Supplier<Set<ExtensionModel>> dependenciesSupplier,
                            ArtifactAst parentArtifact,
                            PropertiesResolver propertiesResolver,
                            NamespaceDefinition namespaceDefinition,
                            List<ImportedResource> importedResources) {
    this.topLevelComponents = topLevelComponents;
    this.dependenciesSupplier =
        new LazyValue<>(() -> getDependencyResolutionMode().resolveDependencies(dependenciesSupplier, this));
    this.parentArtifact = parentArtifact;
    this.propertiesResolver = propertiesResolver;
    this.namespaceDefinition = namespaceDefinition;
    this.importedResources = importedResources;
  }

  @Override
  public Set<ExtensionModel> dependencies() {
    return dependenciesSupplier.get();
  }

  @Override
  public Optional<ArtifactAst> getParent() {
    return ofNullable(parentArtifact);
  }

  @Override
  public List<ComponentAst> topLevelComponents() {
    return topLevelComponents;
  }

  @Override
  public void updatePropertiesResolver(UnaryOperator<String> newPropertiesResolver) {
    propertiesResolver.setMappingFunction(newPropertiesResolver);
  }

  @Override
  public NamespaceDefinition namespaceDefinition() {
    return namespaceDefinition;
  }

  @Override
  public Collection<ImportedResource> getImportedResources() {
    return importedResources;
  }
}
