/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.builder;

import static org.mule.metadata.api.utils.MetadataTypeUtils.getTypeId;
import static org.mule.runtime.api.component.ComponentIdentifier.builder;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.meta.ExpressionSupport.REQUIRED;
import static org.mule.runtime.api.meta.ExpressionSupport.SUPPORTED;
import static org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.DEFAULT_GROUP_NAME;
import static org.mule.runtime.api.meta.model.parameter.ParameterRole.BEHAVIOUR;
import static org.mule.runtime.ast.internal.builder.BaseComponentAstBuilder.BODY_RAW_PARAM_NAME;
import static org.mule.runtime.ast.internal.builder.adapter.MetadataTypeModelAdapter.createKeyValueWrapperTypeModelAdapter;
import static org.mule.runtime.ast.internal.builder.adapter.MetadataTypeModelAdapter.createMetadataTypeModelAdapterWithSterotype;
import static org.mule.runtime.ast.internal.builder.adapter.MetadataTypeModelAdapter.createParameterizedTypeModelAdapter;
import static org.mule.runtime.ast.internal.builder.adapter.MetadataTypeModelAdapter.createSimpleWrapperTypeModelAdapter;
import static org.mule.runtime.dsl.api.xml.parser.XmlApplicationParser.IS_CDATA;
import static org.mule.runtime.extension.api.util.ExtensionMetadataTypeUtils.getId;
import static org.mule.runtime.extension.api.util.ExtensionMetadataTypeUtils.isMap;
import static org.mule.runtime.extension.api.util.ExtensionModelUtils.getGroupAndParametersPairs;
import static org.mule.runtime.extension.api.util.ExtensionModelUtils.isContent;
import static org.mule.runtime.extension.api.util.NameUtils.getAliasName;
import static org.mule.runtime.extension.internal.loader.util.InfrastructureTypeMapping.getNameMap;
import static org.mule.runtime.extension.internal.loader.util.InfrastructureTypeMapping.getTypeFor;
import static org.mule.runtime.internal.dsl.DslConstants.KEY_ATTRIBUTE_NAME;
import static org.mule.runtime.internal.dsl.DslConstants.VALUE_ATTRIBUTE_NAME;

import static java.lang.Boolean.TRUE;
import static java.lang.String.format;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toList;

import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isEmpty;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.metadata.api.ClassTypeLoader;
import org.mule.metadata.api.model.ArrayType;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.model.SimpleType;
import org.mule.metadata.api.model.UnionType;
import org.mule.metadata.api.visitor.MetadataTypeVisitor;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.meta.NamedObject;
import org.mule.runtime.api.meta.model.config.ConfigurationModel;
import org.mule.runtime.api.meta.model.connection.ConnectionProviderModel;
import org.mule.runtime.api.meta.model.construct.ConstructModel;
import org.mule.runtime.api.meta.model.nested.NestableElementModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.api.meta.model.stereotype.HasStereotypeModel;
import org.mule.runtime.api.util.Pair;
import org.mule.runtime.ast.api.builder.ComponentAstBuilder;
import org.mule.runtime.ast.internal.DefaultComponentGenerationInformation;
import org.mule.runtime.ast.internal.DefaultComponentGenerationInformation.Builder;
import org.mule.runtime.ast.internal.DefaultComponentParameterAst;
import org.mule.runtime.ast.internal.builder.adapter.MetadataTypeModelAdapter;
import org.mule.runtime.ast.internal.model.ExtensionModelHelper;
import org.mule.runtime.ast.internal.model.ExtensionModelHelper.ExtensionWalkerModelDelegate;
import org.mule.runtime.extension.api.declaration.type.ExtensionsTypeLoaderFactory;
import org.mule.runtime.extension.api.dsl.syntax.DslElementSyntax;
import org.mule.runtime.extension.api.dsl.syntax.DslElementSyntaxBuilder;
import org.mule.runtime.extension.api.model.parameter.ImmutableParameterGroupModel;
import org.mule.runtime.extension.api.model.parameter.ImmutableParameterModel;
import org.mule.runtime.extension.api.property.NoWrapperModelProperty;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Predicate;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import org.slf4j.Logger;

/**
 * Provides utilities to obtain the models/types for the elements of a mule config.
 *
 * @since 1.0
 */
public final class ApplicationModelTypeUtils {

  private static final Logger LOGGER = getLogger(ApplicationModelTypeUtils.class);

  private static final ClassTypeLoader typeLoader = ExtensionsTypeLoaderFactory.getDefault().createTypeLoader();

  private ApplicationModelTypeUtils() {
    // Nothing to do
  }

  public static void resolveTypedComponentIdentifier(DefaultComponentAstBuilder componentAstBuilder,
                                                     boolean topLevel,
                                                     ExtensionModelHelper extensionModelHelper) {
    LOGGER.debug("resolveTypedComponentIdentifier for {}", componentAstBuilder);
    if (componentAstBuilder.getExtensionModel() != null) {
      extensionModelHelper.walkToComponent(componentAstBuilder.getIdentifier(), topLevel, new ExtensionWalkerModelDelegate() {

        @Override
        public void onConfiguration(ConfigurationModel model) {
          componentAstBuilder.withConfigurationModel(model);
          onParameterizedModel(componentAstBuilder, model, model.getParameterGroupModels(), extensionModelHelper);
        }

        @Override
        public void onConnectionProvider(ConnectionProviderModel model) {
          componentAstBuilder.withConnectionProviderModel(model);
          onParameterizedModel(componentAstBuilder, model, model.getParameterGroupModels(), extensionModelHelper);
        }

        @Override
        public void onOperation(OperationModel model) {
          componentAstBuilder.withComponentModel(model);
          onParameterizedModel(componentAstBuilder, model, model.getParameterGroupModels(), extensionModelHelper);
        }

        @Override
        public void onSource(SourceModel model) {
          componentAstBuilder.withComponentModel(model);

          List<ParameterGroupModel> sourceParamGroups = new ArrayList<>();

          sourceParamGroups.addAll(model.getParameterGroupModels());
          model.getSuccessCallback().ifPresent(scb -> sourceParamGroups.addAll(scb.getParameterGroupModels()));
          model.getErrorCallback().ifPresent(ecb -> sourceParamGroups.addAll(ecb.getParameterGroupModels()));

          onParameterizedModel(componentAstBuilder, model, sourceParamGroups, extensionModelHelper);
        }

        @Override
        public void onConstruct(ConstructModel model) {
          componentAstBuilder.withComponentModel(model);
          onParameterizedModel(componentAstBuilder, model, model.getParameterGroupModels(), extensionModelHelper);
        }

        @Override
        public void onNestableElement(NestableElementModel model) {
          componentAstBuilder.withNestableElementModel(model);
          onParameterizedModel(componentAstBuilder, model, model.getParameterGroupModels(), extensionModelHelper);
        }

        @Override
        public void onType(MetadataType type) {
          type.accept(new MetadataTypeVisitor() {

            @Override
            public void visitObject(ObjectType objectType) {
              final MetadataTypeModelAdapter model = createMetadataTypeModelAdapterWithSterotype(objectType, extensionModelHelper)
                  .orElseGet(() -> createParameterizedTypeModelAdapter(objectType, extensionModelHelper));
              componentAstBuilder.withParameterizedModel(model);
              onParameterizedModel(componentAstBuilder, model, model.getParameterGroupModels(), extensionModelHelper);
            }

            @Override
            public void visitArrayType(ArrayType arrayType) {
              arrayType.getType().accept(this);
            }

            @Override
            public void visitUnion(UnionType unionType) {
              unionType.getTypes().forEach(type -> type.accept(this));
            }
          });
        }

      }, componentAstBuilder.getExtensionModel());
    }

    // Check for infrastructure types, that are not present in an extension model
    if (!componentAstBuilder.getModel(HasStereotypeModel.class).isPresent()) {
      getTypeFor(componentAstBuilder.getIdentifier())
          .flatMap(extensionModelHelper::findMetadataType)
          .ifPresent(type -> {
            final MetadataTypeModelAdapter model = createMetadataTypeModelAdapterWithSterotype(type, extensionModelHelper)
                .orElseGet(() -> createParameterizedTypeModelAdapter(type, extensionModelHelper));
            componentAstBuilder.withParameterizedModel(model);
            extensionModelHelper.resolveDslElementModel(type, componentAstBuilder.getExtensionModel())
                .ifPresent(s -> componentAstBuilder.getGenerationInformation().withSyntax(s));
            onParameterizedModel(componentAstBuilder, model, model.getParameterGroupModels(), extensionModelHelper);
          });
    }

    componentAstBuilder.withComponentType(extensionModelHelper.findComponentType(componentAstBuilder.getIdentifier()));
  }

  private static void onParameterizedModel(DefaultComponentAstBuilder componentModel, NamedObject model,
                                           List<ParameterGroupModel> paramGroups,
                                           ExtensionModelHelper extensionModelHelper) {
    DslElementSyntax elementDsl;
    if (componentModel.getType() != null) {
      // Make sure the elementDsl is resolved consistently with previous implementations
      elementDsl =
          extensionModelHelper.resolveDslElementModel(componentModel.getType(), componentModel.getExtensionModel())
              .orElseGet(() -> extensionModelHelper.resolveDslElementModel(model, componentModel.getIdentifier()));
    } else {
      elementDsl = extensionModelHelper.resolveDslElementModel(model, componentModel.getIdentifier());
    }

    componentModel.getGenerationInformation().withSyntax(elementDsl);

    Multimap<ComponentIdentifier, DefaultComponentAstBuilder> nestedComponents = getNestedComponents(componentModel);

    List<Pair<ParameterModel, ParameterGroupModel>> inlineGroupedParameters = paramGroups
        .stream()
        .filter(ParameterGroupModel::isShowInDsl)
        .map(group -> addInlineGroup(componentModel, nestedComponents, group, extensionModelHelper))
        .flatMap(k -> k.getParameterModels().stream().map(v -> new Pair<>(v, k)))
        .collect(toList());

    final List<Pair<ParameterModel, ParameterGroupModel>> parameters = paramGroups.stream()
        .flatMap(k -> k.getParameterModels().stream().map(v -> new Pair<>(v, k)))
        .distinct()
        .collect(toList());

    parameters
        .stream()
        .filter(paramModel -> !inlineGroupedParameters.contains(paramModel))
        .forEach(paramModel -> elementDsl.getAttribute(paramModel.getFirst().getName())
            .ifPresent(attrDsl -> {
              if (setSimpleParameterValue(componentModel, componentModel, paramModel.getFirst(), paramModel.getSecond(),
                                          attrDsl, nestedComponents)) {
                // Avoid this param from being reprocessed by the handleNestedParameters below
                inlineGroupedParameters.add(paramModel);
              }
            }));

    handleNestedParameters(componentModel, nestedComponents, extensionModelHelper, parameters,
                           parameterModel -> !inlineGroupedParameters.contains(parameterModel));
  }

  private static ParameterGroupModel addInlineGroup(DefaultComponentAstBuilder componentModel,
                                                    Multimap<ComponentIdentifier, DefaultComponentAstBuilder> nestedComponents,
                                                    ParameterGroupModel group, ExtensionModelHelper extensionModelHelper) {
    componentModel.getGenerationInformation().getSyntax().getChild(group.getName())
        .ifPresent(groupDsl -> {
          Optional<ComponentIdentifier> groupIdentifier = getIdentifier(groupDsl);
          if (!groupIdentifier.isPresent()) {
            return;
          }

          DefaultComponentAstBuilder groupComponent = getSingleComponentModel(nestedComponents, groupIdentifier);
          if (groupComponent != null) {
            groupComponent.getGenerationInformation().withSyntax(groupDsl);
            List<Pair<ParameterModel, ParameterGroupModel>> inlineGroupedParameters = new ArrayList<>();

            group.getParameterModels()
                .stream()
                .filter(paramModel -> groupComponent.getRawParameters().containsKey(paramModel.getName()))
                .forEach(paramModel -> groupDsl.getAttribute(paramModel.getName())
                    .ifPresent(attr -> {
                      if (setSimpleParameterValue(componentModel, groupComponent, paramModel, group, attr,
                                                  nestedComponents)) {
                        // Avoid this param from being reprocessed by the handleNestedParameters below
                        inlineGroupedParameters.add(new Pair<>(paramModel, group));
                      }
                    }));

            handleNestedParameters(componentModel, getNestedComponents(groupComponent), extensionModelHelper,
                                   group.getParameterModels().stream().map(k -> new Pair<>(k, group)).collect(toList()),
                                   parameterModel -> !inlineGroupedParameters.contains(parameterModel));
          }
        });

    return group;
  }

  private static void handleNestedParameters(DefaultComponentAstBuilder componentModel,
                                             Multimap<ComponentIdentifier, DefaultComponentAstBuilder> nestedComponents,
                                             ExtensionModelHelper extensionModelHelper,
                                             List<Pair<ParameterModel, ParameterGroupModel>> parameters,
                                             Predicate<Pair<ParameterModel, ParameterGroupModel>> parameterModelFilter) {
    parameters
        .stream()
        .filter(parameterModelFilter)
        .filter(paramModel -> paramModel.getFirst().getDslConfiguration() != null
            && paramModel.getFirst().getDslConfiguration().allowsInlineDefinition())
        .forEach(paramModel -> {
          final DslElementSyntax paramSyntax =
              extensionModelHelper.resolveDslElementModel(paramModel.getFirst(), componentModel.getIdentifier());

          getIdentifier(paramSyntax)
              .ifPresent(id -> handleNestedExtensionParameter(componentModel, nestedComponents, extensionModelHelper,
                                                              paramModel, paramSyntax, nestedComponents.get(id)));
        });
  }

  private static void handleNestedExtensionParameter(DefaultComponentAstBuilder componentModel,
                                                     Multimap<ComponentIdentifier, DefaultComponentAstBuilder> nestedComponents,
                                                     ExtensionModelHelper extensionModelHelper,
                                                     Pair<ParameterModel, ParameterGroupModel> paramModel,
                                                     final DslElementSyntax paramSyntax,
                                                     final Collection<DefaultComponentAstBuilder> nestedForId) {
    paramModel.getFirst().getType().accept(new MetadataTypeVisitor() {

      @Override
      protected void defaultVisit(MetadataType metadataType) {
        if (isContent(paramModel.getFirst()) || paramModel.getFirst().getExpressionSupport() == REQUIRED) {
          // If this is a Map for example, this has to be set using
          // text content as this does not depend on the dsl.
          addParameterUsingTextContent(componentModel, nestedForId);
        } else {
          enrichComponentModels(componentModel, nestedComponents,
                                of(paramSyntax),
                                paramModel.getFirst(), paramModel.getSecond(), metadataType, true, extensionModelHelper);
        }
      }

      @Override
      public void visitUnion(UnionType unionType) {
        if (isContent(paramModel.getFirst()) || paramModel.getFirst().getExpressionSupport() == REQUIRED) {
          // Account for some oddity in XML SDK union type definitions.
          addParameterUsingTextContent(componentModel, nestedForId);
        } else {
          unionType.getTypes().forEach(type -> type.accept(this));
        }
      }

      @Override
      public void visitSimpleType(SimpleType stringType) {
        addParameterUsingTextContent(componentModel, nestedForId);
      }

      private void addParameterUsingTextContent(DefaultComponentAstBuilder componentModel,
                                                final Collection<DefaultComponentAstBuilder> nestedForId) {
        nestedForId
            .forEach(childComp -> {
              String body = childComp.getRawParameters().get(BODY_RAW_PARAM_NAME);

              body = body == null
                  // previous implementations were assuming that an empty string was the same as the param not being present...
                  // unless it was a CDATA, where an empty string is the actual param value :S
                  ? resolveNullBody(childComp)
                  : resolveNonNullBody(body, childComp);

              componentModel.withParameter(paramModel.getFirst(),
                                           paramModel.getSecond(),
                                           new DefaultComponentParameterAst(body,
                                                                            paramModel.getFirst(),
                                                                            paramModel.getSecond(),
                                                                            childComp.getMetadata(),
                                                                            DefaultComponentGenerationInformation
                                                                                .builder()
                                                                                .withSyntax(paramSyntax)
                                                                                .build(),
                                                                            componentModel.getPropertiesResolver()),
                                           of(childComp.getIdentifier()));
            });
      }

      private String resolveNullBody(DefaultComponentAstBuilder childComp) {
        if (isCdata(childComp)) {
          return "";
        } else {
          return null;
        }
      }

      private String resolveNonNullBody(String body, DefaultComponentAstBuilder childComp) {
        if (isCdata(childComp)) {
          return body;
        } else {
          return body.trim();
        }
      }

      protected boolean isCdata(DefaultComponentAstBuilder childComp) {
        return TRUE.equals(childComp.getMetadata().getParserAttributes().get(IS_CDATA));
      }
    });
  }

  private static Multimap<ComponentIdentifier, DefaultComponentAstBuilder> getNestedComponents(DefaultComponentAstBuilder componentModel) {
    Multimap<ComponentIdentifier, DefaultComponentAstBuilder> result = ArrayListMultimap.create();
    componentModel.childComponentsStream()
        .forEach(nestedComponent -> result.put(nestedComponent.getIdentifier(), nestedComponent));
    return result;
  }

  private static DefaultComponentAstBuilder getSingleComponentModel(Multimap<ComponentIdentifier, DefaultComponentAstBuilder> innerComponents,
                                                                    Optional<ComponentIdentifier> identifier) {
    return identifier.filter(innerComponents::containsKey)
        .map(innerComponents::get)
        .map(collection -> collection.iterator().next())
        .orElse(null);
  }

  private static void enrichComponentModels(DefaultComponentAstBuilder component,
                                            Multimap<ComponentIdentifier, DefaultComponentAstBuilder> innerComponents,
                                            Optional<DslElementSyntax> optionalParamDsl,
                                            ParameterModel paramModel, ParameterGroupModel parameterGroupModel,
                                            MetadataType paramConcreteType, ExtensionModelHelper extensionModelHelper) {
    enrichComponentModels(component, innerComponents, optionalParamDsl, paramModel, parameterGroupModel, paramConcreteType,
                          false, extensionModelHelper);
  }

  private static void enrichComponentModels(DefaultComponentAstBuilder componentModel,
                                            Multimap<ComponentIdentifier, DefaultComponentAstBuilder> innerComponents,
                                            Optional<DslElementSyntax> optionalParamDsl,
                                            ParameterModel paramModel, ParameterGroupModel parameterGroupModel,
                                            MetadataType paramConcreteType, boolean useConcreteTypeForUnion,
                                            ExtensionModelHelper extensionModelHelper) {
    if (!optionalParamDsl.isPresent()) {
      return;
    }

    DslElementSyntax paramDsl = optionalParamDsl.get();
    DefaultComponentAstBuilder paramComponent = null;

    if (!isNoWrapperParam(paramModel)) {
      paramComponent = getSingleComponentModel(innerComponents, getIdentifier(paramDsl));
      if (paramComponent == null && !isEmpty(paramDsl.getPrefix())) {
        paramDsl = extensionModelHelper.resolveDslElementModel(paramConcreteType, paramDsl.getPrefix()).orElse(paramDsl);
        if (paramDsl != null) {
          paramComponent = getSingleComponentModel(innerComponents, getIdentifier(paramDsl));
          if (paramComponent == null && componentModel.getIdentifier() != null) {
            paramComponent = getSingleComponentModel(innerComponents, getIdentifier(optionalParamDsl.get().getElementName(),
                                                                                    componentModel.getIdentifier()
                                                                                        .getNamespace()));
          }
        }
      }
    }

    if (paramDsl != null) {
      if (isNoWrapperParam(paramModel)) {
        paramModel.getType()
            .accept(getComponentChildVisitor(componentModel, paramModel,
                                             parameterGroupModel, paramDsl, componentModel, extensionModelHelper,
                                             innerComponents));
      } else if (paramComponent != null) {
        paramComponent.getGenerationInformation().withSyntax(paramDsl);
        if (paramDsl.isWrapped()) {
          handleWrappedElement(componentModel, paramComponent, paramModel, parameterGroupModel, paramConcreteType,
                               useConcreteTypeForUnion, paramDsl, extensionModelHelper);
        } else {
          paramModel.getType()
              .accept(getComponentChildVisitor(componentModel, paramModel,
                                               parameterGroupModel, paramDsl, paramComponent, extensionModelHelper,
                                               innerComponents));
        }
      } else {
        setSimpleParameterValue(componentModel, componentModel, paramModel, parameterGroupModel,
                                optionalParamDsl.get(),
                                innerComponents);
      }
    }
  }

  private static boolean isNoWrapperParam(ParameterModel paramModel) {
    return paramModel.getModelProperty(NoWrapperModelProperty.class).isPresent();
  }

  private static void handleWrappedElement(DefaultComponentAstBuilder componentModel, DefaultComponentAstBuilder wrappedComponent,
                                           ParameterModel paramModel, ParameterGroupModel parameterGroupModel,
                                           MetadataType paramConcreteType, boolean useConcreteTypeForUnion,
                                           DslElementSyntax paramDsl,
                                           ExtensionModelHelper extensionModelHelper) {
    Multimap<ComponentIdentifier, DefaultComponentAstBuilder> nestedWrappedComponents = getNestedComponents(wrappedComponent);

    paramModel.getType().accept(new MetadataTypeVisitor() {

      @Override
      public void visitObject(ObjectType objectType) {
        extensionModelHelper.resolveSubTypes((ObjectType) paramModel.getType()).entrySet()
            .stream()
            .filter(entry -> (entry.getValue().isPresent()
                && getSingleComponentModel(nestedWrappedComponents, getIdentifier(entry.getValue().get())) != null))
            .findFirst()
            .ifPresent(wrappedEntryType -> {
              DslElementSyntax wrappedDsl = wrappedEntryType.getValue().get();
              wrappedEntryType.getKey()
                  .accept(getComponentChildVisitor(componentModel,
                                                   paramModel,
                                                   parameterGroupModel,
                                                   wrappedDsl,
                                                   getSingleComponentModel(nestedWrappedComponents, getIdentifier(wrappedDsl)),
                                                   extensionModelHelper,
                                                   nestedWrappedComponents));
            });
      }

      @Override
      public void visitUnion(UnionType unionType) {
        if (useConcreteTypeForUnion) {
          doVisitUnionType(paramConcreteType);
        } else {
          unionType.getTypes().forEach(this::doVisitUnionType);
        }

      }

      protected void doVisitUnionType(MetadataType type) {
        getTypeId(type)
            .flatMap(paramDsl::getChild)
            .flatMap(ApplicationModelTypeUtils::getIdentifier)
            .flatMap(childComponentId -> ofNullable(nestedWrappedComponents.get(childComponentId)))
            .orElse(emptySet())
            .forEach(nestedChild -> type
                .accept(getComponentChildVisitor(componentModel, paramModel, parameterGroupModel, paramDsl, nestedChild,
                                                 extensionModelHelper, nestedWrappedComponents)));
      }
    });
  }

  private static boolean setSimpleParameterValue(DefaultComponentAstBuilder componentTo,
                                                 DefaultComponentAstBuilder componentFrom,
                                                 ParameterModel paramModel,
                                                 ParameterGroupModel paramGroupModel,
                                                 DslElementSyntax paramDsl,
                                                 Multimap<ComponentIdentifier, DefaultComponentAstBuilder> nestedComponents) {
    String value = paramDsl.supportsAttributeDeclaration()
        ? componentFrom.getRawParameters()
            .get(getInfrastructureParameterName(paramModel.getType()).orElse(paramModel.getName()))
        : null;

    Optional<ComponentIdentifier> paramIdentifier = empty();

    if (isBlank(value) && isContent(paramModel)) {
      value = componentFrom.getRawParameters().get(BODY_RAW_PARAM_NAME);
      paramIdentifier = of(componentFrom.getIdentifier());
    }

    if (isNotBlank(value)) {
      boolean trim = !(paramDsl.supportsAttributeDeclaration() && componentTo.disableTrimWhitespaces());
      if (trim) {
        value = value.trim();
      }
      DefaultComponentParameterAst paramValue = new DefaultComponentParameterAst(value,
                                                                                 paramModel,
                                                                                 paramGroupModel,
                                                                                 componentFrom.getMetadata(),
                                                                                 DefaultComponentGenerationInformation
                                                                                     .builder()
                                                                                     .withSyntax(paramDsl)
                                                                                     .build(),
                                                                                 componentFrom.getPropertiesResolver());

      Optional<ComponentIdentifier> childParamId = getIdentifier(paramDsl);
      // there is also a nested value for this param
      if (childParamId.map(paramId -> nestedComponents.containsKey(paramId)).orElse(false)) {
        if (paramValue.isDefaultValue()) {
          // If the attribute value was not explicitly set, continue and handle the nested element for this parameter.
          return false;
        } else {
          throw new MuleRuntimeException(createStaticMessage(format("[%s:%d]: Component '%s' has a child element '%s' which is used for the same purpose of the configuration parameter '%s'. Only one must be used.",
                                                                    componentTo.getMetadata().getFileName()
                                                                        .orElse("unknown"),
                                                                    componentTo.getMetadata().getStartLine().orElse(-1),
                                                                    componentFrom.getIdentifier(), childParamId.get(),
                                                                    paramModel.getName())));
        }
      }

      componentTo.withParameter(paramModel,
                                paramGroupModel,
                                paramValue,
                                paramIdentifier);
      return true;
    }

    return false;
  }

  private static MetadataTypeVisitor getComponentChildVisitor(DefaultComponentAstBuilder componentModel,
                                                              ParameterModel paramModel,
                                                              ParameterGroupModel parameterGroupModel,
                                                              DslElementSyntax paramDsl,
                                                              DefaultComponentAstBuilder paramComponent,
                                                              ExtensionModelHelper extensionModelHelper,
                                                              Multimap<ComponentIdentifier, DefaultComponentAstBuilder> nestedComponents) {
    return new MetadataTypeVisitor() {

      @Override
      public void visitSimpleType(SimpleType simpleType) {
        DefaultComponentAstBuilder paramComponentFrom = getSingleComponentModel(nestedComponents, getIdentifier(paramDsl));
        setSimpleParameterValue(componentModel, paramComponentFrom, paramModel, parameterGroupModel, paramDsl,
                                getNestedComponents(paramComponentFrom));
      }

      @Override
      public void visitArrayType(ArrayType arrayType) {
        MetadataType itemType = arrayType.getType();
        itemType.accept(getArrayItemTypeVisitor(componentModel, paramModel, parameterGroupModel, paramDsl, paramComponent,
                                                extensionModelHelper));
      }

      @Override
      public void visitObject(ObjectType objectType) {
        if (isMap(objectType)) {
          List<ComponentAstBuilder> componentModels =
              handleMapChild(paramDsl, paramComponent, parameterGroupModel, extensionModelHelper, objectType);

          componentModel.withParameter(paramModel,
                                       parameterGroupModel,
                                       new DefaultComponentParameterAst(componentModels,
                                                                        paramModel,
                                                                        parameterGroupModel,
                                                                        paramComponent.getMetadata(),
                                                                        DefaultComponentGenerationInformation
                                                                            .builder()
                                                                            .withSyntax(paramDsl)
                                                                            .build(),
                                                                        componentModel.getPropertiesResolver()),
                                       of(paramComponent.getIdentifier()));
          return;
        }

        componentModel.withParameter(paramModel,
                                     parameterGroupModel,
                                     new DefaultComponentParameterAst(paramComponent,
                                                                      paramModel,
                                                                      parameterGroupModel,
                                                                      paramComponent.getMetadata(),
                                                                      DefaultComponentGenerationInformation
                                                                          .builder()
                                                                          .withSyntax(paramDsl)
                                                                          .build(),
                                                                      componentModel.getPropertiesResolver()),
                                     getIdentifier(paramDsl));

        ParameterizedModel parameterizedModel =
            resolveParameterizedModel(extensionModelHelper, objectType, paramComponent.getIdentifier());
        paramComponent.withParameterizedModel(parameterizedModel);
        paramComponent.getGenerationInformation().withSyntax(paramDsl);

        final Multimap<ComponentIdentifier, DefaultComponentAstBuilder> nestedComponents = getNestedComponents(paramComponent);
        parameterizedModel.getParameterGroupModels()
            .forEach(nestedGroup -> nestedGroup.getParameterModels()
                .forEach(nestedParameter -> {
                  enrichComponentModels(paramComponent,
                                        nestedComponents,
                                        paramDsl.getContainedElement(nestedParameter.getName()),
                                        nestedParameter,
                                        nestedGroup,
                                        nestedParameter.getType(),
                                        extensionModelHelper);

                  // This is needed for reconnection strategies
                  nestedParameter.getType().accept(new MetadataTypeVisitor() {

                    @Override
                    public void visitUnion(UnionType unionType) {
                      unionType.getTypes()
                          .forEach(type -> enrichComponentModels(paramComponent,
                                                                 nestedComponents,
                                                                 extensionModelHelper.resolveDslElementModel(type,
                                                                                                             paramDsl
                                                                                                                 .getPrefix()),
                                                                 nestedParameter,
                                                                 nestedGroup,
                                                                 nestedParameter.getType(),
                                                                 extensionModelHelper));
                    }
                  });
                }));
      }

      @Override
      public void visitUnion(UnionType unionType) {
        componentModel.withParameter(paramModel,
                                     parameterGroupModel,
                                     new DefaultComponentParameterAst(paramComponent,
                                                                      paramModel,
                                                                      parameterGroupModel,
                                                                      paramComponent.getMetadata(),
                                                                      DefaultComponentGenerationInformation
                                                                          .builder()
                                                                          .withSyntax(paramDsl)
                                                                          .build(),
                                                                      componentModel.getPropertiesResolver()),
                                     of(paramComponent.getIdentifier()));

        final Multimap<ComponentIdentifier, DefaultComponentAstBuilder> nestedComponents = getNestedComponents(paramComponent);
        unionType.getTypes()
            .forEach(type -> {
              ParameterizedModel innerTypeParameterizedModel =
                  resolveParameterizedModel(extensionModelHelper, type, paramComponent.getIdentifier());

              if (paramDsl.isWrapped()) {
                paramDsl.getContainedElement(getAliasName(type))
                    .ifPresent(innerParamDsl -> {
                      ParameterizedModel parameterizedModel =
                          resolveParameterizedModel(extensionModelHelper, type, paramComponent.getIdentifier());
                      paramComponent.withParameterizedModel(parameterizedModel);

                      getGroupAndParametersPairs(innerTypeParameterizedModel)
                          .forEach(groupAndParam -> {
                            final Collection<DefaultComponentAstBuilder> children = getIdentifier(innerParamDsl)
                                .map(nestedComponents::get)
                                .orElse(emptyList());

                            children.forEach(nested -> {
                              nested.withParameterizedModel(innerTypeParameterizedModel);
                              enrichComponentModels(nested,
                                                    getNestedComponents(nested),
                                                    innerParamDsl.getContainedElement(groupAndParam.getSecond().getName()),
                                                    groupAndParam.getSecond(),
                                                    groupAndParam.getFirst(),
                                                    groupAndParam.getSecond().getType(),
                                                    extensionModelHelper);
                            });
                          });
                    });
              } else {
                if (paramComponent.getIdentifier().getName().equals(innerTypeParameterizedModel.getName())) {
                  ParameterizedModel parameterizedModel =
                      resolveParameterizedModel(extensionModelHelper, type, paramComponent.getIdentifier());
                  paramComponent.withParameterizedModel(parameterizedModel);

                  getGroupAndParametersPairs(innerTypeParameterizedModel).forEach(groupAndParam -> {
                    final Optional<DslElementSyntax> containedElement =
                        paramDsl.getContainedElement(groupAndParam.getSecond().getName());
                    enrichComponentModels(paramComponent,
                                          nestedComponents,
                                          containedElement,
                                          groupAndParam.getSecond(),
                                          groupAndParam.getFirst(),
                                          groupAndParam.getSecond().getType(),
                                          extensionModelHelper);

                  });
                }
              }
            });
      }
    };
  }

  private static List<ComponentAstBuilder> handleMapChild(DslElementSyntax paramDsl,
                                                          DefaultComponentAstBuilder paramComponent,
                                                          ParameterGroupModel parameterGroupModel,
                                                          ExtensionModelHelper extensionModelHelper,
                                                          ObjectType objectType) {
    return paramComponent.childComponentsStream()
        .filter(entryComponent -> {
          MetadataType entryType = objectType.getOpenRestriction().get();
          Optional<DslElementSyntax> entryValueDslOptional = paramDsl.getGeneric(entryType);
          if (entryValueDslOptional.isPresent()) {
            DslElementSyntax entryValueDsl = entryValueDslOptional.get();
            entryComponent.getGenerationInformation().withSyntax(entryValueDsl);

            ParameterModel keyParamModel =
                new ImmutableParameterModel(KEY_ATTRIBUTE_NAME, "", typeLoader.load(String.class), false, true, false, false,
                                            SUPPORTED, null, BEHAVIOUR, null, null, null, null, emptyList(), emptySet());
            ParameterModel valueParamModel =
                new ImmutableParameterModel(VALUE_ATTRIBUTE_NAME, "", entryType, false, true, false, false, SUPPORTED, null,
                                            BEHAVIOUR, null, null, null, null, emptyList(), emptySet());

            ImmutableParameterGroupModel keyMapEntryGroupModel =
                new ImmutableParameterGroupModel(DEFAULT_GROUP_NAME, "", asList(keyParamModel, valueParamModel), emptyList(),
                                                 false, null, null, emptySet());

            String key = entryComponent.getRawParameters().get(KEY_ATTRIBUTE_NAME);

            final Builder keyParamGenerationInfoBuilder = DefaultComponentGenerationInformation.builder();
            entryValueDsl.getAttribute(KEY_ATTRIBUTE_NAME).ifPresent(keyParamGenerationInfoBuilder::withSyntax);
            final Builder valueParamGenerationInfoBuilder = DefaultComponentGenerationInformation.builder();
            entryValueDsl.getAttribute(VALUE_ATTRIBUTE_NAME).ifPresent(valueParamGenerationInfoBuilder::withSyntax);

            entryComponent.withParameter(keyParamModel,
                                         keyMapEntryGroupModel,
                                         new DefaultComponentParameterAst(key,
                                                                          keyParamModel,
                                                                          keyMapEntryGroupModel,
                                                                          entryComponent.getMetadata(),
                                                                          keyParamGenerationInfoBuilder.build(),
                                                                          paramComponent.getPropertiesResolver()),
                                         empty());

            String value = entryComponent.getRawParameters().get(VALUE_ATTRIBUTE_NAME);

            if (isBlank(value)) {
              Optional<DslElementSyntax> genericValueDslOptional = entryValueDsl.getGeneric(keyParamModel.getType());

              Multimap<ComponentIdentifier, DefaultComponentAstBuilder> nestedComponents =
                  getNestedComponents(entryComponent);

              if (genericValueDslOptional.isPresent()) {
                DslElementSyntax genericValueDsl = genericValueDslOptional.get();
                List<ComponentAstBuilder> itemsComponentModels = entryComponent.childComponentsStream()
                    .filter(valueComponent -> valueComponent.getIdentifier()
                        .equals(getIdentifier(genericValueDsl).orElse(null)))
                    .map(entryValueComponent -> {
                      Multimap<ComponentIdentifier, DefaultComponentAstBuilder> nested = ArrayListMultimap.create();
                      nested.put(entryValueComponent.getIdentifier(), entryValueComponent);
                      enrichComponentModels(entryComponent,
                                            nested,
                                            of(genericValueDsl),
                                            valueParamModel,
                                            keyMapEntryGroupModel,
                                            valueParamModel.getType(),
                                            extensionModelHelper);
                      return entryValueComponent;
                    })
                    .collect(toList());

                entryComponent.withParameter(valueParamModel,
                                             keyMapEntryGroupModel,
                                             new DefaultComponentParameterAst(itemsComponentModels,
                                                                              valueParamModel,
                                                                              keyMapEntryGroupModel,
                                                                              entryComponent.getMetadata(),
                                                                              valueParamGenerationInfoBuilder.build(),
                                                                              paramComponent.getPropertiesResolver()),
                                             of(paramComponent.getIdentifier()));
              } else {
                handleLaxMapEntry(extensionModelHelper, entryComponent, entryType, entryValueDsl, valueParamModel,
                                  keyMapEntryGroupModel,
                                  nestedComponents);
              }
            } else {
              entryComponent.withParameter(valueParamModel,
                                           keyMapEntryGroupModel,
                                           new DefaultComponentParameterAst(value,
                                                                            valueParamModel,
                                                                            keyMapEntryGroupModel,
                                                                            entryComponent.getMetadata(),
                                                                            valueParamGenerationInfoBuilder.build(),
                                                                            paramComponent.getPropertiesResolver()),
                                           empty());
            }

            entryComponent.withParameterizedModel(createKeyValueWrapperTypeModelAdapter(keyParamModel.getName(),
                                                                                        keyParamModel.getType(),
                                                                                        valueParamModel.getName(),
                                                                                        valueParamModel.getType(),
                                                                                        extensionModelHelper));

            return true;
          }
          return false;
        }).collect(toList());
  }

  private static void handleLaxMapEntry(ExtensionModelHelper extensionModelHelper, DefaultComponentAstBuilder entryComponent,
                                        MetadataType entryType, DslElementSyntax entryValueDsl,
                                        ParameterModel valueParamModel, ParameterGroupModel parameterGroupModel,
                                        Multimap<ComponentIdentifier, DefaultComponentAstBuilder> nestedComponents) {
    Optional<DslElementSyntax> valueDslElementOptional = entryValueDsl.getContainedElement(VALUE_ATTRIBUTE_NAME);
    if (valueDslElementOptional.isPresent() && !valueDslElementOptional.get().isWrapped()) {
      // Either a simple value or an objectType
      enrichComponentModels(entryComponent,
                            nestedComponents,
                            valueDslElementOptional,
                            valueParamModel,
                            parameterGroupModel,
                            valueParamModel.getType(),
                            extensionModelHelper);
    } else if (entryType instanceof ObjectType) {
      // This case the value is a baseType therefore we need to go with subTypes
      extensionModelHelper.resolveSubTypes((ObjectType) entryType)
          .entrySet()
          .stream()
          .filter(entrySubTypeDslOptional -> entrySubTypeDslOptional.getValue().isPresent())
          .forEach(entrySubTypeDslOptional -> enrichComponentModels(entryComponent, nestedComponents,
                                                                    of(entrySubTypeDslOptional.getValue().get()),
                                                                    new ImmutableParameterModel(VALUE_ATTRIBUTE_NAME,
                                                                                                "",
                                                                                                entrySubTypeDslOptional
                                                                                                    .getKey(),
                                                                                                false,
                                                                                                true,
                                                                                                false, false,
                                                                                                SUPPORTED, null,
                                                                                                BEHAVIOUR, null, null,
                                                                                                null, null,
                                                                                                emptyList(),
                                                                                                emptySet()),
                                                                    parameterGroupModel,
                                                                    entrySubTypeDslOptional.getKey(),
                                                                    extensionModelHelper));
    }
  }

  private static MetadataTypeVisitor getArrayItemTypeVisitor(DefaultComponentAstBuilder componentModel,
                                                             ParameterModel paramModel,
                                                             ParameterGroupModel parameterGroupModel,
                                                             DslElementSyntax paramDsl,
                                                             DefaultComponentAstBuilder paramComponent,
                                                             ExtensionModelHelper extensionModelHelper) {
    return new MetadataTypeVisitor() {

      @Override
      public void visitSimpleType(SimpleType simpleType) {
        if (paramComponent.getRawParameters().containsKey(VALUE_ATTRIBUTE_NAME)) {
          paramComponent.withParameterizedModel(createSimpleWrapperTypeModelAdapter(simpleType, extensionModelHelper));
          return;
        }

        paramDsl.getGeneric(simpleType)
            .ifPresent(itemDsl -> {
              ComponentIdentifier itemIdentifier = getIdentifier(itemDsl).get();

              List<ComponentAstBuilder> componentModels = paramComponent.childComponentsStream()
                  .filter(c -> c.getIdentifier().equals(itemIdentifier))
                  .filter(valueComponentModel -> valueComponentModel.getRawParameters().containsKey(VALUE_ATTRIBUTE_NAME))
                  .peek(valueComponentModel -> {
                    valueComponentModel
                        .withParameterizedModel(createSimpleWrapperTypeModelAdapter(simpleType, extensionModelHelper));
                    valueComponentModel.getGenerationInformation().withSyntax(itemDsl);
                  })
                  .collect(toList());

              componentModel.withParameter(paramModel,
                                           parameterGroupModel,
                                           new DefaultComponentParameterAst(componentModels,
                                                                            paramModel,
                                                                            parameterGroupModel,
                                                                            paramComponent.getMetadata(),
                                                                            DefaultComponentGenerationInformation
                                                                                .builder()
                                                                                .withSyntax(paramDsl)
                                                                                .build(),
                                                                            componentModel.getPropertiesResolver()),
                                           of(paramComponent.getIdentifier()));

            });
      }

      @Override
      public void visitObject(ObjectType itemType) {
        paramDsl.getGeneric(itemType)
            .ifPresent(itemDsl -> {
              ComponentIdentifier itemIdentifier = getIdentifier(itemDsl).get();

              Map<String, ObjectType> objectTypeByTypeId = new HashMap<>();
              Map<String, Optional<DslElementSyntax>> typesDslMap = new HashMap<>();
              Map<ComponentIdentifier, String> itemIdentifiers = new HashMap<>();

              LOGGER.debug("getArrayItemTypeVisitor.visitObject: visiting itemType '{}'.", itemType);

              extensionModelHelper.resolveSubTypes(itemType).entrySet()
                  .stream()
                  .forEach(entry -> getTypeId(entry.getKey())
                      .ifPresent(subTypeTypeId -> {
                        typesDslMap.put(subTypeTypeId, entry.getValue());
                        objectTypeByTypeId.put(subTypeTypeId, entry.getKey());
                        entry.getValue().ifPresent(dslElementSyntax -> getIdentifier(dslElementSyntax)
                            .ifPresent(subTypeIdentifier -> itemIdentifiers.put(subTypeIdentifier, subTypeTypeId)));
                      }));

              getTypeId(itemType).ifPresent(itemTypeId -> {
                typesDslMap.putIfAbsent(itemTypeId, of(itemDsl));
                objectTypeByTypeId.putIfAbsent(itemTypeId, itemType);

                itemIdentifiers.putIfAbsent(itemIdentifier, itemTypeId);
              });

              LOGGER.debug("getArrayItemTypeVisitor.visitObject: itemIdentifiers: '{}'.", itemIdentifiers);

              List<ComponentAstBuilder> componentModels = paramComponent.childComponentsStream()
                  .filter(c -> itemIdentifiers.containsKey(c.getIdentifier()))
                  .peek(itemComponent -> {
                    String typeId = itemIdentifiers.get(itemComponent.getIdentifier());
                    typesDslMap.get(typeId).ifPresent(subTypeDsl -> {
                      ParameterizedModel parameterizedModel =
                          resolveParameterizedModel(extensionModelHelper, objectTypeByTypeId.get(typeId),
                                                    itemComponent.getIdentifier());
                      itemComponent.withParameterizedModel(parameterizedModel);
                      itemComponent.getGenerationInformation().withSyntax(subTypeDsl);

                      final Multimap<ComponentIdentifier, DefaultComponentAstBuilder> nestedComponents =
                          getNestedComponents(itemComponent);

                      parameterizedModel.getParameterGroupModels()
                          .forEach(nestedGroup -> nestedGroup.getParameterModels()
                              .forEach(nestedParameter -> enrichComponentModels(itemComponent,
                                                                                nestedComponents,
                                                                                recursiveAwareContainedElement(typesDslMap,
                                                                                                               subTypeDsl,
                                                                                                               nestedParameter),
                                                                                nestedParameter,
                                                                                nestedGroup,
                                                                                nestedParameter.getType(),
                                                                                extensionModelHelper)));
                    });
                  })
                  .collect(toList());

              componentModel.withParameter(paramModel,
                                           parameterGroupModel,
                                           new DefaultComponentParameterAst(componentModels,
                                                                            paramModel,
                                                                            parameterGroupModel,
                                                                            paramComponent.getMetadata(),
                                                                            DefaultComponentGenerationInformation
                                                                                .builder()
                                                                                .withSyntax(paramDsl)
                                                                                .build(),
                                                                            componentModel.getPropertiesResolver()),
                                           of(paramComponent.getIdentifier()));
            });
      }

      private Optional<DslElementSyntax> recursiveAwareContainedElement(Map<String, Optional<DslElementSyntax>> typesDslMap,
                                                                        DslElementSyntax subTypeDsl,
                                                                        ParameterModel nestedParameter) {
        return subTypeDsl.getContainedElement(nestedParameter.getName())
            .map(innerElement -> getTypeId(nestedParameter.getType())
                .flatMap(typeId -> {
                  if (typesDslMap.containsKey(typeId)) {
                    LOGGER
                        .debug("getArrayItemTypeVisitor.recursiveAwareContainedElement: No entry for '{}' in typesDslMap, ignoring.",
                               typeId);
                    return typesDslMap.get(typeId);
                  } else {
                    return empty();
                  }
                })
                .map(referencedDslElement -> {
                  LOGGER.debug("getArrayItemTypeVisitor.recursiveAwareContainedElement: processing typeId {}",
                               referencedDslElement);
                  final DslElementSyntaxBuilder baseReferenced = DslElementSyntaxBuilder.create()
                      .withAttributeName(innerElement.getAttributeName())
                      .withElementName(innerElement.getElementName())
                      .withNamespace(innerElement.getPrefix(), innerElement.getNamespace())
                      .requiresConfig(innerElement.requiresConfig())
                      .supportsAttributeDeclaration(innerElement.supportsAttributeDeclaration())
                      .supportsChildDeclaration(innerElement.supportsChildDeclaration())
                      .supportsTopLevelDeclaration(innerElement.supportsTopLevelDeclaration())
                      .asWrappedElement(innerElement.isWrapped());

                  // Handle recursive types by getting the element structure from the referenced type
                  referencedDslElement.getContainedElementsByName().entrySet()
                      .forEach(containedEntry -> baseReferenced.containing(containedEntry.getKey(), containedEntry.getValue()));
                  referencedDslElement.getGenerics().entrySet()
                      .forEach(genericEntry -> baseReferenced.withGeneric(genericEntry.getKey(), genericEntry.getValue()));

                  return baseReferenced.build();
                })
                .orElse(innerElement));
      }
    };
  }

  private static ParameterizedModel resolveParameterizedModel(ExtensionModelHelper extensionModelHelper, MetadataType type,
                                                              ComponentIdentifier identifier) {
    return extensionModelHelper.findComponentModel(identifier)
        .map(compModel -> (ParameterizedModel) compModel)
        .orElseGet(() -> createParameterizedTypeModelAdapter(type, extensionModelHelper));
  }


  private static Optional<ComponentIdentifier> getIdentifier(DslElementSyntax dsl) {
    return getIdentifier(dsl.getElementName(), dsl.getPrefix());
  }

  private static Optional<ComponentIdentifier> getIdentifier(String elementName, String prefix) {
    if (isNotBlank(elementName) && isNotBlank(prefix)) {
      return Optional.of(builder()
          .name(elementName)
          .namespace(prefix)
          .build());
    }
    return empty();
  }

  protected static Optional<String> getInfrastructureParameterName(MetadataType fieldType) {
    return getId(fieldType).map(getNameMap()::get);
  }

}
