/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.builder.adapter;

import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static org.mule.metadata.api.utils.MetadataTypeUtils.getTypeId;
import static org.mule.runtime.extension.api.util.NameUtils.getAliasName;
import static org.mule.runtime.internal.dsl.DslConstants.VALUE_ATTRIBUTE_NAME;

import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.ObjectTypeBuilder;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.model.SimpleType;
import org.mule.metadata.api.model.StringType;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.meta.model.stereotype.HasStereotypeModel;
import org.mule.runtime.ast.api.MetadataTypeAdapter;
import org.mule.runtime.ast.internal.model.ExtensionModelHelper;
import org.mule.runtime.extension.api.declaration.type.annotation.StereotypeTypeAnnotation;

import java.util.List;
import java.util.Optional;

public class MetadataTypeModelAdapter implements ParameterizedModel, MetadataTypeAdapter {

  /**
   * Adapts the provided {@code type} for treatment as both a {@link ParameterizedModel} and {@link HasStereotypeModel}.
   *
   * @param type                 the {@link MetadataType} to adapt.
   * @param extensionModelHelper used for actions relative to {@link MetadataType}s.
   * @return the newly created adapter if the provided {@code} is stereotyped, {@link Optional#empty()} if not.
   *
   * @since 1.0
   */
  public static Optional<MetadataTypeModelAdapter> createMetadataTypeModelAdapterWithSterotype(MetadataType type,
                                                                                               ExtensionModelHelper extensionModelHelper) {
    return type.getAnnotation(StereotypeTypeAnnotation.class)
        .flatMap(sta -> sta.getAllowedStereotypes().stream().findFirst())
        .map(st -> new MetadataTypeModelAdapterWithStereotype(type, st, extensionModelHelper));
  }

  /**
   * Adapts the provided {@code singleType} for treatment as a {@link ParameterizedModel}.
   * <p>
   * A new type is created around the provided {@code simpleType} to represent how it is represented in a DSL when it is nested in
   * an array.
   *
   * @param simpleType           the {@link MetadataType} to adapt.
   * @param extensionModelHelper used for actions relative to {@link MetadataType}s.
   * @return the newly created adapter.
   *
   * @since 1.0
   */
  public static MetadataTypeModelAdapter createSimpleWrapperTypeModelAdapter(SimpleType simpleType,
                                                                             ExtensionModelHelper extensionModelHelper) {
    ObjectTypeBuilder entryObjectTypeBuilder = new BaseTypeBuilder(MetadataFormat.JAVA).objectType();
    entryObjectTypeBuilder.addField().key(VALUE_ATTRIBUTE_NAME).value(simpleType);

    return new MetadataTypeModelAdapter(entryObjectTypeBuilder.build(), extensionModelHelper) {

      @Override
      public boolean isWrapperFor(MetadataType type) {
        return type instanceof StringType || super.isWrapperFor(type);
      }
    };
  }

  /**
   * Adapts the provided types representing the key/values of a map for treatment as a {@link ParameterizedModel}.
   * <p>
   * A new type is created around the provided types with the given paramNames to represent how it is represented in a DSL when it
   * is nested in a map.
   *
   * @param keyParamName         the name of the attribute in the DSL containing the entry key.
   * @param simpleKeyType        the {@link MetadataType} of the entry key
   * @param valueParamName       the name of the attribute in the DSL containing the entry value.
   * @param simpleValueType      the {@link MetadataType} of the entry value
   * @param extensionModelHelper used for actions relative to {@link MetadataType}s.
   * @return the newly created adapter.
   *
   * @since 1.0
   */
  public static MetadataTypeModelAdapter createKeyValueWrapperTypeModelAdapter(String keyParamName,
                                                                               MetadataType simpleKeyType,
                                                                               String valueParamName,
                                                                               MetadataType simpleValueType,
                                                                               ExtensionModelHelper extensionModelHelper) {
    ObjectTypeBuilder entryObjectTypeBuilder = new BaseTypeBuilder(MetadataFormat.JAVA).objectType();
    entryObjectTypeBuilder.addField().key(keyParamName).value(simpleKeyType);
    entryObjectTypeBuilder.addField().key(valueParamName).value(simpleValueType);

    return new MetadataTypeModelAdapter(entryObjectTypeBuilder.build(), extensionModelHelper);
  }

  /**
   * Adapts the provided {@code type} for treatment as a {@link ParameterizedModel}.
   *
   * @param type                 the {@link MetadataType} to adapt.
   * @param extensionModelHelper used for actions relative to {@link MetadataType}s.
   * @return the newly created adapter.
   *
   * @since 1.0
   */
  public static MetadataTypeModelAdapter createParameterizedTypeModelAdapter(MetadataType type,
                                                                             ExtensionModelHelper extensionModelHelper) {
    return new MetadataTypeModelAdapter(type, extensionModelHelper);
  }

  private final MetadataType type;
  private final List<ParameterGroupModel> parameterGroupModels;

  MetadataTypeModelAdapter(MetadataType type, ExtensionModelHelper extensionModelHelper) {
    this.type = type;
    if (type instanceof ObjectType) {
      parameterGroupModels = singletonList(new ObjectTypeAsParameterGroupAdapter((ObjectType) type,
                                                                                 extensionModelHelper
                                                                                     .findMetadataType(String.class)
                                                                                     .orElse(null)));
    } else {
      parameterGroupModels = emptyList();
    }
  }

  @Override
  public String getName() {
    return getAliasName(type);
  }

  @Override
  public String getDescription() {
    return "MetadataTypeModelAdapter for " + getTypeId(type).orElse(type.toString());
  }

  @Override
  public List<ParameterGroupModel> getParameterGroupModels() {
    return parameterGroupModels;
  }

  @Override
  public MetadataType getType() {
    return type;
  }

  @Override
  public boolean isWrapperFor(MetadataType type) {
    return this.type.equals(type);
  }

  @Override
  public String toString() {
    return "MetadataTypeModelAdapter{" + type.toString() + "}";
  }
}
