/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.api;

import org.mule.runtime.api.functional.Either;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;

import java.util.Optional;

/**
 * Represents the user configuration of a parameter of a component.
 *
 * @since 1.0
 */
public interface ComponentParameterAst {

  /**
   * @return the model that represents this parameter.
   */
  ParameterModel getModel();

  /**
   * @return the group model that contains the represented parameter or {@code null}.
   */
  ParameterGroupModel getGroupModel();

  /**
   * Returns either the expression for this parameter or a fixed value, depending on what is set in the DSL.
   * <p>
   * When using the {@code right} part returned by this method for a parameter that is a complex object defined inline, the
   * definition of the object will be returned (a {@link ComponentAst}), not the object itself.
   *
   * @return the value of the parameter, or its default value if there is one defined and the parameter is not set in the DSL.
   */
  <T> Either<String, T> getValue();

  /**
   * This method WILL NOT resolve any property placeholders to the actual values if present. Use {@link #getResolvedRawValue()} or
   * {@link #getValue()} if properties resolution is needed.
   *
   * @return the value of the parameter as defined in the DSL.
   */
  String getRawValue();

  /**
   * This method WILL resolve any property placeholders to the actual values if present. Use {@link #getRawValue()} if properties
   * resolution is not wanted.
   * <p>
   * Property placeholders resolution is done with the object given to
   * {@link ArtifactAst#updatePropertiesResolver(java.util.function.UnaryOperator)}.
   *
   * @return the value of the parameter as defined in the DSL and its property placeholders resolved.
   */
  String getResolvedRawValue();

  /**
   * @return the parser metadata for this parameter if it is defined as a nested component, or {@link Optional#empty()} otherwise.
   */
  Optional<ComponentMetadataAst> getMetadata();

  /**
   * @return an object containing information about the generation of this parameter.
   */
  ComponentGenerationInformation getGenerationInformation();

  /**
   * Returns {@code true} if this is a default parameter; returns {@code false} otherwise.
   *
   * @return true if and only if this is a default parameter
   */
  boolean isDefaultValue();
}
