/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.api.util;

import static java.util.Arrays.asList;
import static org.mule.runtime.ast.api.error.ErrorTypeRepositoryProvider.getCoreErrorTypeRepo;
import static org.mule.runtime.ast.api.util.AstTraversalDirection.TOP_DOWN;
import static org.mule.runtime.ast.internal.error.ErrorTypeRepositoryBuildingUtils.addErrorsFromArtifact;
import static org.mule.runtime.ast.internal.error.ErrorTypeRepositoryBuildingUtils.addErrorsFromExtensions;
import static org.slf4j.LoggerFactory.getLogger;
import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.internal.error.CompositeErrorTypeRepository;
import org.mule.runtime.ast.internal.error.DefaultErrorTypeRepository;

import java.util.Spliterator;
import java.util.function.Predicate;
import java.util.stream.Stream;

import org.slf4j.Logger;

/**
 * Base implementation of {@link ArtifactAst}.
 * <p>
 * Extending classes need only implement {@link #topLevelComponents()}, and the values returned by the other methods are derived
 * from it.
 *
 * @since 1.0
 */
public abstract class BaseArtifactAst implements ArtifactAst {

  private static final Logger LOGGER = getLogger(BaseArtifactAst.class);

  @Override
  public Stream<ComponentAst> topLevelComponentsStream() {
    return topLevelComponents().stream();
  }

  @Override
  public Spliterator<ComponentAst> topLevelComponentsSpliterator() {
    return topLevelComponentsStream().spliterator();
  }

  @Override
  public Spliterator<ComponentAst> recursiveSpliterator(AstTraversalDirection direction) {
    return recursiveStream(direction).spliterator();
  }

  @Override
  public Spliterator<ComponentAst> recursiveSpliterator() {
    return recursiveSpliterator(TOP_DOWN);
  }

  @Override
  public Stream<ComponentAst> recursiveStream(AstTraversalDirection direction) {
    return topLevelComponentsStream().flatMap(cm -> cm.recursiveStream(direction));
  }

  @Override
  public Stream<ComponentAst> recursiveStream() {
    return recursiveStream(TOP_DOWN);
  }

  @Override
  public Stream<ComponentAst> filteredComponents(Predicate<ComponentAst> predicate) {
    return recursiveStream().filter(predicate);
  }

  @Override
  public ErrorTypeRepository getErrorTypeRepository() {
    LOGGER.debug("Creating errorTypeRepository for Artifact '{}'...", this);

    ErrorTypeRepository errorTypeRepository = new DefaultErrorTypeRepository();

    addErrorsFromExtensions(dependencies(), errorTypeRepository);
    addErrorsFromArtifact(this, errorTypeRepository);

    return new CompositeErrorTypeRepository(asList(getParent()
        .map(ArtifactAst::getErrorTypeRepository)
        .orElse(getCoreErrorTypeRepo()), errorTypeRepository));
  }
}
