/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.api.util;

import static org.mule.runtime.api.functional.Either.right;
import static org.mule.runtime.extension.api.ExtensionConstants.ERROR_MAPPINGS_PARAMETER_NAME;
import static org.mule.runtime.extension.api.util.ExtensionMetadataTypeUtils.isMap;

import static java.util.Collections.emptyList;
import static java.util.Collections.unmodifiableList;
import static java.util.stream.Collectors.toList;

import org.mule.metadata.api.model.ArrayType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.visitor.MetadataTypeVisitor;
import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.api.functional.Either;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.ast.api.ImportedResource;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.function.UnaryOperator;
import java.util.stream.Stream;

/**
 * Provides common utility methods to do AST transformations.
 *
 * @since 1.0
 */
public final class MuleArtifactAstCopyUtils {

  private MuleArtifactAstCopyUtils() {
    // Nothing to do
  }

  /**
   * Copies and transforms a given {@link ArtifactAst}, mapping its internal {@link ComponentAst} instances with the provided
   * {@code compoenentMapper} function.
   *
   * @param source          the {@link ArtifactAst} to copy and transform.
   * @param componentMapper transformation to apply to each {@link ComponentAst} from {@code source}. If no transformation is to
   *                        be done for a given {@link ComponentAst}, the same instance must be returned.
   * @return a transformed copy of the {@code source} AST.
   */
  public static ArtifactAst copyRecursively(ArtifactAst source, UnaryOperator<ComponentAst> componentMapper) {
    return copyRecursively(source, componentMapper, Collections::emptyList, c -> false);
  }

  /**
   * Copies and transforms a given {@link ArtifactAst}, mapping its internal {@link ComponentAst} instances with the provided
   * {@code compoenentMapper} function.
   * <p>
   * This method also allows for adding or removing top level elements from the transformed AST.
   * <p>
   * In the case that for a newly added component, {@code topLevelToRemove} returns {@code true}, that component will NOT be on
   * the returned AST.
   *
   * @param source           the {@link ArtifactAst} to copy and transform.
   * @param componentMapper  transformation to apply to each {@link ComponentAst} from {@code source}. If no transformation is to
   *                         be done for a given {@link ComponentAst}, the same instance must be returned.
   * @param topLeveltoAdd    components to be added as top-level elements in the transformed AST
   * @param topLevelToRemove any top-level elements for which this returns true will be removed from the returned AST.
   * @return a transformed copy of the {@code source} AST.
   */
  public static ArtifactAst copyRecursively(ArtifactAst source, UnaryOperator<ComponentAst> componentMapper,
                                            Supplier<List<ComponentAst>> topLeveltoAdd,
                                            Predicate<ComponentAst> topLevelToRemove) {

    final List<ComponentAst> mappedTopLevelTemp = source.topLevelComponentsStream()
        .map(c -> copyComponentTreeRecursively(c, componentMapper))
        .collect(toList());

    mappedTopLevelTemp.addAll(topLeveltoAdd.get());
    mappedTopLevelTemp.removeIf(topLevelToRemove);
    final List<ComponentAst> mappedTopLevel = unmodifiableList(mappedTopLevelTemp);

    return new BaseArtifactAst() {

      @Override
      public Set<ExtensionModel> dependencies() {
        return source.dependencies();
      }

      @Override
      public Optional<ArtifactAst> getParent() {
        return source.getParent();
      }

      @Override
      public List<ComponentAst> topLevelComponents() {
        return mappedTopLevel;
      }

      @Override
      public void updatePropertiesResolver(UnaryOperator<String> newPropertiesResolver) {
        source.updatePropertiesResolver(newPropertiesResolver);
      }

      @Override
      public ErrorTypeRepository getErrorTypeRepository() {
        return source.getErrorTypeRepository();
      }

      @Override
      public Collection<ImportedResource> getImportedResources() {
        return source.getImportedResources();
      }
    };
  }

  /**
   * Copies and transforms a given {@link ComponentAst}, mapping it and its children {@link ComponentAst} instances recursively
   * with the provided {@code compoenentMapper} function.
   *
   * @param source          the {@link ComponentAst} to copy and transform recursively.
   * @param componentMapper transformation to apply to each {@link ComponentAst} from {@code source}. If no transformation is to
   *                        be done for a given {@link ComponentAst}, the same instance must be returned.
   * @return a transformed copy of the {@code source} AST.
   */
  public static ComponentAst copyComponentTreeRecursively(ComponentAst source,
                                                          UnaryOperator<ComponentAst> componentMapper) {
    AtomicBoolean anyChildMapped = new AtomicBoolean(false);
    AtomicBoolean anyParamMapped = new AtomicBoolean(false);
    final List<ComponentAst> mappedDirectChildren = unmodifiableList(source.directChildrenStream()
        .map(child -> {
          final ComponentAst mappedChild = copyComponentTreeRecursively(child, componentMapper);

          if (mappedChild != child) {
            anyChildMapped.set(true);
          }

          return mappedChild;
        })
        .collect(toList()));

    final List<ComponentParameterAst> mappedParameters = unmodifiableList(source.getModel(ParameterizedModel.class)
        .map(pmzd -> source.getParameters().stream()
            .map(param -> {
              AtomicReference mappedValue = new AtomicReference<>();

              if (param.getModel().getName().equals(ERROR_MAPPINGS_PARAMETER_NAME)) {
                return param;
              }
              param.getModel().getType().accept(new MetadataTypeVisitor() {

                @Override
                public void visitArrayType(ArrayType arrayType) {
                  final Object value = param.getValue().getRight();
                  if (value != null && value instanceof Collection) {
                    mappedValue.set(((Collection<ComponentAst>) value).stream()
                        .map(this::doMap)
                        .collect(toList()));
                  }
                }

                @Override
                public void visitObject(ObjectType objectType) {
                  final Object value = param.getValue().getRight();
                  if (isMap(objectType)) {
                    if (value != null && value instanceof Collection) {
                      mappedValue.set(((Collection<ComponentAst>) value).stream()
                          .map(this::doMap)
                          .collect(toList()));
                    }
                  } else {
                    if (value instanceof ComponentAst) {
                      mappedValue.set(doMap((ComponentAst) value));
                    }
                  }
                }

                private ComponentAst doMap(ComponentAst component) {
                  return copyComponentTreeRecursively(component, componentMapper);
                }
              });

              if (mappedValue.get() == null) {
                return param;
              } else {
                anyParamMapped.set(true);
                return new BaseComponentParameterAstDecorator(param) {

                  @Override
                  public <T> Either<String, T> getValue() {
                    return right(String.class, (T) mappedValue.get());
                  }
                };
              }
            })
            .collect(toList()))
        .orElse(emptyList()));

    if (anyChildMapped.get() || anyParamMapped.get()) {
      return componentMapper.apply(new BaseComponentAstDecorator(source) {

        @Override
        public List<ComponentAst> directChildren() {
          return anyChildMapped.get() ? mappedDirectChildren : super.directChildren();
        }

        @Override
        public Stream<ComponentAst> directChildrenStream() {
          return anyChildMapped.get() ? mappedDirectChildren.stream() : super.directChildrenStream();
        }

        @Override
        public Collection<ComponentParameterAst> getParameters() {
          return anyParamMapped.get() ? mappedParameters : super.getParameters();
        }

        @Override
        public ComponentParameterAst getParameter(String groupName, String paramName) {
          return anyParamMapped.get()
              ? mappedParameters.stream()
                  .filter(p -> p.getModel().getName().equals(paramName)
                      && p.getGroupModel().getName().equals(groupName))
                  .findAny().orElse(null)
              : super.getParameter(groupName, paramName);
        }
      });

    } else {
      return componentMapper.apply(source);
    }
  }
}
