/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.builder;

import static com.google.common.base.Objects.equal;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static java.util.Optional.of;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.nullValue;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static org.mule.metadata.api.builder.BaseTypeBuilder.create;
import static org.mule.metadata.api.model.MetadataFormat.JAVA;
import static org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.DEFAULT_GROUP_NAME;
import static org.mule.runtime.ast.api.util.TestUtils.createMockParameterGroup;

import org.mule.metadata.api.model.MetadataType;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType;
import org.mule.runtime.api.functional.Either;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentGenerationInformation;
import org.mule.runtime.ast.api.ComponentMetadataAst;
import org.mule.runtime.ast.api.builder.ComponentAstBuilder;
import org.mule.runtime.ast.internal.DefaultComponentMetadataAst;
import org.mule.runtime.ast.internal.DefaultComponentParameterAst;
import org.mule.runtime.ast.internal.model.ExtensionModelHelper;
import org.mule.runtime.extension.api.dsl.syntax.DslElementSyntax;

import java.util.function.UnaryOperator;

import org.junit.Before;
import org.junit.Test;

public class PropertiesResolverTestCase {

  private static final String PARAM_NAME = "param";
  private static final String PARAM_FIXED_VALUE = "fixed";
  private static final String PARAM_PLACEHOLDER_VALUE = "value!";
  private static final String PARAM_PLACEHOLDER = "${key}";

  private static final UnaryOperator<String> SIMPLE_PROPERTY_MAPPING = key -> {
    if (equal(PARAM_PLACEHOLDER, key)) {
      return PARAM_PLACEHOLDER_VALUE;
    } else {
      return key;
    }
  };

  private PropertiesResolver propertiesResolver;

  private ExtensionModelHelper extModelHelper;

  private ParameterModel componentParamModel;
  private ParameterGroupModel componentParamGroupModel;
  private ParameterizedModel parameterized;

  private ParameterModel idComponentParamModel;
  private ParameterGroupModel idComponentParamGroupModel;
  private ParameterizedModel idParameterized;

  @Before
  public void before() {
    propertiesResolver = new PropertiesResolver();

    extModelHelper = mock(ExtensionModelHelper.class);

    componentParamModel = mock(ParameterModel.class);
    when(componentParamModel.getName()).thenReturn(PARAM_NAME);
    when(componentParamModel.getType()).thenReturn(create(JAVA).stringType().build());

    componentParamGroupModel = mock(ParameterGroupModel.class);
    when(componentParamGroupModel.getName()).thenReturn(DEFAULT_GROUP_NAME);
    when(componentParamGroupModel.getParameterModels()).thenReturn(singletonList(componentParamModel));
    when(componentParamGroupModel.getParameter(PARAM_NAME)).thenReturn(of(componentParamModel));

    parameterized = mock(ParameterizedModel.class);
    when(parameterized.getAllParameterModels()).thenReturn(singletonList(componentParamModel));
    when(parameterized.getParameterGroupModels()).thenReturn(singletonList(componentParamGroupModel));

    idComponentParamModel = mock(ParameterModel.class);
    when(idComponentParamModel.getName()).thenReturn("theId");
    when(idComponentParamModel.isComponentId()).thenReturn(true);
    when(idComponentParamModel.isRequired()).thenReturn(true);
    when(idComponentParamModel.getType()).thenReturn(mock(MetadataType.class));

    idComponentParamGroupModel = mock(ParameterGroupModel.class);
    when(idComponentParamGroupModel.getName()).thenReturn(DEFAULT_GROUP_NAME);
    when(idComponentParamGroupModel.getParameterModels()).thenReturn(singletonList(idComponentParamModel));
    when(idComponentParamGroupModel.getParameter(PARAM_NAME)).thenReturn(of(idComponentParamModel));

    idParameterized = mock(ParameterizedModel.class);
    when(idParameterized.getAllParameterModels()).thenReturn(singletonList(idComponentParamModel));
    when(idParameterized.getParameterGroupModels()).thenReturn(singletonList(idComponentParamGroupModel));
  }

  @Test
  public void nullValueParam() {
    propertiesResolver.setMappingFunction(SIMPLE_PROPERTY_MAPPING);

    assertThat(new DefaultComponentParameterAst((String) null, mock(ParameterModel.class), createMockParameterGroup(),
                                                mock(ComponentGenerationInformation.class), propertiesResolver)
                                                    .getRawValue(),
               nullValue());
  }

  @Test
  public void fixedParam() {
    propertiesResolver.setMappingFunction(SIMPLE_PROPERTY_MAPPING);

    assertThat(new DefaultComponentParameterAst(PARAM_FIXED_VALUE, mock(ParameterModel.class), createMockParameterGroup(),
                                                mock(ComponentGenerationInformation.class), propertiesResolver)
                                                    .getRawValue(),
               is(PARAM_FIXED_VALUE));
  }

  @Test
  public void fixedComplexValueParam() {
    propertiesResolver.setMappingFunction(SIMPLE_PROPERTY_MAPPING);

    final DefaultComponentAstBuilder complexParamValue = complexParamValueStub();
    complexParamValue.withRawParameter(PARAM_NAME, PARAM_FIXED_VALUE);

    final Either<String, Object> mappedComplexParam =
        new DefaultComponentParameterAst(complexParamValue, mock(ParameterModel.class), createMockParameterGroup(),
                                         mock(ComponentMetadataAst.class),
                                         mock(ComponentGenerationInformation.class), propertiesResolver)
                                             .getValue();

    assertThat(mappedComplexParam.reduce(UnaryOperator.identity(),
                                         complexParam -> ((ComponentAst) complexParam)
                                             .getParameter(DEFAULT_GROUP_NAME, PARAM_NAME).getRawValue()),
               is(PARAM_FIXED_VALUE));
  }

  @Test
  public void propertyParam() {
    propertiesResolver.setMappingFunction(SIMPLE_PROPERTY_MAPPING);

    assertThat(new DefaultComponentParameterAst(PARAM_PLACEHOLDER, mock(ParameterModel.class), createMockParameterGroup(),
                                                mock(ComponentGenerationInformation.class), propertiesResolver)
                                                    .getResolvedRawValue(),
               is(PARAM_PLACEHOLDER_VALUE));
  }

  @Test
  public void propertyComplexValueParam() {
    propertiesResolver.setMappingFunction(SIMPLE_PROPERTY_MAPPING);

    final DefaultComponentAstBuilder complexParamValue = complexParamValueStub();
    complexParamValue.withRawParameter(PARAM_NAME, PARAM_PLACEHOLDER);

    final Either<String, Object> mappedComplexParam =
        new DefaultComponentParameterAst(complexParamValue, mock(ParameterModel.class), createMockParameterGroup(),
                                         mock(ComponentMetadataAst.class),
                                         mock(ComponentGenerationInformation.class), propertiesResolver)
                                             .getValue();

    assertThat(mappedComplexParam.reduce(UnaryOperator.identity(),
                                         complexParam -> ((ComponentAst) complexParam)
                                             .getParameter(DEFAULT_GROUP_NAME, PARAM_NAME)
                                             .getRawValue()),
               is(PARAM_PLACEHOLDER));

    assertThat(mappedComplexParam.reduce(UnaryOperator.identity(),
                                         complexParam -> ((ComponentAst) complexParam)
                                             .getParameter(DEFAULT_GROUP_NAME, PARAM_NAME)
                                             .getResolvedRawValue()),
               is(PARAM_PLACEHOLDER_VALUE));
  }

  @Test
  public void propertyParamNoMapping() {
    assertThat(new DefaultComponentParameterAst(PARAM_PLACEHOLDER, mock(ParameterModel.class), createMockParameterGroup(),
                                                mock(ComponentGenerationInformation.class), propertiesResolver)
                                                    .getRawValue(),
               is(PARAM_PLACEHOLDER));
  }

  @Test
  public void propertyComplexValueParamNoMapping() {
    final DefaultComponentAstBuilder complexParamValue = complexParamValueStub();
    complexParamValue.withRawParameter(PARAM_NAME, PARAM_PLACEHOLDER);

    final Either<String, Object> mappedComplexParam =
        new DefaultComponentParameterAst(complexParamValue, mock(ParameterModel.class), createMockParameterGroup(),
                                         mock(ComponentMetadataAst.class),
                                         mock(ComponentGenerationInformation.class), propertiesResolver)
                                             .getValue();

    assertThat(mappedComplexParam.reduce(UnaryOperator.identity(),
                                         complexParam -> ((ComponentAst) complexParam)
                                             .getParameter(DEFAULT_GROUP_NAME, PARAM_NAME).getRawValue()),
               is(PARAM_PLACEHOLDER));
  }

  @Test
  public void componentIdMapped() {
    final DefaultComponentAstBuilder idParamValue =
        new DefaultComponentAstBuilder(propertiesResolver, extModelHelper, emptyList(), 0);
    idParamValue.getGenerationInformation().withSyntax(mock(DslElementSyntax.class));

    final ComponentIdentifier identifier = ComponentIdentifier.builder()
        .namespace("mockns").name("compWithId").build();
    idParamValue.withIdentifier(identifier);
    idParamValue.withMetadata(DefaultComponentMetadataAst.builder().build());
    final ComponentAstBuilder builder = idParamValue
        .withParameterizedModel(idParameterized)
        .withRawParameter("theId", PARAM_PLACEHOLDER);

    propertiesResolver.setMappingFunction(SIMPLE_PROPERTY_MAPPING);

    final ComponentAst comp = builder.build();

    assertThat(comp.getComponentId().get(), is(PARAM_PLACEHOLDER_VALUE));
  }

  private DefaultComponentAstBuilder complexParamValueStub() {
    final ComponentIdentifier identifier = ComponentIdentifier.builder()
        .namespace("mockns").name("comp").build();

    when(extModelHelper.findComponentType(identifier)).thenReturn(ComponentType.UNKNOWN);

    final DefaultComponentAstBuilder complexParamValue = new DefaultComponentAstBuilder(propertiesResolver,
                                                                                        extModelHelper,
                                                                                        emptyList(),
                                                                                        0);
    complexParamValue.withIdentifier(identifier);
    complexParamValue.withMetadata(DefaultComponentMetadataAst.builder().build());
    complexParamValue.withParameterizedModel(parameterized);
    complexParamValue.getGenerationInformation().withSyntax(mock(DslElementSyntax.class));
    return complexParamValue;
  }
}
