/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.error;

import static java.util.Arrays.asList;
import static java.util.stream.Collectors.toSet;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsInAnyOrder;
import static org.mule.runtime.ast.AllureConstants.ArtifactAst.Errors.ERROR_TYPES;
import static org.mule.runtime.ast.api.error.ErrorTypeRepositoryProvider.getCoreErrorTypeRepo;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.exception.ErrorTypeRepository;

import java.util.Set;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

import io.qameta.allure.Story;

@Story(ERROR_TYPES)
public class CompositeErrorTypeRepositoryTestCase {

  private ErrorTypeRepository errorTypeRepositoryA;
  private final ComponentIdentifier INTERNAL_ERROR =
      ComponentIdentifier.builder().namespace("NSA").name("NAME").build();
  private final ComponentIdentifier EXPOSED_ERROR =
      ComponentIdentifier.builder().namespace("NSB").name("OTHER_NAME").build();
  private final ComponentIdentifier OTHER_INTERNAL_ERROR =
      ComponentIdentifier.builder().namespace("NSB").name("NAME").build();

  private ErrorTypeRepository errorTypeRepositoryB;
  private final ComponentIdentifier ns1error = ComponentIdentifier.builder().namespace("NS1").name("AN_ERROR").build();
  private final ComponentIdentifier ns2error = ComponentIdentifier.builder().namespace("NS2").name("AN_ERROR").build();
  private final ComponentIdentifier ns1internalError =
      ComponentIdentifier.builder().namespace("NS1").name("AN_INTERNAL_ERROR").build();
  private final ComponentIdentifier ns2internalError =
      ComponentIdentifier.builder().namespace("NS2").name("AN_INTERNAL_ERROR").build();

  @Rule
  public ExpectedException expectedException = ExpectedException.none();

  @Before
  public void setUp() {
    errorTypeRepositoryA = new DefaultErrorTypeRepository();
    errorTypeRepositoryA.addErrorType(EXPOSED_ERROR, getCoreErrorTypeRepo().getAnyErrorType());
    errorTypeRepositoryA.addInternalErrorType(INTERNAL_ERROR, getCoreErrorTypeRepo().getCriticalErrorType());
    errorTypeRepositoryA.addInternalErrorType(OTHER_INTERNAL_ERROR, getCoreErrorTypeRepo().getCriticalErrorType());

    errorTypeRepositoryB = new DefaultErrorTypeRepository();
    errorTypeRepositoryB.addErrorType(ns1error, getCoreErrorTypeRepo().getAnyErrorType());
    errorTypeRepositoryB.addErrorType(ns2error, getCoreErrorTypeRepo().getAnyErrorType());
    errorTypeRepositoryB.addInternalErrorType(ns1internalError, getCoreErrorTypeRepo().getAnyErrorType());
    errorTypeRepositoryB.addInternalErrorType(ns2internalError, getCoreErrorTypeRepo().getAnyErrorType());
  }

  @Test
  public void getErrorNamespaces() {
    final CompositeErrorTypeRepository composite =
        new CompositeErrorTypeRepository(asList(errorTypeRepositoryA, errorTypeRepositoryB));

    assertThat(composite.getErrorNamespaces(), containsInAnyOrder("NSA", "NSB", "NS1", "NS2"));
  }

  @Test
  public void getErrorTypes() {
    final CompositeErrorTypeRepository composite =
        new CompositeErrorTypeRepository(asList(errorTypeRepositoryA, errorTypeRepositoryB));

    final Set<ComponentIdentifier> errorTypeIdentifiers = composite.getErrorTypes().stream()
        .map(err -> ComponentIdentifier.builder().namespace(err.getNamespace()).name(err.getIdentifier()).build())
        .collect(toSet());

    assertThat(errorTypeIdentifiers, containsInAnyOrder(EXPOSED_ERROR, ns1error, ns2error));
  }

  @Test
  public void getInternalErrorTypes() {
    final CompositeErrorTypeRepository composite =
        new CompositeErrorTypeRepository(asList(errorTypeRepositoryA, errorTypeRepositoryB));

    final Set<ComponentIdentifier> errorTypeIdentifiers = composite.getInternalErrorTypes().stream()
        .map(err -> ComponentIdentifier.builder().namespace(err.getNamespace()).name(err.getIdentifier()).build())
        .collect(toSet());

    assertThat(errorTypeIdentifiers,
               containsInAnyOrder(INTERNAL_ERROR, OTHER_INTERNAL_ERROR, ns1internalError, ns2internalError));
  }

}
