/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.error;

import static java.lang.String.format;
import static org.mule.runtime.api.component.ComponentIdentifier.builder;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.meta.model.error.ErrorModelBuilder.newError;
import static org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.DEFAULT_GROUP_NAME;
import static org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.ERROR_MAPPINGS;
import static org.mule.runtime.ast.api.error.ErrorTypeRepositoryProvider.getCoreErrorTypeRepo;
import static org.mule.runtime.extension.api.ExtensionConstants.ERROR_MAPPINGS_PARAMETER_NAME;
import static org.mule.runtime.internal.dsl.DslConstants.CORE_PREFIX;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.message.ErrorType;
import org.mule.runtime.api.meta.model.ComponentModel;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.construct.ConstructModel;
import org.mule.runtime.api.meta.model.error.ErrorModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.api.meta.model.util.ExtensionWalker;
import org.mule.runtime.api.meta.model.util.IdempotentExtensionWalker;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.extension.api.error.ErrorMapping;
import org.mule.runtime.extension.internal.property.NoErrorMappingModelProperty;

import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;

/**
 * Provides the functionality to navigate {@link ExtensionModel}s and {@link ArtifactAst}s in order to properly populate an
 * {@link ErrorTypeRepository} for a given artifact.
 *
 * @since 1.0
 */
public class ErrorTypeRepositoryBuildingUtils {

  private static final Logger LOGGER = getLogger(ErrorTypeRepositoryBuildingUtils.class);

  public static final String CORE_ERROR_NS = CORE_PREFIX.toUpperCase();

  public static final String RETRY_EXHAUSTED_ERROR_IDENTIFIER = "RETRY_EXHAUSTED";
  public static final String CONNECTIVITY_ERROR_IDENTIFIER = "CONNECTIVITY";

  public static final String RAISE_ERROR = "raise-error";
  public static final String ERROR_TYPE_PARAM = "type";

  public static final ComponentIdentifier RETRY_EXHAUSTED = builder()
      .namespace(CORE_ERROR_NS).name(RETRY_EXHAUSTED_ERROR_IDENTIFIER).build();
  public static final ComponentIdentifier CONNECTIVITY =
      builder().namespace(CORE_ERROR_NS).name(CONNECTIVITY_ERROR_IDENTIFIER).build();

  public static final ComponentIdentifier RAISE_ERROR_IDENTIFIER =
      builder().namespace(CORE_PREFIX).name(RAISE_ERROR).build();

  /**
   * Populates the provided {@code errorTypeRepository} with the errors declared in the provided {@code muleErrorTypeRepository}.
   *
   * @param extensionModels     the extension models to get the error types from.
   * @param errorTypeRepository the repository to populate
   */
  public static void addErrorsFromExtensions(Set<ExtensionModel> extensionModels, ErrorTypeRepository errorTypeRepository) {
    extensionModels
        .stream()
        .filter(em -> !em.getName().equals(CORE_PREFIX))
        .forEach(extensionModel -> addErrorsFromExtension(errorTypeRepository, extensionModel));
  }

  private static void addErrorsFromExtension(ErrorTypeRepository errorTypeRepository, ExtensionModel extensionModel) {
    LOGGER.debug("Registering errorTypes from extension '{}'...", extensionModel.getName());

    Set<ErrorModel> errorTypes = extensionModel.getErrorModels();
    String extensionNamespace = extensionModel.getXmlDslModel().getPrefix();
    String errorExtensionNamespace = extensionNamespace.toUpperCase();

    errorTypes
        .forEach(errorModel -> getErrorType(errorTypeRepository, errorModel, extensionModel));

    ErrorModel connectivityErrorModel = newError(CONNECTIVITY_ERROR_IDENTIFIER, errorExtensionNamespace)
        .withParent(newError(CONNECTIVITY).build()).build();

    ErrorModel retryExhaustedError = newError(RETRY_EXHAUSTED_ERROR_IDENTIFIER, errorExtensionNamespace)
        .withParent(newError(RETRY_EXHAUSTED).build()).build();

    ExtensionWalker extensionWalker = new IdempotentExtensionWalker() {

      @Override
      protected void onSource(SourceModel model) {
        registerErrors(model);
        stop();
      }

      @Override
      protected void onOperation(OperationModel model) {
        registerErrors(model);
        stop();
      }

      @Override
      protected void onConstruct(ConstructModel model) {
        registerErrors(model);
        stop();
      }

      private void registerErrors(ComponentModel model) {
        if (!errorTypes.isEmpty()) {
          getErrorType(errorTypeRepository, connectivityErrorModel, extensionModel);
          getErrorType(errorTypeRepository, retryExhaustedError, extensionModel);
        }
      }
    };
    extensionWalker.walk(extensionModel);
  }

  /**
   * Populates the provided {@code errorTypeRepository} with the errors declared in the components that may declare new errors.
   *
   * @param artifact            the AST to navigate to detect the declared error types
   * @param errorTypeRepository the repository to populate
   */
  public static void addErrorsFromArtifact(ArtifactAst artifact, ErrorTypeRepository errorTypeRepository) {
    Set<String> syntheticErrorNamespaces = new HashSet<>();

    artifact.topLevelComponentsStream()
        .forEach(componentModel -> resolveErrorTypes(errorTypeRepository, componentModel, syntheticErrorNamespaces));
  }

  private static void resolveErrorTypes(ErrorTypeRepository errorTypeRepository, ComponentAst componentModel,
                                        Set<String> syntheticErrorNamespaces) {
    componentModel.directChildrenStream()
        // Only take into account what was actually in the original AST, not any modifications done afterwards
        .filter(innerComponent -> componentModel.getMetadata().getFileName()
            .flatMap(fileName -> innerComponent.getMetadata().getFileName().map(fileName::equals))
            .orElse(false))
        .forEach(innerComponent -> {
          processRaiseError(errorTypeRepository, innerComponent, syntheticErrorNamespaces);
          resolveErrorTypes(errorTypeRepository, innerComponent, syntheticErrorNamespaces);
        });

    registerErrorMappings(errorTypeRepository, componentModel, syntheticErrorNamespaces);
  }

  private static void registerErrorMappings(ErrorTypeRepository errorTypeRepository, ComponentAst componentModel,
                                            Set<String> syntheticErrorNamespaces) {
    forEachErrorMappingDo(componentModel, mappings -> mappings
        .forEach(mapping -> resolveErrorType(errorTypeRepository, mapping.getTarget(), syntheticErrorNamespaces)));
  }

  private static void resolveErrorType(ErrorTypeRepository errorTypeRepository, String representation,
                                       Set<String> syntheticErrorNamespaces) {
    if (StringUtils.isEmpty(representation)) {
      return;
    }

    ComponentIdentifier errorIdentifier = parserErrorType(representation);
    String namespace = errorIdentifier.getNamespace();

    if (CORE_ERROR_NS.equals(namespace)) {
      return;
    }

    Optional<ErrorType> optionalErrorType = errorTypeRepository.lookupErrorType(errorIdentifier);
    if (syntheticErrorNamespaces.contains(namespace) && optionalErrorType.isPresent()) {
      return;
    } else {
      syntheticErrorNamespaces.add(namespace);
    }

    LOGGER.debug("Registering errorType '{}'", errorIdentifier);
    errorTypeRepository.addErrorType(errorIdentifier, getCoreErrorTypeRepo().getAnyErrorType());
  }

  private static ComponentIdentifier parserErrorType(String representation) {
    int separator = representation.indexOf(':');
    String namespace;
    String identifier;
    if (separator > 0) {
      namespace = representation.substring(0, separator).toUpperCase();
      identifier = representation.substring(separator + 1).toUpperCase();
    } else {
      namespace = CORE_ERROR_NS;
      identifier = representation.toUpperCase();
    }

    return createIdentifier(identifier, namespace);
  }

  /**
   * For the given AST node representing an operation, execute the given {@code action} for each error mapping it has.
   *
   * @param operation the operation from which to iterate the error mappings.
   * @param action    what is executed for every error mapping.
   */
  private static void forEachErrorMappingDo(ComponentAst operation, Consumer<List<ErrorMapping>> action) {
    operation.getModel(OperationModel.class).ifPresent(opModel -> {
      if (!opModel.getModelProperty(NoErrorMappingModelProperty.class).isPresent()) {
        final ComponentParameterAst errorMappingsParam = operation.getParameter(ERROR_MAPPINGS, ERROR_MAPPINGS_PARAMETER_NAME);
        if (errorMappingsParam != null) {
          if (LOGGER.isTraceEnabled()) {
            LOGGER.trace("Registering errorTypes from operation error-mapping's @ '{}'...",
                         operation.getLocation().getLocation());
          }

          errorMappingsParam.<List<ErrorMapping>>getValue().applyRight(action);
        }
      }
    });
  }

  private static void processRaiseError(ErrorTypeRepository errorTypeRepository, ComponentAst componentModel,
                                        Set<String> syntheticErrorNamespaces) {
    if (componentModel.getIdentifier().equals(RAISE_ERROR_IDENTIFIER)) {
      if (LOGGER.isTraceEnabled()) {
        LOGGER.trace("Registering errorType from raise-error @ '{}'...", componentModel.getLocation().getLocation());
      }

      final ComponentParameterAst parameter = componentModel.getParameter(DEFAULT_GROUP_NAME, ERROR_TYPE_PARAM);

      if (parameter != null) {
        parameter.getValue().getValue()
            .map(r -> (String) r)
            .ifPresent(representation -> resolveErrorType(errorTypeRepository, representation, syntheticErrorNamespaces));
      }
    }
  }

  private static ErrorType getErrorType(ErrorTypeRepository errorTypeRepository, ErrorModel errorModel,
                                        ExtensionModel extensionModel) {
    ComponentIdentifier identifier = createIdentifier(errorModel.getType(), errorModel.getNamespace());
    Optional<ErrorType> optionalError = errorTypeRepository.getErrorType(identifier);
    return optionalError
        .orElseGet(() -> getCoreErrorTypeRepo().getErrorType(identifier)
            .orElseGet(() -> createErrorType(errorTypeRepository, errorModel, identifier,
                                             extensionModel)));
  }

  private static ComponentIdentifier createIdentifier(String name, String namespace) {
    return builder().name(name).namespace(namespace).build();
  }

  private static ErrorType createErrorType(ErrorTypeRepository errorTypeRepository, ErrorModel errorModel,
                                           ComponentIdentifier identifier, ExtensionModel extensionModel) {
    final ErrorType errorType;

    if (identifier.getNamespace().equals(CORE_ERROR_NS)) {
      throw new MuleRuntimeException(createStaticMessage(format("The extension [%s] tried to register the [%s] error with [%s] namespace, which is not allowed.",
                                                                extensionModel.getName(), identifier, CORE_ERROR_NS)));
    }

    LOGGER.debug("Registering errorType '{}'", identifier);
    Optional<ErrorModel> optionalErrorModel = errorModel.getParent();
    if (optionalErrorModel.isPresent()) {
      errorType = errorTypeRepository
          .addErrorType(identifier,
                        getErrorType(errorTypeRepository, optionalErrorModel.get(), extensionModel));
    } else {
      errorType = errorTypeRepository.addErrorType(identifier, null);
    }
    return errorType;
  }
}
