/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.model;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static java.util.Optional.of;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.collection.IsMapContaining.hasKey;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.spy;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.annotation.TypeIdAnnotation;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.model.impl.DefaultObjectType;
import org.mule.metadata.java.api.annotation.ClassInformationAnnotation;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.SubTypesModel;
import org.mule.runtime.api.meta.model.XmlDslModel;
import org.mule.runtime.api.meta.model.construct.ConstructModel;
import org.mule.runtime.api.meta.model.nested.NestableElementModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.ast.internal.model.ExtensionModelHelper.ExtensionWalkerModelDelegate;
import org.mule.runtime.extension.api.dsl.syntax.DslElementSyntax;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.junit.Before;
import org.junit.Test;

import io.qameta.allure.Description;
import io.qameta.allure.Issue;

public class ExtensionModelHelperTestCase {

  private ExtensionModel extensionModel;
  private ExtensionModelHelper extensionModelHelper;
  private ComponentIdentifier componentIdentifier;
  private ExtensionWalkerModelDelegate delegate;
  private OperationModel operationModel;
  private NestableElementModel nestableElementModel;
  private ConstructModel constructModel;

  @Before
  public void setup() {
    extensionModel = mock(ExtensionModel.class);
    when(extensionModel.getName()).thenReturn("SomeTestExtensionName");

    componentIdentifier = mock(ComponentIdentifier.class);
    when(componentIdentifier.getName()).thenReturn("nestable-that-is-also-a-construct");
    delegate = spy(mock(ExtensionWalkerModelDelegate.class));

    operationModel = mock(OperationModel.class);
    when(operationModel.getName()).thenReturn("SomeOperationName");

    when(extensionModel.getOperationModels()).thenReturn(singletonList(operationModel));

    XmlDslModel xmlDslModel = XmlDslModel.builder().setNamespace("ns").setPrefix("prefix").build();
    when(extensionModel.getXmlDslModel()).thenReturn(xmlDslModel);

    nestableElementModel = mock(NestableElementModel.class);
    List allNestedComponents = singletonList(nestableElementModel);
    when(nestableElementModel.getName()).thenReturn("NestableThatIsAlsoAConstruct");
    when(operationModel.getNestedComponents()).thenReturn(allNestedComponents);

    constructModel = mock(ConstructModel.class);
    when(constructModel.getName()).thenReturn("NestableThatIsAlsoAConstruct");
    when(extensionModel.getConstructModel(eq("NestableThatIsAlsoAConstruct"))).thenReturn(of(constructModel));
    when(extensionModel.getConstructModels()).thenReturn(singletonList(constructModel));
  }

  @Test
  @Issue("MULE-19570")
  @Description("When a nestable and a construct have the same name, the onConstruct is called and the onNestableElement is not")
  public void whenANestableAndAConstructHaveTheSameNameTheOnConstructIsCalledAndTheOnNestableElementIsNot() {
    extensionModelHelper = new ExtensionModelHelper(singleton(extensionModel));
    extensionModelHelper.walkToComponent(componentIdentifier, false, delegate, extensionModel);

    verify(delegate, never()).onNestableElement(eq(nestableElementModel));
    verify(delegate, times(1)).onConstruct(eq(constructModel));
  }

  @Test
  @Issue("W-10920842")
  public void nonInstantiableTypesFilteredFromSubTypes() {
    Set<SubTypesModel> subTypes = new HashSet<>();
    DefaultObjectType superInterface = createObjectType(false, "SuperInterface");
    DefaultObjectType subType = createObjectType(true, "SubType");
    subTypes.add(new SubTypesModel(superInterface,
                                   singleton(subType)));
    when(extensionModel.getSubTypes()).thenReturn(subTypes);
    when(extensionModel.getTypes()).thenReturn(new HashSet<>(asList(superInterface, subType)));

    extensionModelHelper = new ExtensionModelHelper(singleton(extensionModel));
    Map<ObjectType, Optional<DslElementSyntax>> resolvedSubTypes = extensionModelHelper.resolveSubTypes(superInterface);

    assertThat(resolvedSubTypes, hasKey(subType));
    assertThat(resolvedSubTypes, not(hasKey(superInterface)));
  }

  protected DefaultObjectType createObjectType(boolean isInstantiable, String className) {
    Map<Class<? extends TypeAnnotation>, TypeAnnotation> extensions = new HashMap<>();
    extensions.put(TypeIdAnnotation.class, new TypeIdAnnotation(className));
    extensions.put(ClassInformationAnnotation.class,
                   new ClassInformationAnnotation(className, false, false, isInstantiable,
                                                  false, false, emptyList(),
                                                  null, emptyList(), false));

    return new DefaultObjectType(emptySet(), false, null, null, extensions);
  }
}
