/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.error;

import static java.lang.String.format;
import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toSet;
import static java.util.stream.Stream.concat;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.api.message.ErrorType;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

/**
 * Repository for the different {@link ErrorType Error Types} in a mule artifact.
 *
 * Only one instance of {@link ErrorType} must exists describing the same combination of error identifier and namespace.
 *
 * @since 1.0
 */
public class DefaultErrorTypeRepository implements ErrorTypeRepository {

  private final Map<ComponentIdentifier, ErrorType> errorTypes = new HashMap<>();
  private final Map<ComponentIdentifier, ErrorType> internalErrorTypes = new HashMap<>();

  @Override
  public ErrorType addErrorType(ComponentIdentifier errorTypeIdentifier, ErrorType parentErrorType) {
    ErrorType errorType = buildErrorType(errorTypeIdentifier, parentErrorType);
    errorTypes.put(errorTypeIdentifier, errorType);
    return errorType;
  }

  @Override
  public ErrorType addInternalErrorType(ComponentIdentifier errorTypeIdentifier, ErrorType parentErrorType) {
    ErrorType errorType = buildErrorType(errorTypeIdentifier, parentErrorType);
    internalErrorTypes.put(errorTypeIdentifier, errorType);
    return errorType;
  }

  private ErrorType buildErrorType(ComponentIdentifier identifier, ErrorType parent) {
    if (errorTypes.containsKey(identifier) || internalErrorTypes.containsKey(identifier)) {
      throw new IllegalStateException(format("An error type with identifier '%s' already exists", identifier));
    }
    return ErrorTypeBuilder.builder()
        .namespace(identifier.getNamespace())
        .identifier(identifier.getName())
        .parentErrorType(parent)
        .build();
  }

  @Override
  public Optional<ErrorType> lookupErrorType(ComponentIdentifier errorTypeComponentIdentifier) {
    return ofNullable(this.errorTypes.get(errorTypeComponentIdentifier));
  }

  @Override
  public Optional<ErrorType> getErrorType(ComponentIdentifier errorTypeIdentifier) {
    Optional<ErrorType> errorType = lookupErrorType(errorTypeIdentifier);
    if (!errorType.isPresent()) {
      errorType = ofNullable(this.internalErrorTypes.get(errorTypeIdentifier));
    }
    return errorType;
  }

  @Override
  public Collection<String> getErrorNamespaces() {
    return concat(this.errorTypes.keySet().stream(), this.internalErrorTypes.keySet().stream())
        .map(id -> id.getNamespace().toUpperCase())
        .collect(toSet());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ErrorType getAnyErrorType() {
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ErrorType getSourceErrorType() {
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ErrorType getSourceResponseErrorType() {
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ErrorType getCriticalErrorType() {
    return null;
  }


  @Override
  public Set<ErrorType> getErrorTypes() {
    return new HashSet<>(errorTypes.values());
  }

  @Override
  public Set<ErrorType> getInternalErrorTypes() {
    return new HashSet<>(internalErrorTypes.values());
  }
}
