/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.model;

import static org.mule.metadata.api.utils.MetadataTypeUtils.getTypeId;
import static org.mule.runtime.api.component.ComponentIdentifier.builder;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.CHAIN;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.ERROR_HANDLER;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.FLOW;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.ON_ERROR;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.OPERATION;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.ROUTE;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.ROUTER;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.SCOPE;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.SOURCE;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.UNKNOWN;
import static org.mule.runtime.extension.api.util.ExtensionMetadataTypeUtils.getSubstitutionGroup;
import static org.mule.runtime.internal.dsl.DslConstants.CORE_PREFIX;
import static org.mule.runtime.internal.dsl.DslConstants.TLS_PREFIX;

import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toMap;

import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.java.api.JavaTypeLoader;
import org.mule.metadata.java.api.annotation.ClassInformationAnnotation;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.TypedComponentIdentifier;
import org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType;
import org.mule.runtime.api.dsl.DslResolvingContext;
import org.mule.runtime.api.meta.NamedObject;
import org.mule.runtime.api.meta.model.ComponentModel;
import org.mule.runtime.api.meta.model.ComponentModelVisitor;
import org.mule.runtime.api.meta.model.ComposableModel;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.config.ConfigurationModel;
import org.mule.runtime.api.meta.model.connection.ConnectionProviderModel;
import org.mule.runtime.api.meta.model.construct.ConstructModel;
import org.mule.runtime.api.meta.model.nested.NestableElementModel;
import org.mule.runtime.api.meta.model.nested.NestedChainModel;
import org.mule.runtime.api.meta.model.nested.NestedComponentModel;
import org.mule.runtime.api.meta.model.nested.NestedRouteModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.api.meta.model.util.ExtensionWalker;
import org.mule.runtime.api.meta.model.util.IdempotentExtensionWalker;
import org.mule.runtime.api.util.Pair;
import org.mule.runtime.api.util.Reference;
import org.mule.runtime.extension.api.declaration.type.ExtensionsTypeHandlerManagerFactory;
import org.mule.runtime.extension.api.dsl.syntax.DslElementSyntax;
import org.mule.runtime.extension.api.dsl.syntax.resolver.DslSyntaxResolver;
import org.mule.runtime.extension.api.stereotype.MuleStereotypes;

import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;

/**
 * Helper class to work with a set of {@link ExtensionModel}s
 * <p/>
 * Contains a cache for searches within the extension models so we avoid processing each extension model twice.
 * <p/>
 * It's recommended that the application only has one instance of this class to avoid processing the extension models several
 * times.
 * <p>
 * since 4.0
 */
public class ExtensionModelHelper {

  private static final String ON_ERROR_MODEL = "onError";
  private static final String ON_ERROR_CONTINUE_MODEL = "onErrorContinue";
  private static final String ON_ERROR_PROPAGATE_MODEL = "onErrorPropagate";

  private static final String ON_ERROR_CONTINUE = "on-error-continue";
  private static final String ON_ERROR_PROPAGATE = "on-error-propagate";

  public static final ComponentIdentifier ON_ERROR_CONTINUE_IDENTIFIER =
      builder().namespace(CORE_PREFIX).name(ON_ERROR_CONTINUE).build();
  public static final ComponentIdentifier ON_ERROR_PROPAGATE_IDENTIFIER =
      builder().namespace(CORE_PREFIX).name(ON_ERROR_PROPAGATE).build();

  private final Set<ExtensionModel> extensionsModels;
  private final Cache<ComponentIdentifier, Optional<ComponentModel>> extensionComponentModelByComponentIdentifier =
      Caffeine.newBuilder().build();
  private final Cache<ComponentIdentifier, Optional<ConnectionProviderModel>> extensionConnectionProviderModelByComponentIdentifier =
      Caffeine.newBuilder().build();
  private final Cache<ComponentIdentifier, Optional<ConfigurationModel>> extensionConfigurationModelByComponentIdentifier =
      Caffeine.newBuilder().build();
  private final Optional<DslSyntaxResolver> muleTopLevelDslSyntaxResolver;
  private final LoadingCache<ExtensionModel, DslSyntaxResolver> dslSyntaxResolversByExtension;

  private final JavaTypeLoader javaTypeLoader = new JavaTypeLoader(ExtensionModelHelper.class.getClassLoader(),
                                                                   new ExtensionsTypeHandlerManagerFactory());
  private final DslResolvingContext dslResolvingContext;

  /**
   * @param extensionModels the set of {@link ExtensionModel}s to work with. Usually this is the set of models configured within a
   *                        mule artifact.
   */
  public ExtensionModelHelper(Set<ExtensionModel> extensionModels) {
    this(extensionModels, DslResolvingContext.getDefault(extensionModels));
  }

  public ExtensionModelHelper(Set<ExtensionModel> extensionModels, DslResolvingContext dslResolvingCtx) {
    this.extensionsModels = extensionModels;
    this.muleTopLevelDslSyntaxResolver =
        extensionModels.stream()
            .filter(extModel -> extModel.getName().equals(CORE_PREFIX))
            .map(extModel -> DslSyntaxResolver.getDefault(extModel, dslResolvingCtx))
            .findAny();
    this.dslSyntaxResolversByExtension =
        Caffeine.newBuilder().build(key -> DslSyntaxResolver.getDefault(key, dslResolvingCtx));
    this.dslResolvingContext = dslResolvingCtx;
  }

  /**
   * Find a {@link ComponentType} for a given {@link ComponentModel}
   *
   * @param componentIdentifier the identifier to use for the search.
   * @return the {@link ComponentType} associated with the configuration or an {@link Optional#empty()} if there isn't one.
   */
  public ComponentType findComponentType(ComponentIdentifier componentIdentifier) {
    return findComponentModel(componentIdentifier)
        .map(this::findComponentType)
        .orElse(UNKNOWN);
  }

  public ComponentType findComponentType(ComponentModel extensionComponentModel) {
    Reference<ComponentType> componentTypeReference = new Reference<>();
    extensionComponentModel.accept(new ComponentModelVisitor() {

      @Override
      public void visit(OperationModel model) {
        componentTypeReference.set(OPERATION);
        determineFromNested(componentTypeReference, model);
      }

      @Override
      public void visit(SourceModel model) {
        componentTypeReference.set(SOURCE);
      }

      @Override
      public void visit(ConstructModel model) {
        if (model.getStereotype().equals(MuleStereotypes.ERROR_HANDLER)) {
          componentTypeReference.set(ERROR_HANDLER);
          return;
        }
        if (model.getStereotype().equals(MuleStereotypes.ON_ERROR)) {
          componentTypeReference.set(ON_ERROR);
          return;
        }
        if (model.getStereotype().equals(MuleStereotypes.FLOW)) {
          componentTypeReference.set(FLOW);
          return;
        }
        if (model.getStereotype().equals(MuleStereotypes.CHAIN)) {
          componentTypeReference.set(CHAIN);
          return;
        }
        determineFromNested(componentTypeReference, model);
      }

      protected void determineFromNested(Reference<ComponentType> componentTypeReference, ComposableModel model) {
        NestedComponentVisitor nestedComponentVisitor = new NestedComponentVisitor(componentTypeReference);
        for (NestableElementModel nestableElementModel : model.getNestedComponents()) {
          nestableElementModel.accept(nestedComponentVisitor);
          if (componentTypeReference.get() != null) {
            return;
          }
        }
      }

      @Override
      public void visit(NestedComponentModel model) {
        if (model.getAllowedStereotypes().contains(MuleStereotypes.ERROR_HANDLER)) {
          componentTypeReference.set(ERROR_HANDLER);
          return;
        }
        if (model.getAllowedStereotypes().contains(MuleStereotypes.ON_ERROR)) {
          componentTypeReference.set(ON_ERROR);
          return;
        }
      }

      @Override
      public void visit(NestedChainModel model) {
        componentTypeReference.set(CHAIN);
      }

      @Override
      public void visit(NestedRouteModel model) {
        componentTypeReference.set(ROUTE);
      }
    });
    return componentTypeReference.get() == null ? UNKNOWN : componentTypeReference.get();
  }

  /**
   * Finds a {@link ComponentModel} within the provided set of {@link ExtensionModel}s by a {@link ComponentIdentifier}.
   *
   * @param componentIdentifier the identifier to use for the search.
   * @return the found {@link ComponentModel} or {@link Optional#empty()} if it couldn't be found.
   */
  public Optional<ComponentModel> findComponentModel(ComponentIdentifier componentId) {
    return extensionComponentModelByComponentIdentifier
        .get(componentId,
             componentIdentifier -> lookupExtensionModelFor(componentIdentifier)
                 .flatMap(extensionModel -> {
                   AtomicReference<ComponentModel> modelRef = new AtomicReference<>();

                   new IdempotentExtensionWalker() {

                     final DslSyntaxResolver dslSyntaxResolver = dslSyntaxResolversByExtension.get(extensionModel);

                     @Override
                     protected void onOperation(OperationModel model) {
                       onModel(componentIdentifier, modelRef, model);
                     }

                     @Override
                     protected void onSource(SourceModel model) {
                       onModel(componentIdentifier, modelRef, model);
                     }

                     @Override
                     protected void onConstruct(ConstructModel model) {
                       onModel(componentIdentifier, modelRef, model);
                     }

                     @Override
                     protected void onNestable(NestableElementModel model) {
                       onModel(componentIdentifier, modelRef, model);
                     }

                     protected void onModel(ComponentIdentifier componentIdentifier,
                                            AtomicReference<ComponentModel> modelRef,
                                            ComponentModel model) {
                       if (dslSyntaxResolver.resolve(model).getElementName().equals(componentIdentifier.getName())) {
                         modelRef.set(model);
                       }
                     }

                   }.walk(extensionModel);

                   return ofNullable(modelRef.get());
                 }));
  }

  /**
   * Finds a {@link org.mule.runtime.api.meta.model.ConnectionProviderModel} within the provided set of {@link ExtensionModel}s by
   * a {@link ComponentIdentifier}.
   *
   * @param componentIdentifier the identifier to use for the search.
   * @return the found {@link org.mule.runtime.api.meta.model.ConnectionProviderModel} or {@link Optional#empty()} if it couldn't
   *         be found.
   */
  public Optional<ConnectionProviderModel> findConnectionProviderModel(ComponentIdentifier componentId) {
    return extensionConnectionProviderModelByComponentIdentifier
        .get(componentId, componentIdentifier -> lookupExtensionModelFor(componentIdentifier)
            .flatMap(currentExtension -> {
              AtomicReference<ConnectionProviderModel> modelRef = new AtomicReference<>();
              new IdempotentExtensionWalker() {

                final DslSyntaxResolver dslSyntaxResolver = dslSyntaxResolversByExtension.get(currentExtension);

                @Override
                protected void onConnectionProvider(ConnectionProviderModel model) {
                  if (dslSyntaxResolver.resolve(model).getElementName().equals(componentIdentifier.getName())) {
                    modelRef.set(model);
                  }
                }
              }.walk(currentExtension);

              return ofNullable(modelRef.get());
            }));
  }

  /**
   * Finds a {@link org.mule.runtime.api.meta.model.ConfigurationModel} within the provided set of {@link ExtensionModel}s by a
   * {@link ComponentIdentifier}.
   *
   * @param componentIdentifier the identifier to use for the search.
   * @return the found {@link org.mule.runtime.api.meta.model.ConfigurationModel} or {@link Optional#empty()} if it couldn't be
   *         found.
   */
  public Optional<ConfigurationModel> findConfigurationModel(ComponentIdentifier componentId) {
    return extensionConfigurationModelByComponentIdentifier
        .get(componentId, componentIdentifier -> lookupExtensionModelFor(componentIdentifier)
            .flatMap(currentExtension -> {
              AtomicReference<ConfigurationModel> modelRef = new AtomicReference<>();

              new IdempotentExtensionWalker() {

                final DslSyntaxResolver dslSyntaxResolver = dslSyntaxResolversByExtension.get(currentExtension);

                @Override
                protected void onConfiguration(ConfigurationModel model) {
                  if (dslSyntaxResolver.resolve(model).getElementName().equals(componentIdentifier.getName())) {
                    modelRef.set(model);
                  }
                }

              }.walk(currentExtension);

              return ofNullable(modelRef.get());
            }));
  }

  public Optional<ParameterModel> findParameterModel(ComponentIdentifier nestedComponentId,
                                                     ParameterizedModel model) {
    return lookupExtensionModelFor(nestedComponentId)
        .flatMap(currentExtension -> {
          final DslSyntaxResolver dslSyntaxResolver = dslSyntaxResolversByExtension.get(currentExtension);

          return model.getAllParameterModels()
              .stream()
              .filter(pm -> dslSyntaxResolver.resolve(pm).getElementName().equals(nestedComponentId.getName()))
              .findAny();
        });
  }

  /**
   * Navigates the extension model for the provided {@code componentIdentifier} and calls the corresponding method on the provided
   * {@code delegate} when found.
   *
   * @param componentIdentifier the identifier to use for the search.
   * @param topLevel            whether the element with the given identifier is top-level (true) or nested (false).
   * @param delegate            the callback to execute on the found model.
   * @param currentExtension    the extension that declares the component with the given {@code compoenntIdentifier}.
   */
  public void walkToComponent(ComponentIdentifier componentIdentifier, boolean topLevel, ExtensionWalkerModelDelegate delegate,
                              ExtensionModel currentExtension) {
    new IdempotentExtensionWalker() {

      final DslSyntaxResolver dslSyntaxResolver = dslSyntaxResolversByExtension.get(currentExtension);
      final List<ConstructModel> overriders = currentExtension.getConstructModels();

      @Override
      protected void onConfiguration(ConfigurationModel model) {
        if (dslSyntaxResolver.resolve(model).getElementName().equals(componentIdentifier.getName())) {
          delegate.onConfiguration(model);
          stop();
        }
      }

      @Override
      protected void onConnectionProvider(ConnectionProviderModel model) {
        if (dslSyntaxResolver.resolve(model).getElementName().equals(componentIdentifier.getName())) {
          delegate.onConnectionProvider(model);
          stop();
        }
      }

      @Override
      protected void onOperation(OperationModel model) {
        if (dslSyntaxResolver.resolve(model).getElementName().equals(componentIdentifier.getName())) {
          delegate.onOperation(model);
          stop();
        }
      }

      @Override
      protected void onSource(SourceModel model) {
        if (dslSyntaxResolver.resolve(model).getElementName().equals(componentIdentifier.getName())) {
          delegate.onSource(model);
          stop();
        }
      }

      @Override
      protected void onConstruct(ConstructModel model) {
        final DslElementSyntax dslSyntax = muleTopLevelDslSyntaxResolver
            .map(resolver -> resolver.resolve(model))
            .orElseGet(() -> dslSyntaxResolver.resolve(model));

        if (dslSyntax.getElementName().equals(componentIdentifier.getName())) {
          delegate.onConstruct(model);
          stop();
        }
      }

      private Optional<ConstructModel> findOnErrorOverrider(NestableElementModel model) {
        if (!ON_ERROR_MODEL.equals(model.getName())) {
          return empty();
        }
        return overriders.stream().filter(e -> e.getName().equals(model.getName())).findFirst();
      }

      @Override
      protected void onNestable(ComposableModel owner, NestableElementModel model) {
        if (!topLevel && dslSyntaxResolver.resolve(model).getElementName().equals(componentIdentifier.getName())) {
          Optional<ConstructModel> overrider = findOnErrorOverrider(model);
          if (overrider.isPresent()) {
            delegate.onConstruct(overrider.get());
          } else {
            if (shouldSkipNestable(model, currentExtension)) {
              return;
            }
            delegate.onNestableElement(model);
          }
          stop();
        }
      }

      /**
       * A {@code NestableElementModel} must be skipped during the extension walking if there is some construct with the same
       * name. This situation where there exist both a nestable component and a construct with the same name is present in several
       * components as a hack to add parameters in nestable components, which is not supported yet. See MULE-19570 and MULE-19579.
       * We don't want to skip the {@code onErrorContinue} and {@code onErrorPropagate} because we want to interpret them as
       * nestable.
       *
       * @param model            The nestable element model.
       * @param currentExtension The extension that we are walking.
       *
       * @return {@code true} if the element should be skipped given the mentioned condition, or {@code false} otherwise.
       */
      private boolean shouldSkipNestable(NestableElementModel model, ExtensionModel currentExtension) {
        if (ON_ERROR_PROPAGATE_MODEL.equals(model.getName())) {
          return false;
        }

        if (ON_ERROR_CONTINUE_MODEL.equals(model.getName())) {
          return false;
        }

        return currentExtension.getConstructModel(model.getName()).isPresent();
      }

      @Override
      protected void onType(MetadataType type) {
        resolveDslElementModel(type, currentExtension)
            .filter(dsl -> builder().name(dsl.getElementName()).namespace(dsl.getPrefix()).build()
                .equals(componentIdentifier))
            .ifPresent(dsl -> {
              if (walkableType(type, dsl)) {
                delegate.onType(type);
              }
              stop();
            });
      }

    }.walk(currentExtension);
  }

  private boolean walkableType(MetadataType type, DslElementSyntax dsl) {
    return dsl.supportsTopLevelDeclaration()
        || (dsl.supportsChildDeclaration() && dsl.isWrapped())
        || getSubstitutionGroup(type).isPresent()
        || getAllSubTypes().contains(type);
  }

  public Optional<MetadataType> findMetadataType(Class<?> type) {
    if (type != null) {
      return of(javaTypeLoader.load(type));
    } else {
      return empty();
    }
  }

  public Optional<ExtensionModel> lookupExtensionModelFor(ComponentIdentifier componentIdentifier) {
    return lookupExtensionModelFor(componentIdentifier.getNamespace());
  }

  private Optional<ExtensionModel> lookupExtensionModelFor(String namespacePrefix) {
    return extensionsModels.stream()
        .filter(e -> e.getXmlDslModel().getPrefix().equals(namespacePrefix.equals(TLS_PREFIX) ? CORE_PREFIX : namespacePrefix))
        .findFirst();
  }

  public Optional<DslElementSyntax> resolveDslElementModel(MetadataType type, String namespacePrefix) {
    final DslSyntaxResolver dslSyntaxResolver = getDslSyntaxResolver(namespacePrefix);

    return dslSyntaxResolver.resolve(type);
  }

  public DslElementSyntax resolveDslElementModel(NamedObject component, ComponentIdentifier componentIdentifier) {
    final DslSyntaxResolver dslSyntaxResolver = getDslSyntaxResolver(componentIdentifier);

    return dslSyntaxResolver.resolve(component);
  }

  private DslSyntaxResolver getDslSyntaxResolver(ComponentIdentifier componentIdentifier) {
    Optional<ExtensionModel> optionalExtensionModel = lookupExtensionModelFor(componentIdentifier);
    ExtensionModel extensionModel = optionalExtensionModel
        .orElseThrow(() -> new IllegalStateException("Extension Model in context not present for componentIdentifier: "
            + componentIdentifier));

    return dslSyntaxResolversByExtension.get(extensionModel);
  }

  private DslSyntaxResolver getDslSyntaxResolver(String namespacePrefix) {
    Optional<ExtensionModel> optionalExtensionModel = lookupExtensionModelFor(namespacePrefix);
    ExtensionModel extensionModel = optionalExtensionModel
        .orElseThrow(() -> new IllegalStateException("Extension Model in context not present for namespace: "
            + namespacePrefix));

    return dslSyntaxResolversByExtension.get(extensionModel);
  }

  public Optional<DslElementSyntax> resolveDslElementModel(MetadataType metadataType, ExtensionModel extensionModel) {
    return dslSyntaxResolversByExtension.get(extensionModel).resolve(metadataType);
  }

  public DslElementSyntax resolveDslElementModel(ParameterModel parameterModel, ComponentIdentifier componentIdentifier) {
    final DslSyntaxResolver dslSyntaxResolver = getDslSyntaxResolver(componentIdentifier);
    return dslSyntaxResolver.resolve(parameterModel);
  }

  /**
   * Note: Only instantiable subtypes will be returned by this method!
   * 
   * @param type the type to resolve subtypes for
   * @return the subtypes along with their {@link DslElementSyntax}, if available.
   */
  public Map<ObjectType, Optional<DslElementSyntax>> resolveSubTypes(ObjectType type) {
    // Make iteration order consistent regardless of the hashCode of the types
    Map<Pair<ExtensionModel, String>, ObjectType> singletonizedSubTypes = new LinkedHashMap<>();

    getTypeId(type)
        .ifPresent(typeId -> dslResolvingContext.getTypeCatalog().getDeclaringExtension(typeId)
            .flatMap(dslResolvingContext::getExtension)
            .ifPresent(declaringExtension -> singletonizedSubTypes.putIfAbsent(new Pair<>(declaringExtension, typeId),
                                                                               type)));

    for (ObjectType subType : dslResolvingContext.getTypeCatalog().getSubTypes(type)) {
      getTypeId(subType)
          .ifPresent(subTypeId -> dslResolvingContext.getTypeCatalog().getDeclaringExtension(subTypeId)
              .flatMap(dslResolvingContext::getExtension)
              .ifPresent(declaringExtension -> singletonizedSubTypes.putIfAbsent(new Pair<>(declaringExtension, subTypeId),
                                                                                 subType)));
    }

    return singletonizedSubTypes.entrySet().stream()
        .filter(t -> t.getValue().getAnnotation(ClassInformationAnnotation.class)
            .map(ClassInformationAnnotation::isInstantiable)
            // assume non-java types as instantiable (i.e.: a type derived form a json schema)
            .orElse(true))
        .collect(toMap(Entry::getValue,
                       e -> dslSyntaxResolversByExtension.get(e.getKey().getFirst())
                           .resolve(e.getValue()),
                       (x, y) -> x,
                       // Make iteration order consistent regardless of the hashCode of the types
                       LinkedHashMap::new));
  }

  private Collection<ObjectType> getAllSubTypes() {
    return dslResolvingContext.getTypeCatalog().getAllSubTypes();
  }

  /**
   * This interface is used along with an ExtensionWalker. The {@link ExtensionWalker} makes same validation/filter and then calls
   * the appropriate method form this interface if applicable.
   *
   * @since 4.3
   */
  public interface ExtensionWalkerModelDelegate {

    void onConfiguration(ConfigurationModel model);

    void onConnectionProvider(ConnectionProviderModel model);

    void onOperation(OperationModel model);

    void onSource(SourceModel model);

    void onConstruct(ConstructModel model);

    void onNestableElement(NestableElementModel model);

    void onType(MetadataType type);
  }

  /**
   * Visitor of {@link ConstructModel} that determines it
   * {@link org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType}
   */
  static class NestedComponentVisitor implements ComponentModelVisitor {

    private final Reference<TypedComponentIdentifier.ComponentType> reference;

    public NestedComponentVisitor(Reference<TypedComponentIdentifier.ComponentType> reference) {
      this.reference = reference;
    }

    @Override
    public void visit(NestedChainModel component) {
      reference.set(SCOPE);
    }

    @Override
    public void visit(NestedRouteModel component) {
      reference.set(ROUTER);
    }

    @Override
    public void visit(NestedComponentModel component) {
      // nothing to do
    }

    @Override
    public void visit(OperationModel model) {
      // nothing to do
    }

    @Override
    public void visit(SourceModel model) {
      // nothing to do
    }

    @Override
    public void visit(ConstructModel model) {
      // nothing to do
    }
  }

  public Set<ExtensionModel> getExtensionsModels() {
    return extensionsModels;
  }
}
