/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.builder;

import static java.util.Collections.emptyMap;
import static java.util.Collections.unmodifiableMap;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static org.apache.commons.lang3.StringUtils.capitalize;
import static org.mule.runtime.ast.internal.error.ErrorTypeBuilder.ANY_IDENTIFIER;
import static org.mule.runtime.extension.api.ExtensionConstants.ERROR_MAPPINGS_PARAMETER_NAME;
import static org.mule.runtime.extension.api.stereotype.MuleStereotypes.APP_CONFIG;
import static org.mule.runtime.extension.api.util.ExtensionMetadataTypeUtils.isMap;
import static org.mule.runtime.extension.api.util.NameUtils.singularize;
import static org.mule.runtime.internal.dsl.DslConstants.CORE_PREFIX;
import static org.mule.runtime.internal.dsl.DslConstants.ERROR_MAPPING_ELEMENT_IDENTIFIER;

import org.mule.metadata.api.model.ArrayType;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.visitor.MetadataTypeVisitor;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.meta.NamedObject;
import org.mule.runtime.api.meta.model.ComponentModel;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.config.ConfigurationModel;
import org.mule.runtime.api.meta.model.connection.ConnectionProviderModel;
import org.mule.runtime.api.meta.model.nested.NestableElementModel;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.api.meta.model.stereotype.HasStereotypeModel;
import org.mule.runtime.api.util.Pair;
import org.mule.runtime.ast.api.ComponentGenerationInformation;
import org.mule.runtime.ast.api.ComponentMetadataAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.ast.api.MetadataTypeAdapter;
import org.mule.runtime.ast.api.builder.ComponentAstBuilder;
import org.mule.runtime.ast.internal.DefaultComponentGenerationInformation;
import org.mule.runtime.ast.internal.DefaultComponentGenerationInformation.Builder;
import org.mule.runtime.ast.internal.DefaultComponentParameterAst;
import org.mule.runtime.extension.api.dsl.syntax.DslElementSyntax;
import org.mule.runtime.extension.api.error.ErrorMapping;
import org.mule.runtime.extension.api.property.NoWrapperModelProperty;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Stream;

public abstract class BaseComponentAstBuilder<B extends BaseComponentAstBuilder> implements ComponentAstBuilder {

  public static final String BODY_RAW_PARAM_NAME = "$%body%$";

  private static final String ERROR_HANDLER = "error-handler";
  private static final ComponentIdentifier ERROR_HANDLER_IDENTIFIER =
      ComponentIdentifier.builder().namespace(CORE_PREFIX).name(ERROR_HANDLER).build();

  private static final ComponentIdentifier ERROR_MAPPING_IDENTIFIER =
      ComponentIdentifier.builder().namespace(CORE_PREFIX).name(ERROR_MAPPING_ELEMENT_IDENTIFIER).build();
  public static final String SOURCE_TYPE = "sourceType";
  public static final String TARGET_TYPE = "targetType";
  private final PropertiesResolver propertiesResolver;

  private final List<B> childComponents = new ArrayList<>();
  private final Set<B> paramsChildren = new HashSet<>();

  private final Map<String, String> rawParameters = new HashMap<>();
  private final Map<Pair<ParameterModel, ParameterGroupModel>, ComponentParameterAst> parameters = new HashMap<>();
  private final Set<String> parameterNames = new HashSet<>();

  private final Map<String, Object> annotations = new HashMap<>();

  private ExtensionModel extensionModel;
  private ComponentModel componentModel;
  private NestableElementModel nestableElementModel;
  private ConfigurationModel configurationModel;
  private ConnectionProviderModel connectionProviderModel;
  private ParameterizedModel parameterizedModel;
  private MetadataType type;

  private ComponentMetadataAst metadata;
  private ComponentLocation location;
  private ComponentIdentifier identifier;
  private ComponentType componentType = ComponentType.UNKNOWN;

  private final DefaultComponentGenerationInformation.Builder generationInformationBuilder =
      DefaultComponentGenerationInformation.builder();

  public BaseComponentAstBuilder(PropertiesResolver propertiesResolver) {
    this.propertiesResolver = propertiesResolver;
  }

  public ComponentAstBuilder addChildComponent(B childComponent) {
    childComponents.add(childComponent);
    return this;
  }

  // parameters
  @Override
  public ComponentAstBuilder withRawParameter(String paramName, String paramRawValue) {
    this.rawParameters.put(paramName, paramRawValue);
    return this;
  }

  @Override
  public ComponentAstBuilder withBodyParameter(String parameterRawValue) {
    return this.withRawParameter(BODY_RAW_PARAM_NAME, parameterRawValue);
  }

  @Override
  public ComponentAstBuilder withAnnotation(String name, Object value) {
    this.annotations.put(name, value);
    return this;
  }

  public BaseComponentAstBuilder<B> withParameter(ParameterModel paramModel, ParameterGroupModel parameterGroupModel,
                                                  ComponentParameterAst paramValue,
                                                  Optional<ComponentIdentifier> paramIdentifier) {
    paramIdentifier.ifPresent(this::doRemoveParamChild);

    this.parameterNames.add(paramModel.getName());
    Pair<ParameterModel, ParameterGroupModel> paramAndGroupPair = new Pair<>(paramModel, parameterGroupModel);
    if (this.parameters.put(paramAndGroupPair, paramValue) != null) {
      throw new IllegalStateException("Param '" + paramModel.getName() + "' is already present in " + getIdentifier() + " @ "
          + getMetadata().getStartLine());
    }
    return this;
  }

  protected void prepareForBuild() {
    getModel(ParameterizedModel.class)
        .ifPresent(pmzd -> pmzd.getParameterGroupModels()
            .forEach(pmg -> pmg.getParameterModels()
                .forEach(pm -> removeParamChild(pm, pmg))));

    getModel(SourceModel.class)
        .ifPresent(sm -> {
          sm.getSuccessCallback()
              .ifPresent(scbk -> scbk.getParameterGroupModels()
                  .forEach(pmg -> pmg.getParameterModels()
                      .forEach(pm -> removeParamChild(pm, pmg))));

          sm.getErrorCallback()
              .ifPresent(ecbk -> ecbk.getParameterGroupModels()
                  .forEach(pmg -> pmg.getParameterModels()
                      .forEach(pm -> removeParamChild(pm, pmg))));
        });

  }

  protected void removeParamChild(ParameterModel paramModel, ParameterGroupModel parameterGroupModel) {
    AtomicReference<ComponentIdentifier> childToRemove = new AtomicReference<>();

    if (parameterGroupModel.isShowInDsl()) {
      final DslElementSyntax syntax = this.generationInformationBuilder.getSyntax();
      if (syntax != null) {
        syntax.getChild(parameterGroupModel.getName())
            .map(syntaxToComponentIdentifier())
            .ifPresent(childToRemove::set);
      }
    } else {
      final DslElementSyntax syntax = this.generationInformationBuilder.getSyntax();
      if (syntax != null) {
        paramModel.getType().accept(new MetadataTypeVisitor() {

          @Override
          public void visitObject(ObjectType objectType) {
            if (isMap(objectType)) {
              syntax.getChild(paramModel.getName())
                  .map(s -> paramModel.getModelProperty(NoWrapperModelProperty.class)
                      .flatMap(noWrapper -> objectType.getOpenRestriction()
                          .flatMap(r -> s.getGeneric(r)))
                      .orElse(s))
                  .map(syntaxToComponentIdentifier())
                  .ifPresent(childToRemove::set);
            } else {
              syntax.getChild(paramModel.getName())
                  .filter(x -> x.isWrapped())
                  .map(syntaxToComponentIdentifier())
                  .ifPresent(childToRemove::set);
            }
          }

          @Override
          public void visitArrayType(ArrayType arrayType) {
            arrayType.getType().accept(new MetadataTypeVisitor() {

              @Override
              public void defaultVisit(MetadataType metadataType) {
                syntax.getGeneric(metadataType)
                    .map(syntaxToComponentIdentifier())
                    .ifPresent(childToRemove::set);
              }
            });
          }
        });
      }
    }

    doRemoveParamChild(childToRemove.get());
  }

  private Function<? super DslElementSyntax, ? extends ComponentIdentifier> syntaxToComponentIdentifier() {
    return syntax -> ComponentIdentifier.builder()
        .name(syntax.getElementName())
        .namespace(syntax.getPrefix())
        .namespaceUri(syntax.getNamespace())
        .build();
  }

  protected void doRemoveParamChild(ComponentIdentifier componentIdentifier) {
    if (componentIdentifier != null) {
      final Iterator<B> iterator = childComponents.iterator();
      while (iterator.hasNext()) {
        B b = iterator.next();

        if (b.getIdentifier().equals(componentIdentifier)
            || singularizeComponentIdentifier(b.getIdentifier()).equals(componentIdentifier)) {
          iterator.remove();
          paramsChildren.add(b);
        }
      }
    }
  }

  private ComponentIdentifier singularizeComponentIdentifier(ComponentIdentifier identifier) {
    return ComponentIdentifier.builder()
        .name(singularize(identifier.getName()))
        .namespace(identifier.getNamespace())
        .namespaceUri(identifier.getNamespaceUri())
        .build();
  }

  public Map<String, String> getRawParameters() {
    return unmodifiableMap(rawParameters);
  }

  protected Map<Pair<ParameterModel, ParameterGroupModel>, ComponentParameterAst> createParameterAsts() {
    if (!getModel(ParameterizedModel.class).isPresent()) {
      return emptyMap();
    }

    final Map<Pair<ParameterModel, ParameterGroupModel>, ComponentParameterAst> parameterAsts = new HashMap<>(this.parameters);

    getModel(ParameterizedModel.class)
        .ifPresent(parameterized -> {
          this.parameterizedModel = parameterized;
          getModel(SourceModel.class)
              // For sources, we need to account for the case where parameters in the callbacks may have colliding names.
              // This logic ensures that the parameter fetching logic is consistent with the logic that handles this scenario in
              // previous implementations.
              .map(sourceModel -> Stream
                  .concat(parameterized.getParameterGroupModels().stream(),
                          Stream.concat(sourceModel.getSuccessCallback().map(cb -> cb.getParameterGroupModels().stream())
                              .orElse(Stream.empty()),
                                        sourceModel.getErrorCallback().map(cb -> cb.getParameterGroupModels().stream())
                                            .orElse(Stream.empty()))))
              .orElse(parameterized.getParameterGroupModels().stream())
              .forEach(pg -> {
                if (pg.isShowInDsl()) {
                  populateNestedParameterAsts(parameterAsts, pg);
                } else {
                  populateFlatGroupParameterAsts(parameterAsts, pg);
                }
              });
        });


    // Keep parameter order defined on parameterized model

    return parameterAsts.values().stream().sorted(new Comparator<ComponentParameterAst>() {

      final List<String> params = parameterizedModel.getAllParameterModels().stream().map(NamedObject::getName)
          .collect(toList());

      @Override
      public int compare(ComponentParameterAst o1, ComponentParameterAst o2) {
        return Integer.compare(params.indexOf(o1.getModel().getName()), params.indexOf(o2.getModel().getName()));
      }
    }).collect(toMap(k -> new Pair<>(k.getModel(), k.getGroupModel()), identity(), (u, v) -> u, LinkedHashMap::new));
  }

  private void populateNestedParameterAsts(final Map<Pair<ParameterModel, ParameterGroupModel>, ComponentParameterAst> parameterAsts,
                                           ParameterGroupModel pg) {
    final Optional<B> paramGroupComp = paramsChildren.stream()
        .filter(comp -> generationInformationBuilder.getSyntax()
            .getChild(pg.getName())
            .map(childSyntax -> childSyntax.getElementName().equals(comp.getIdentifier().getName()))
            .orElse(false))
        .findAny();

    if (paramGroupComp.isPresent()) {
      pg.getParameterModels()
          .forEach(paramModel -> populateParameterAst(parameterAsts,
                                                      ofNullable((String) paramGroupComp.get().getRawParameters()
                                                          .get(paramModel.getName())),
                                                      pg, paramModel));
    } else {
      pg.getParameterModels()
          .forEach(paramModel -> populateParameterAst(parameterAsts, empty(), pg, paramModel));
    }
  }

  private void populateFlatGroupParameterAsts(final Map<Pair<ParameterModel, ParameterGroupModel>, ComponentParameterAst> parameterAsts,
                                              ParameterGroupModel pg) {
    pg.getParameterModels().forEach(paramModel -> {
      if (ERROR_MAPPINGS_PARAMETER_NAME.equals(paramModel.getName())) {
        populateErrorMappings(parameterAsts, pg, paramModel);
      } else {
        populateParameterAst(parameterAsts, ofNullable(rawParameters.get(paramModel.getName())), pg, paramModel);
      }
    });
  }

  private void populateErrorMappings(final Map<Pair<ParameterModel, ParameterGroupModel>, ComponentParameterAst> parameterAsts,
                                     ParameterGroupModel pg, ParameterModel paramModel) {
    final List<ErrorMapping> errorMappings = childComponents.stream()
        .filter(child -> ERROR_MAPPING_IDENTIFIER.equals(child.getIdentifier()))
        .map(child -> new ErrorMapping(ofNullable((String) child.getRawParameters().get(SOURCE_TYPE))
            .orElse(ANY_IDENTIFIER), (String) child.getRawParameters().get(TARGET_TYPE)))
        .collect(toList());

    final Builder generationInfoBuilder = DefaultComponentGenerationInformation.builder();
    if (generationInformationBuilder.getSyntax() != null) {
      generationInformationBuilder.getSyntax().getChild(paramModel.getName())
          .ifPresent(generationInfoBuilder::withSyntax);
    }

    parameterAsts.put(new Pair<>(paramModel, pg),
                      new DefaultComponentParameterAst(errorMappings.isEmpty() ? null : errorMappings, paramModel, pg,
                                                       generationInfoBuilder.build(),
                                                       propertiesResolver));
  }

  private void populateParameterAst(Map<Pair<ParameterModel, ParameterGroupModel>, ComponentParameterAst> parameterAsts,
                                    Optional<String> rawValue,
                                    ParameterGroupModel pg, ParameterModel paramModel) {
    if (!parameterNames.contains(paramModel.getName())) {
      final Builder generationInfoBuilder = DefaultComponentGenerationInformation.builder();

      DslElementSyntax syntax = generationInformationBuilder.getSyntax();
      if (syntax != null) {
        if (pg.isShowInDsl()) {
          syntax = syntax.getChild(pg.getName()).orElse(syntax);
        }
        syntax.getContainedElement(paramModel.getName()).ifPresent(generationInfoBuilder::withSyntax);
      }
      final ComponentGenerationInformation generationInfo = generationInfoBuilder.build();

      parameterAsts.put(new Pair<>(paramModel, pg),
                        rawValue
                            .map(rawParamValue -> new DefaultComponentParameterAst(rawParamValue, paramModel, pg,
                                                                                   generationInfo, propertiesResolver))
                            .orElseGet(() -> new DefaultComponentParameterAst((String) null, paramModel, pg,
                                                                              generationInfo, propertiesResolver)));
    }
  }

  public Map<String, Object> getAnnotations() {
    return annotations;
  }

  /**
   * @param extensionModel the extension that declares the model that represents this compoennt.
   * @return this builder
   */
  public ComponentAstBuilder withExtensionModel(ExtensionModel extensionModel) {
    this.extensionModel = extensionModel;
    return this;
  }

  protected ExtensionModel getExtensionModel() {
    return extensionModel;
  }

  /**
   * Sets the model declaration of the target component.
   * <p>
   * This method is exclusive with {@link #withNestableElementModel(NestableElementModel)},
   * {@link #withConfigurationModel(ConfigurationModel)}, {@link #withConnectionProviderModel(ConnectionProviderModel)} and
   * {@link #withParameterizedModel(ParameterizedModel)}; only one of these may be called for a single builder.
   *
   * @param componentModel the model that represents this component.
   * @return this builder
   */
  public ComponentAstBuilder withComponentModel(ComponentModel componentModel) {
    validateSingleModelSet();
    this.componentModel = componentModel;
    return this;
  }

  /**
   * Sets the model declaration of the target component.
   * <p>
   * This method is exclusive with {@link #withComponentModel(ComponentModel)},
   * {@link #withConfigurationModel(ConfigurationModel)}, {@link #withConnectionProviderModel(ConnectionProviderModel)} and
   * {@link #withParameterizedModel(ParameterizedModel)}; only one of these may be called for a single builder.
   *
   * @param nestableElementModel the model that represents this component.
   * @return this builder
   */
  public ComponentAstBuilder withNestableElementModel(NestableElementModel nestableElementModel) {
    validateSingleModelSet();
    this.nestableElementModel = nestableElementModel;
    return this;
  }

  /**
   * Sets the model declaration of the target component.
   * <p>
   * This method is exclusive with {@link #withComponentModel(ComponentModel)},
   * {@link #withNestableElementModel(NestableElementModel)}, {@link #withConnectionProviderModel(ConnectionProviderModel)} and
   * {@link #withParameterizedModel(ParameterizedModel)}; only one of these may be called for a single builder.
   *
   * @param configurationModel the model that represents this component.
   * @return this builder
   */
  public ComponentAstBuilder withConfigurationModel(ConfigurationModel configurationModel) {
    validateSingleModelSet();
    this.configurationModel = configurationModel;
    return this;
  }

  /**
   * Sets the model declaration of the target component.
   * <p>
   * This method is exclusive with {@link #withComponentModel(ComponentModel)},
   * {@link #withNestableElementModel(NestableElementModel)}, {@link #withConfigurationModel(ConfigurationModel)} and
   * {@link #withParameterizedModel(ParameterizedModel)}; only one of these may be called for a single builder.
   *
   * @param connectionProviderModel the model that represents this component.
   * @return this builder
   */
  public ComponentAstBuilder withConnectionProviderModel(ConnectionProviderModel connectionProviderModel) {
    validateSingleModelSet();
    this.connectionProviderModel = connectionProviderModel;
    return this;
  }

  /**
   * Sets the model declaration of the target component.
   * <p>
   * This method is exclusive with {@link #withComponentModel(ComponentModel)},
   * {@link #withNestableElementModel(NestableElementModel)}, {@link #withConfigurationModel(ConfigurationModel)} and
   * {@link #withConnectionProviderModel(ConnectionProviderModel)}; only one of these may be called for a single builder.
   *
   * @param parameterizedModel the model that represents this component.
   * @return this builder
   */
  @Override
  public BaseComponentAstBuilder<B> withParameterizedModel(ParameterizedModel parameterizedModel) {
    validateSingleModelSet();
    this.parameterizedModel = parameterizedModel;

    if (parameterizedModel instanceof MetadataTypeAdapter) {
      this.type = ((MetadataTypeAdapter) parameterizedModel).getType();
    }

    return this;
  }

  protected void validateSingleModelSet() {
    getModel(Object.class)
        .ifPresent(model -> {
          throw new IllegalStateException("Model previously set: " + model);
        });
  }

  <M> Optional<M> getModel(Class<M> modelClass) {
    if (componentModel != null && modelClass.isInstance(componentModel)) {
      return of((M) componentModel);
    }

    if (configurationModel != null && modelClass.isInstance(configurationModel)) {
      return of((M) configurationModel);
    }

    if (connectionProviderModel != null && modelClass.isInstance(connectionProviderModel)) {
      return of((M) connectionProviderModel);
    }

    if (nestableElementModel != null && modelClass.isInstance(nestableElementModel)) {
      return of((M) nestableElementModel);
    }

    if (parameterizedModel != null && modelClass.isInstance(parameterizedModel)) {
      return of((M) parameterizedModel);
    }

    return empty();
  }

  public MetadataType getType() {
    return type;
  }

  @Override
  public ComponentAstBuilder withMetadata(ComponentMetadataAst metadata) {
    this.metadata = metadata;
    return this;
  }

  public ComponentMetadataAst getMetadata() {
    return metadata;
  }

  /**
   *
   * @param location the location of the component in the configuration.
   * @return this builder
   */
  public ComponentAstBuilder withLocation(ComponentLocation location) {
    this.location = location;
    return this;
  }

  /**
   *
   * @param componentType a general typification of the role of this component.
   * @return this builder
   */
  public ComponentAstBuilder withComponentType(ComponentType componentType) {
    this.componentType = componentType;
    return this;
  }

  public ComponentType getComponentType() {
    return componentType;
  }

  @Override
  public ComponentAstBuilder withIdentifier(ComponentIdentifier identifier) {
    this.identifier = identifier;
    return this;
  }

  public ComponentIdentifier getIdentifier() {
    return identifier;
  }

  public ComponentLocation getLocation() {
    return location;
  }

  public DefaultComponentGenerationInformation.Builder getGenerationInformation() {
    return generationInformationBuilder;
  }

  public Stream<B> childComponentsStream() {
    return childComponents.stream();
  }

  protected Set<B> getParamsChildren() {
    return paramsChildren;
  }

  protected Optional<String> resolveComponentId(final Map<Pair<ParameterModel, ParameterGroupModel>, ComponentParameterAst> parameterAsts) {
    final Optional<String> compIdFromParams = getModel(ParameterizedModel.class)
        .flatMap(paramzdModel -> paramzdModel.getAllParameterModels()
            .stream()
            .filter(ParameterModel::isComponentId)
            .findAny()
            .flatMap(paramModel -> {
              if (getIdentifier().equals(ERROR_HANDLER_IDENTIFIER)
                  && parameterAsts.entrySet().stream()
                      .anyMatch(e -> e.getKey().getFirst().getName().equals("ref")
                          && e.getValue().getValue().getValue().isPresent())) {
                return empty();
              }

              final ComponentParameterAst componentParameterAst = parameterAsts.entrySet().stream()
                  .filter(f -> f.getKey().getFirst().equals(paramModel))
                  .findAny().map(Map.Entry::getValue).orElse(null);

              if (componentParameterAst != null) {
                return ofNullable(componentParameterAst.getRawValue());
              } else {
                return empty();
              }
            }));

    if (!compIdFromParams.isPresent() && getModel(HasStereotypeModel.class)
        .map(hsm -> APP_CONFIG.equals(hsm.getStereotype()))
        .orElse(false)) {
      return getModel(NamedObject.class)
          .map(no -> "_mule" + capitalize(no.getName()));
    }

    return compIdFromParams;
  }

  protected Supplier<Optional<String>> componentId;

  protected void setComponentId(Supplier<Optional<String>> componentId) {
    this.componentId = componentId;
  }

  public Optional<String> getComponentId() {
    return componentId.get().map(propertiesResolver);
  }

  protected PropertiesResolver getPropertiesResolver() {
    return propertiesResolver;
  }

}
