/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.api.builder;

import org.mule.api.annotation.NoImplement;
import org.mule.runtime.ast.api.AstParserAttribute;
import org.mule.runtime.ast.api.ComponentMetadataAst;
import org.mule.runtime.ast.api.ImportedResource;

import java.net.URI;
import java.util.List;

/**
 * Provides a way of creating {@link ComponentMetadataAst} instances.
 *
 * @since 1.1
 */
@NoImplement
public interface ComponentMetadataAstBuilder {

  /**
   * Sets the file name for the metadata.
   *
   * @param fileName the config file name in which the owner component was defined.
   * @return This instance for chaining purposes.
   */
  ComponentMetadataAstBuilder setFileName(String fileName);

  /**
   * Sets the config file URI for the metadata.
   *
   * @param fileUri the config file URI in which the owner component was defined
   * @return This instance for chaining purposes.
   */
  ComponentMetadataAstBuilder setFileUri(URI fileUri);

  /**
   * Sets the import chain for the metadata.
   *
   * @param importChain a {@link List} containing an element for every {@code import} tag leading to the file containing the
   *                    owning component.
   * @return This instance for chaining purposes.
   */
  ComponentMetadataAstBuilder setImportChain(List<ImportedResource> importChain);

  /**
   * Sets the start line for the metadata.
   *
   * @param startLine the first line number in which the component was defined in the configuration file.
   * @return This instance for chaining purposes.
   */
  ComponentMetadataAstBuilder setStartLine(int startLine);

  /**
   * Sets the start column for the metadata.
   *
   * @param startColumn the start column in which the component was defined in the configuration file.
   * @return This instance for chaining purposes.
   */
  ComponentMetadataAstBuilder setStartColumn(int startColumn);

  /**
   * Sets the last line for the metadata.
   *
   * @param endLine the last line in which the component was defined in the configuration file.
   * @return This instance for chaining purposes.
   */
  ComponentMetadataAstBuilder setEndLine(int endLine);

  /**
   * Sets the end column for the metadata.
   *
   * @param endColumn the end column in which the component was defined in the configuration file.
   * @return This instance for chaining purposes.
   */
  ComponentMetadataAstBuilder setEndColumn(int endColumn);

  /**
   * Sets the source code for the metadata.
   *
   * @param sourceCode the source code associated with the owner component.
   * @return This instance for chaining purposes.
   */
  ComponentMetadataAstBuilder setSourceCode(String sourceCode);

  /**
   * Puts s document attribute value in the metadata.
   *
   * @param key   The key to later reference the given document attribute.
   * @param value The actual value of the attribute.
   * @return This instance for chaining purposes.
   */
  ComponentMetadataAstBuilder putDocAttribute(String key, String value);

  /**
   * Puts a parser attribute value in the metadata.
   *
   * Prefer to use {@link #putParserAttribute(AstParserAttribute, Object)} for type safety.
   *
   * In case you need ot use this one, make sure {@code value} is a basic Java type like Integer, Double, BigDecimal, String,
   * Boolean, etc. to avoid class loading issues when serializing/deserializing these.
   *
   * @param key   The key to later reference the given parser attribute.
   * @param value The actual value of the attribute.
   * @return This instance for chaining purposes.
   */
  ComponentMetadataAstBuilder putParserAttribute(String key, Object value);

  /**
   * Puts a parser attribute value in the metadata.
   *
   * @param attribute The key to later reference the given parser attribute.
   * @param value     The actual value of the attribute.
   * @param <T>       The type of the attribute value.
   * @return This instance for chaining purposes.
   */
  <T> ComponentMetadataAstBuilder putParserAttribute(AstParserAttribute<T> attribute, T value);

  /**
   * Builds the target component metadata.
   *
   * @return the target component metadata.
   */
  ComponentMetadataAst build();
}
