/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.api.validation;

import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;

import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;

import java.util.List;
import java.util.Optional;
import java.util.function.Predicate;

/**
 * Defines a validation for an {@link ArtifactAst}. This is NOT intended to do validations of the components within the artifact.
 * You must implement {@link Validation} for that.
 * <p>
 * Runtime and modules may implement their own validations by implementing this interface.
 * <p>
 * Implementations must be stateless.
 *
 * @since 1.1
 */
public interface ArtifactValidation extends Validation {

  /**
   * @return an internal name for this validation.
   */
  @Override
  String getName();

  /**
   * @return a user-friendly description of the validation.
   */
  @Override
  String getDescription();

  /**
   * @return the severity level of a failure of this validation.
   */
  @Override
  Level getLevel();

  /**
   * Runs this validation in the provided artifact.
   * <p>
   * This method will only be called if {@link #validateMany(ArtifactAst)} is not implemented.
   * <p>
   * Refer to the factory methods in {@link ValidationResultItem} for creating the object to be returned.
   *
   * @param artifact the artifact under validation.
   * @return {@link Optional#empty() empty} if the validation passed, or a result object describing the failure otherwise.
   */
  default Optional<ValidationResultItem> validate(ArtifactAst artifact) {
    return validateMany(artifact).stream().findFirst();
  }

  /**
   * Runs this validation in the provided artifact.
   * <p>
   * Refer to the factory methods in {@link ValidationResultItem} for creating the object to be returned.
   *
   * @param artifact the artifact under validation.
   * @return {@link Optional#empty() empty} if the validation passed, or a result object describing the failure otherwise.
   */
  default List<ValidationResultItem> validateMany(ArtifactAst artifact) {
    return validate(artifact)
        .map(r -> singletonList(r))
        .orElse(emptyList());
  }

  @Override
  default Predicate<List<ComponentAst>> applicable() {
    return h -> true;
  }

  @Override
  default Optional<ValidationResultItem> validate(ComponentAst component, ArtifactAst artifact) {
    return this.validate(artifact);
  }
}
