/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal;

import static org.mule.runtime.api.functional.Either.left;
import static org.mule.runtime.api.functional.Either.right;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.meta.ExpressionSupport.NOT_SUPPORTED;

import static java.lang.String.format;
import static java.util.Objects.isNull;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toList;

import static org.apache.commons.lang3.StringUtils.isEmpty;
import static org.apache.commons.lang3.StringUtils.isNotEmpty;

import org.mule.metadata.api.annotation.EnumAnnotation;
import org.mule.metadata.api.model.BooleanType;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.NumberType;
import org.mule.metadata.api.model.ObjectFieldType;
import org.mule.metadata.api.model.StringType;
import org.mule.metadata.api.visitor.MetadataTypeVisitor;
import org.mule.metadata.java.api.annotation.ClassInformationAnnotation;
import org.mule.runtime.api.functional.Either;
import org.mule.runtime.api.meta.model.ModelProperty;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.ast.api.ComponentGenerationInformation;
import org.mule.runtime.ast.api.ComponentMetadataAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.ast.api.ParameterResolutionException;
import org.mule.runtime.ast.api.builder.ComponentAstBuilder;
import org.mule.runtime.ast.internal.builder.PropertiesResolver;
import org.mule.runtime.extension.api.declaration.type.annotation.LiteralTypeAnnotation;
import org.mule.runtime.extension.api.error.ErrorMapping;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.function.Supplier;


public class DefaultComponentParameterAst implements ComponentParameterAst {

  private static final String DEFAULT_EXPRESSION_PREFIX = "#[";
  private static final String DEFAULT_EXPRESSION_SUFFIX = "]";

  private static final Class<? extends ModelProperty> allowsExpressionWithoutMarkersModelPropertyClass;
  private static final Map<String, Function<String, Number>> fixedNumberMappings = new HashMap<>();

  static {
    Class<? extends ModelProperty> foundClass = null;
    try {
      foundClass = (Class<? extends ModelProperty>) Class
          .forName("org.mule.runtime.module.extension.api.loader.java.property.AllowsExpressionWithoutMarkersModelProperty");
    } catch (ClassNotFoundException | SecurityException e) {
      // No custom location processing
    }
    allowsExpressionWithoutMarkersModelPropertyClass = foundClass;

    fixedNumberMappings.put(Integer.class.getName(), Integer::valueOf);
    fixedNumberMappings.put(int.class.getName(), Integer::valueOf);

    fixedNumberMappings.put(Float.class.getName(), Float::valueOf);
    fixedNumberMappings.put(float.class.getName(), Float::valueOf);

    fixedNumberMappings.put(Long.class.getName(), Long::valueOf);
    fixedNumberMappings.put(long.class.getName(), Long::valueOf);

    fixedNumberMappings.put(Byte.class.getName(), Byte::valueOf);
    fixedNumberMappings.put(byte.class.getName(), Byte::valueOf);

    fixedNumberMappings.put(Short.class.getName(), Short::valueOf);
    fixedNumberMappings.put(short.class.getName(), Short::valueOf);

    fixedNumberMappings.put(Double.class.getName(), Double::valueOf);
    fixedNumberMappings.put(double.class.getName(), Double::valueOf);
  }

  private final String rawValue;
  private final ParameterModel model;
  private final ParameterGroupModel groupModel;
  private final ComponentMetadataAst metadata;
  private final ComponentGenerationInformation generationInformation;

  private volatile LazyValue<String> resolved;
  private volatile LazyValue<Either<String, Object>> value;

  public DefaultComponentParameterAst(String rawValue, ParameterModel model, ParameterGroupModel parameterGroupModel,
                                      ComponentGenerationInformation generationInformation,
                                      PropertiesResolver propertiesResolver) {
    this(rawValue, model, parameterGroupModel, null, generationInformation, propertiesResolver);
  }

  public DefaultComponentParameterAst(String rawValue, ParameterModel model, ParameterGroupModel parameterGroupModel,
                                      ComponentMetadataAst metadata,
                                      ComponentGenerationInformation generationInformation,
                                      PropertiesResolver propertiesResolver) {
    this(rawValue, rawValue, null, model, parameterGroupModel, metadata, generationInformation, propertiesResolver);
  }

  public DefaultComponentParameterAst(ComponentAstBuilder complexValue, ParameterModel model,
                                      ParameterGroupModel parameterGroupModel,
                                      ComponentMetadataAst metadata, ComponentGenerationInformation generationInformation,
                                      PropertiesResolver propertiesResolver) {
    this(null, null, complexValue::build, model, parameterGroupModel, metadata, generationInformation, propertiesResolver);
  }

  public DefaultComponentParameterAst(List<ComponentAstBuilder> complexValue, ParameterModel model,
                                      ParameterGroupModel parameterGroupModel,
                                      ComponentMetadataAst metadata, ComponentGenerationInformation generationInformation,
                                      PropertiesResolver propertiesResolver) {
    this(null, null, () -> complexValue.stream().map(ComponentAstBuilder::build).collect(toList()), model,
         parameterGroupModel, metadata, generationInformation, propertiesResolver);
  }

  public DefaultComponentParameterAst(List<ErrorMapping> complexValue, ParameterModel model,
                                      ParameterGroupModel parameterGroupModel,
                                      ComponentGenerationInformation generationInformation,
                                      PropertiesResolver propertiesResolver) {
    this(null, null, () -> complexValue, model, parameterGroupModel, null, generationInformation, propertiesResolver);
  }

  private DefaultComponentParameterAst(String rawValue, String mappedRawValue, Supplier<Object> complexValue,
                                       ParameterModel model, ParameterGroupModel parameterGroupModel,
                                       ComponentMetadataAst metadata, ComponentGenerationInformation generationInformation,
                                       PropertiesResolver propertiesResolver) {
    this.rawValue = rawValue;
    this.model = model;
    this.groupModel = parameterGroupModel;
    this.metadata = metadata;
    this.generationInformation = generationInformation;

    resetResolvedParam(rawValue, complexValue, propertiesResolver);
    if (complexValue == null && rawValue != null && rawValue.contains("${")) {
      propertiesResolver.onMappingFunctionChanged(() -> resetResolvedParam(rawValue, complexValue, propertiesResolver));
    }
  }

  private void resetResolvedParam(String rawValue, Supplier<Object> complexValue, PropertiesResolver propertiesResolver) {
    if (complexValue != null) {
      this.resolved = new LazyValue<>();
      this.value = new LazyValue<>(() -> right(complexValue.get()));
    } else {
      if (rawValue != null && rawValue.contains("${")) {
        this.resolved = new LazyValue<>(() -> propertiesResolver.apply(rawValue));
      } else {
        this.resolved = new LazyValue<>(rawValue);
      }

      this.value = new LazyValue<>(() -> {
        String resolvedRawValue = getResolvedRawValue();

        if (isNull(resolvedRawValue)) {
          final Object defaultValue = getModel().getDefaultValue();

          if (defaultValue != null) {
            if (defaultValue instanceof String) {
              resolvedRawValue = (String) defaultValue;
            } else if (getModel().getType().getAnnotation(EnumAnnotation.class).isPresent()) {
              resolvedRawValue = ((Enum<?>) defaultValue).name();
            } else {
              return right(defaultValue);
            }
          }
        }

        return resolveParamValue(resolvedRawValue);
      });
    }
  }

  private Either<String, Object> resolveParamValue(String resolvedRawValue) {
    AtomicReference<String> expression = new AtomicReference<>();
    AtomicReference<Object> fixedValue = new AtomicReference<>();

    final MetadataTypeVisitor visitor = new MetadataTypeVisitor() {

      @Override
      public void visitObjectField(ObjectFieldType objectFieldType) {
        objectFieldType.getValue().accept(this);
      }

      @Override
      public void visitBoolean(BooleanType booleanType) {
        if (isExpression(resolvedRawValue)) {
          defaultVisit(booleanType);
        } else if (!isEmpty(resolvedRawValue)) {
          fixedValue.set(Boolean.valueOf(resolvedRawValue));
        }
      }

      @Override
      public void visitNumber(NumberType numberType) {
        if (isExpression(resolvedRawValue)) {
          defaultVisit(numberType);
        } else if (!isEmpty(resolvedRawValue)) {
          visitFixedNumber(resolvedRawValue, fixedValue, numberType);
        }
      }

      @Override
      public void visitString(StringType stringType) {
        if (isExpression(resolvedRawValue)) {
          defaultVisit(stringType);
        } else {
          // Empty string is valid, do not return either.empty for this!
          fixedValue.set(resolvedRawValue);
        }
      }

      @Override
      protected void defaultVisit(MetadataType metadataType) {
        if (!getModel().getAllowedStereotypes().isEmpty() && resolvedRawValue != null) {
          // For references, just return the name of the referenced object if it is a fixed value, but the param may be an
          // expression that builds an object of the expected stereotype
          defaultVisitFixedValue(resolvedRawValue, expression, fixedValue);
        } else if (NOT_SUPPORTED.equals(getModel().getExpressionSupport())
            || getModel().getType().getAnnotation(LiteralTypeAnnotation.class).isPresent()) {
          fixedValue.set(resolvedRawValue);
        } else if (!NOT_SUPPORTED.equals(getModel().getExpressionSupport())) {
          defaultVisitFixedValue(resolvedRawValue, expression, fixedValue);
        } else {
          final Optional<String> extractExpression = extractExpression(resolvedRawValue);
          if (extractExpression.isPresent()) {
            expression.set(extractExpression.get());
          } else {
            fixedValue.set(resolvedRawValue);
          }
        }
      }
    };

    try {
      getModel().getType().accept(visitor);
    } catch (Exception e) {
      throw new ParameterResolutionException(createStaticMessage(format("Exception resolving param '%s' with value '%s' at '%s:%d:%d' (%s)",
                                                                        getModel().getName(), resolvedRawValue,
                                                                        metadata != null ? metadata.getFileName().orElse("") : "",
                                                                        metadata != null ? metadata.getStartLine().orElse(-1)
                                                                            : -1,
                                                                        metadata != null ? metadata.getStartColumn().orElse(-1)
                                                                            : -1,
                                                                        e.toString())),
                                             e);
    }

    if (expression.get() != null) {
      return left(expression.get());
    } else if (fixedValue.get() != null) {
      return right(fixedValue.get());
    } else {
      return Either.empty();
    }
  }

  private void visitFixedNumber(String rawValue, AtomicReference<Object> value, NumberType numberType) {
    value.set(numberType.getAnnotation(ClassInformationAnnotation.class)
        .map(classInfo -> fixedNumberMappings.getOrDefault(classInfo.getClassname(), s -> null).apply(rawValue))
        .orElseGet(() -> {
          try {
            Long longValue = Long.valueOf(rawValue);
            if (longValue <= Integer.MAX_VALUE && longValue >= Integer.MIN_VALUE) {
              return longValue.intValue();
            }
            return longValue;
          } catch (NumberFormatException e) {
            return Double.valueOf(rawValue);
          }
        }));
  }

  private void defaultVisitFixedValue(String rawValue, AtomicReference<String> expression, AtomicReference<Object> fixedValue) {
    Optional<String> expressionOpt = extractExpression(rawValue);
    if (expressionOpt.isPresent()) {
      // For complex types that may be the result of an expression, just return the expression
      expression.set(expressionOpt.get());
    } else {
      fixedValue.set(rawValue);
    }
  }

  private boolean isExpression(Object value) {
    if (value instanceof String) {
      String trim = ((String) value).trim();

      if (trim.startsWith(DEFAULT_EXPRESSION_PREFIX) && trim.endsWith(DEFAULT_EXPRESSION_SUFFIX)) {
        return true;
      }

      return allowsExpressionWithoutMarkersModelPropertyClass != null
          && getModel().getModelProperty(allowsExpressionWithoutMarkersModelPropertyClass).isPresent();
    } else {
      return false;
    }
  }

  /**
   * Parse the given value and remove expression markers if it is considered as an expression.
   *
   * @param value Value to parse
   * @return a String containing the expression without markers or null if the value is not an expression.
   */
  public Optional<String> extractExpression(Object value) {
    Optional<String> result = empty();
    if (isExpression(value)) {
      String expression = (String) value;
      if (isNotEmpty(expression)) {
        String trimmedText = expression.trim();

        if (trimmedText.startsWith(DEFAULT_EXPRESSION_PREFIX) && trimmedText.endsWith(DEFAULT_EXPRESSION_SUFFIX)) {
          result =
              of(trimmedText.substring(DEFAULT_EXPRESSION_PREFIX.length(),
                                       trimmedText.length() - DEFAULT_EXPRESSION_SUFFIX.length()));
        } else {
          result = of(trimmedText);
        }
      }
    }

    return result;
  }

  @Override
  public <T> Either<String, T> getValue() {
    return (Either<String, T>) value.get();
  }

  @Override
  public <T> Either<String, Either<ParameterResolutionException, T>> getValueOrResolutionError() {
    try {
      return getValue().mapRight(fixedValue -> right((T) fixedValue));
    } catch (ParameterResolutionException e) {
      return right(left(e));
    }
  }

  @Override
  public String getRawValue() {
    return rawValue;
  }

  @Override
  public String getResolvedRawValue() {
    return resolved.get();
  }

  @Override
  public ParameterModel getModel() {
    return model;
  }

  @Override
  public ParameterGroupModel getGroupModel() {
    return groupModel;
  }

  @Override
  public Optional<ComponentMetadataAst> getMetadata() {
    return ofNullable(metadata);
  }

  @Override
  public ComponentGenerationInformation getGenerationInformation() {
    return generationInformation;
  }

  @Override
  public boolean isDefaultValue() {
    return ofNullable(getResolvedRawValue())
        .map(v -> {
          if (getModel() != null && getModel().getDefaultValue() != null) {
            return v.equals(getModel().getDefaultValue());
          } else {
            return false;
          }
        }).orElse(true);
  }

  @Override
  public String toString() {
    return "DefaultComponentParameterAst{" + model.getName() + ": " + rawValue + "}";
  }
}
