/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.builder;

import static org.mule.runtime.ast.internal.builder.ApplicationModelTypeUtils.resolveTypedComponentIdentifier;
import static org.mule.runtime.ast.internal.builder.ComponentLocationVisitor.resolveLocation;

import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.toList;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.meta.model.ComponentModel;
import org.mule.runtime.api.meta.model.config.ConfigurationModel;
import org.mule.runtime.api.meta.model.connection.ConnectionProviderModel;
import org.mule.runtime.api.meta.model.nested.NestableElementModel;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.api.util.Pair;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.ast.api.builder.ComponentAstBuilder;
import org.mule.runtime.ast.internal.DefaultComponentAst;
import org.mule.runtime.ast.internal.model.ExtensionModelHelper;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Supplier;

public class DefaultComponentAstBuilder extends BaseComponentAstBuilder<DefaultComponentAstBuilder> {

  private final ExtensionModelHelper extModelHelper;
  private final List<DefaultComponentAstBuilder> hierarchy;

  private final int indexInParent;

  public DefaultComponentAstBuilder(PropertiesResolver propertiesResolver, ExtensionModelHelper extModelHelper,
                                    List<DefaultComponentAstBuilder> hierarchy, int indexInParent) {
    super(propertiesResolver);
    this.extModelHelper = extModelHelper;
    this.hierarchy = hierarchy;
    this.indexInParent = indexInParent;
  }

  @Override
  public ComponentAstBuilder addChildComponent() {
    final List<DefaultComponentAstBuilder> childHierarchy = new ArrayList<>(hierarchy);
    childHierarchy.add(this);

    final DefaultComponentAstBuilder componentAstBuilder =
        new DefaultComponentAstBuilder(getPropertiesResolver(), extModelHelper, childHierarchy,
                                       (int) childComponentsStream().count());
    addChildComponent(componentAstBuilder);
    return componentAstBuilder;
  }

  @Override
  public ComponentAstBuilder withIdentifier(ComponentIdentifier identifier) {
    extModelHelper.lookupExtensionModelFor(identifier).ifPresent(this::withExtensionModel);
    return super.withIdentifier(identifier);
  }

  @Override
  public ComponentAst get() {
    return build();
  }

  @Override
  public ComponentAst build() {
    prepareForBuild();
    return doBuild();
  }

  @Override
  protected void prepareForBuild() {
    requireNonNull(getIdentifier(), "identifier");
    requireNonNull(getMetadata(), "metadata");

    if (!getModel(Object.class).isPresent()) {
      resolveTypedComponentIdentifier(this, hierarchy.isEmpty(), extModelHelper);
    }

    super.prepareForBuild();

    childComponentsStream().forEach(DefaultComponentAstBuilder::prepareForBuild);
  }

  private ComponentAst doBuild() {
    final Map<Pair<ParameterModel, ParameterGroupModel>, ComponentParameterAst> parameterAsts = createParameterAsts();
    Supplier<Optional<String>> componentId = obtainLocation(parameterAsts);

    resolveParamsLocations();

    final List<ComponentAst> builtChildren = childComponentsStream().map(DefaultComponentAstBuilder::doBuild).collect(toList());

    Map<String, String> extraParams = new HashMap<>(getRawParameters());
    parameterAsts.keySet().forEach(k -> extraParams.remove(k.getFirst().getName()));

    return new DefaultComponentAst(builtChildren, parameterAsts, extraParams, getPropertiesResolver(),
                                   getAnnotations(), componentId, getExtensionModel(),
                                   getModel(ComponentModel.class).orElse(null),
                                   getModel(NestableElementModel.class).orElse(null),
                                   getModel(ConfigurationModel.class).orElse(null),
                                   getModel(ConnectionProviderModel.class).orElse(null),
                                   getModel(ParameterizedModel.class).orElse(null),
                                   getType(),
                                   getGenerationInformation().build(),
                                   getMetadata(), getLocation(), getIdentifier(), getComponentType());
  }

  protected void resolveParamsLocations() {
    getParamsChildren()
        .stream()
        .filter(pc -> pc.componentId == null)
        .forEach(pc -> {
          pc.obtainLocation(pc.createParameterAsts());
          pc.resolveParamsLocations();
        });

    // This is required so that the location for complex params inside dsl groups is properly set
    childComponentsStream()
        .filter(pc -> pc.componentId == null)
        .forEach(pc -> {
          pc.obtainLocation(pc.createParameterAsts());
          pc.resolveParamsLocations();
        });
  }

  protected Supplier<Optional<String>> obtainLocation(final Map<Pair<ParameterModel, ParameterGroupModel>, ComponentParameterAst> parameterAsts) {
    Supplier<Optional<String>> componentId = new LazyValue<>(() -> resolveComponentId(parameterAsts));
    setComponentId(componentId);

    resolveLocation(this, hierarchy);
    return componentId;
  }

  public int getIndexInParent() {
    return indexInParent;
  }

}
