/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal;

import static java.util.Optional.ofNullable;

import org.mule.runtime.ast.api.ComponentMetadataAst;
import org.mule.runtime.ast.api.ImportedResource;
import org.mule.runtime.ast.internal.builder.PropertiesResolver;

import java.util.Optional;
import java.util.function.UnaryOperator;

/**
 * Default implementation for {@link ImportedResource}.
 */
public final class DefaultImportedResource implements ImportedResource {

  private final String resourceLocation;
  private final ComponentMetadataAst metadata;
  private final String resolutionFailure;
  private final PropertiesResolver propertiesResolver = new PropertiesResolver();

  public DefaultImportedResource(String resourceLocation, ComponentMetadataAst metadata) {
    this.resourceLocation = resourceLocation;
    this.metadata = metadata;
    this.resolutionFailure = null;
  }

  public DefaultImportedResource(String resourceLocation, ComponentMetadataAst metadata, String resolutionFailure) {
    this.resourceLocation = resourceLocation;
    this.metadata = metadata;
    this.resolutionFailure = resolutionFailure;
  }

  @Override
  public String getRawResourceLocation() {
    return resourceLocation;
  }

  @Override
  public String getResourceLocation() {
    return propertiesResolver.apply(resourceLocation);
  }

  @Override
  public ComponentMetadataAst getMetadata() {
    return metadata;
  }

  @Override
  public Optional<String> getResolutionFailure() {
    return ofNullable(resolutionFailure);
  }

  @Override
  public void updatePropertiesResolver(UnaryOperator<String> propertiesResolver) {
    this.propertiesResolver.setMappingFunction(propertiesResolver);
  }
}
