/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.api.util;

import static org.mule.runtime.api.functional.Either.left;
import static org.mule.runtime.api.functional.Either.right;

import org.mule.runtime.api.functional.Either;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.ast.api.ComponentGenerationInformation;
import org.mule.runtime.ast.api.ComponentMetadataAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.ast.api.ParameterResolutionException;

import java.util.Optional;

/**
 * This class provides a skeletal implementation of the {@link ComponentParameterAst} interface, to minimize the effort required
 * to implement this interface when decorating instances of {@link ComponentParameterAst}.
 *
 * @since 1.0
 */
public class BaseComponentParameterAstDecorator implements ComponentParameterAst {

  private final ComponentParameterAst decorated;

  /**
   * Creates a new decorator for the provided instance.
   *
   * @param decorated the {@link ComponentParameterAst} instance to decorate.
   */
  public BaseComponentParameterAstDecorator(ComponentParameterAst decorated) {
    this.decorated = decorated;
  }

  /**
   * @return the decorated {@link ComponentParameterAst} instance.
   */
  protected ComponentParameterAst getDecorated() {
    return decorated;
  }

  @Override
  public ParameterModel getModel() {
    return decorated.getModel();
  }

  @Override
  public ParameterGroupModel getGroupModel() {
    return decorated.getGroupModel();
  }

  @Override
  public <T> Either<String, T> getValue() {
    return decorated.getValue();
  }

  @Override
  public <T> Either<String, Either<ParameterResolutionException, T>> getValueOrResolutionError() {
    try {
      return getValue().mapRight(fixedValue -> right((T) fixedValue));
    } catch (ParameterResolutionException e) {
      return right(left(e));
    }
  }

  @Override
  public String getRawValue() {
    return decorated.getRawValue();
  }

  @Override
  public String getResolvedRawValue() {
    return decorated.getResolvedRawValue();
  }

  @Override
  public Optional<ComponentMetadataAst> getMetadata() {
    return decorated.getMetadata();
  }

  @Override
  public ComponentGenerationInformation getGenerationInformation() {
    return decorated.getGenerationInformation();
  }

  @Override
  public boolean isDefaultValue() {
    return decorated.isDefaultValue();
  }

}
